﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.DataModelMaker.Core.Template;
using ObsoleteUserDataConverter.Properties;

namespace EffectMaker.ObsoleteUserDataConverter.ObsoleteUserData
{
    /// <summary>
    /// Writes user definition to UI Xaml definition.
    /// </summary>
    public class UserDefinitionXamlWriter
    {
        /// <summary>Constant template tag name "Resources"</summary>
        private const string TemplateTagNameResources = "Resources";

        /// <summary>Constant template tag name "ResourceKey"</summary>
        private const string TemplateTagNameResourceKey = "ResourceKey";

        /// <summary>Constant template tag name "ResourceKey"</summary>
        private const string TemplateTagNameItems = "Items";

        /// <summary>Constant template tag name "Label"</summary>
        private const string TemplateTagNameLabel = "Label";

        /// <summary>Constant template tag name "Value"</summary>
        private const string TemplateTagNameValue = "Value";

        /// <summary>Constant template tag name "BindSource"</summary>
        private const string TemplateTagNameBindSource = "BindSource";

        /// <summary>Constant template tag name "ControlSets"</summary>
        private const string TemplateTagNameControlSets = "ControlSets";

        /// <summary>Constant template tag name "MinValue"</summary>
        private const string TemplateTagNameMinValue = "MinValue";

        /// <summary>Constant template tag name "MaxValue"</summary>
        private const string TemplateTagNameMaxValue = "MaxValue";

        /// <summary>Constant template tag name "Groups"</summary>
        private const string TemplateTagNameGroups = "Groups";

        /// <summary>Constant template tag name "CheckBoxMask"</summary>
        private const string TemplateTagNameCheckBoxMask = "CheckBoxMask";

        /// <summary>The template helpers that parses and stores the Xaml contents.</summary>
        private static readonly TemplateHelper[] TemplateHelpers =
            new TemplateHelper[(int)Templates.TemplateCount];

        /// <summary>The UI language to use.</summary>
        private string language = string.Empty;

        /// <summary>The static resources for the Xaml file.</summary>
        private string xamlResources = string.Empty;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static UserDefinitionXamlWriter()
        {
            for (int i = 0; i < (int)Templates.TemplateCount; ++i)
            {
                TemplateHelpers[i] = new TemplateHelper();
            }

            TemplateHelpers[(int)Templates.Main].ParseSimpleTemplate(
                Resources.UddUITemplate_Main);

            TemplateHelpers[(int)Templates.Group].ParseSimpleTemplate(
                Resources.UddUITemplate_Group);

            TemplateHelpers[(int)Templates.Slider].ParseSimpleTemplate(
                Resources.UddUITemplate_Slider);

            TemplateHelpers[(int)Templates.CheckBox].ParseSimpleTemplate(
                Resources.UddUITemplate_CheckBox);

            TemplateHelpers[(int)Templates.RadioButtonGroup].ParseSimpleTemplate(
                Resources.UddUITemplate_RadioButton);

            TemplateHelpers[(int)Templates.RadioButtonGroupAvailableItems].ParseSimpleTemplate(
                Resources.UddUITemplate_RBGroupAvailableItems);

            TemplateHelpers[(int)Templates.RadioButtonGroupKeyValuePairItem].ParseSimpleTemplate(
                Resources.UddUITemplate_RBGroupKeyValuePairItem);

            TemplateHelpers[(int)Templates.StackPanel].ParseSimpleTemplate(
                Resources.UddUITemplate_StackPanel);
        }

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserDefinitionXamlWriter()
        {
        }

        /// <summary>
        /// Enum for the templates.
        /// </summary>
        private enum Templates : int
        {
            /// <summary>The body of the Xaml.</summary>
            Main = 0,

            /// <summary>The template for groups.</summary>
            Group,

            /// <summary>The template for dial sliders.</summary>
            Slider,

            /// <summary>The template for check boxes.</summary>
            CheckBox,

            /// <summary>The template for radio button groups.</summary>
            RadioButtonGroup,

            /// <summary>The template for available items resource of the radio button groups.</summary>
            RadioButtonGroupAvailableItems,

            /// <summary>The template for a single available item of the radio button groups' resource.</summary>
            RadioButtonGroupKeyValuePairItem,

            /// <summary>The template for a stack panel.</summary>
            StackPanel,

            /// <summary>The template count.</summary>
            TemplateCount,
        }

        /// <summary>
        /// Get or set the data model property name for the first check box group.
        /// If the name is not set, the generator will use the default property
        /// name, like "Flag0", "Flag1", ...etc.
        /// </summary>
        public string FirstCheckBoxGroupPropertyName { get; set; }

        /// <summary>
        /// Get or set the data model property name for the first radio button group.
        /// If the name is not set, the generator will use the default property
        /// name, like "Switch0", "Switch1", ...etc.
        /// </summary>
        public string FirstRadioButtonGroupPropertyName { get; set; }

        /// <summary>
        /// Write UI XAML file that generated from the given user definition.
        /// </summary>
        /// <param name="def">The group definition to generate from.</param>
        /// <param name="filePath">The XAML file path to write to.</param>
        /// <param name="language">The UI language.</param>
        /// <returns>True on success.</returns>
        public bool Write(GroupDefinition def, string filePath, string language = null)
        {
            this.language = language;
            this.xamlResources = string.Empty;

            TemplateHelper template = this.InitAndGetTemplate(Templates.Main);

            // We don't want the root group, only generate the child groups.
            string xaml = string.Empty;
            foreach (UserDefinitionBase child in def.Controls)
            {
                if (child is GroupDefinition)
                {
                    string snippet = this.GenerateFromGroupDefinition((GroupDefinition)child, 0);
                    if (string.IsNullOrEmpty(snippet) == false)
                    {
                        xaml += snippet;
                    }
                }
            }

            template.SetTemplateTagValue(TemplateTagNameLabel, def.GetLabelForLanguage(this.language));
            template.SetTemplateTagValue(TemplateTagNameResources, this.xamlResources);
            template.SetTemplateTagValue(TemplateTagNameGroups, xaml);

            xaml = template.Compose();

            File.WriteAllText(filePath, xaml);

            return true;
        }

        /// <summary>
        /// Generate data model definitions from the group definition and its children.
        /// </summary>
        /// <param name="def">The group definition.</param>
        /// <param name="level">The level in the hierarchy.</param>
        /// <returns>The generated XAML from the given group definition.</returns>
        private string GenerateFromGroupDefinition(GroupDefinition def, int level)
        {
            if (def.Visible == false)
            {
                return null;
            }

            if (def.GroupType == "CheckBox")
            {
                return this.GenerateCheckBoxGroupProperty(def, level);
            }
            else if (def.GroupType == "RadioButton")
            {
                return this.GenerateRadioButtonGroupProperty(def, level + 2);
            }
            else
            {
                var controlSetXamlList = new List<string>();

                string childXaml = string.Empty;
                foreach (UserDefinitionBase child in def.Controls)
                {
                    string snippet = null;
                    if (child is GroupDefinition)
                    {
                        snippet = this.GenerateFromGroupDefinition((GroupDefinition)child, level + 2);
                    }
                    else if (child is IntSliderDefinition)
                    {
                        snippet = this.GenerateIntSliderProperty(child, level);
                    }
                    else if (child is FloatSliderDefinition)
                    {
                        snippet = this.GenerateFloatSliderProperty(child, level);
                    }

                    if (string.IsNullOrEmpty(snippet) == false)
                    {
                        controlSetXamlList.Add(snippet);
                    }
                }

                // Do not generate empty groups.
                if (controlSetXamlList.Count <= 0)
                {
                    return string.Empty;
                }

                TemplateHelper templateStackPanel =
                    this.InitAndGetTemplate(Templates.StackPanel);

                // If there are odd number of controls, the left side takes the extra one.
                int controlCount = controlSetXamlList.Count / 2;
                controlCount += controlSetXamlList.Count % 2 == 1 ? 1 : 0;

                // Combine the generated XAML strings of the check box control sets on the left side.
                string leftPanelXaml = string.Empty;
                for (int i = 0; i < controlCount; ++i)
                {
                    leftPanelXaml += controlSetXamlList[i];
                }

                templateStackPanel.SetTemplateTagValue(TemplateTagNameControlSets, leftPanelXaml);
                childXaml += templateStackPanel.Compose(16);

                // Combine the generated XAML strings of the check box control sets on the right side.
                string rightPanelXaml = string.Empty;
                for (int i = controlCount; i < controlSetXamlList.Count; ++i)
                {
                    rightPanelXaml += controlSetXamlList[i];
                }

                templateStackPanel.SetTemplateTagValue(TemplateTagNameControlSets, rightPanelXaml);
                childXaml += templateStackPanel.Compose(16);

                TemplateHelper template = this.InitAndGetTemplate(Templates.Group);

                template.SetTemplateTagValue(TemplateTagNameLabel, def.GetLabelForLanguage(this.language));
                template.SetTemplateTagValue(TemplateTagNameControlSets, childXaml);

                return template.Compose(level * 4);
            }
        }

        /// <summary>
        /// Generate data model property with the given check box group definition.
        /// </summary>
        /// <param name="def">The check box group definition.</param>
        /// <param name="level">The level in the hierarchy.</param>
        /// <returns>The generated XAML from the given group definition.</returns>
        private string GenerateCheckBoxGroupProperty(UserDefinitionBase def, int level)
        {
            if (def.Visible == false)
            {
                return null;
            }

            var castedUserDef = def as GroupDefinition;
            if (castedUserDef == null || castedUserDef.GroupType != "CheckBox")
            {
                return null;
            }

            TemplateHelper template =
                this.InitAndGetTemplate(Templates.CheckBox);

            TemplateHelper templateStackPanel =
                this.InitAndGetTemplate(Templates.StackPanel);

            string childXaml = string.Empty;

            string dataSrcName = "Flag" + def.Index;
            if (string.IsNullOrEmpty(this.FirstCheckBoxGroupPropertyName) == false && def.Index == 0)
            {
                dataSrcName = this.FirstCheckBoxGroupPropertyName;
            }

            var controlSetXamlList = new List<string>();

            // Determine the default value.
            uint duplicationTester = 0x00000000;
            foreach (UserDefinitionBase childDef in castedUserDef.Controls)
            {
                var checkBoxDef = childDef as CheckBoxDefinition;
                if (checkBoxDef == null || checkBoxDef.Visible == false)
                {
                    continue;
                }

                // A check box group binds to a uint (32-bit), so the index has
                // to be between 0 to 31, each of them represents a bit in the uint.
                if (checkBoxDef.Index < 0 || checkBoxDef.Index >= 32)
                {
                    continue;
                }

                uint value = (uint)(0x00000001 << checkBoxDef.Index);
                if ((duplicationTester & value) != 0)
                {
                    // A check box with the same index has already been defined.
                    // Skip this one.
                    continue;
                }

                // Register this bit so we know a check box with this index is already
                // defined.
                duplicationTester |= value;

                template.SetTemplateTagValue(TemplateTagNameLabel, checkBoxDef.GetLabelForLanguage(this.language));
                template.SetTemplateTagValue(TemplateTagNameBindSource, dataSrcName);
                template.SetTemplateTagValue(TemplateTagNameCheckBoxMask, value.ToString());

                controlSetXamlList.Add(template.Compose(4));
            }

            // Do not generate the group if it's empty.
            if (controlSetXamlList.Count <= 0)
            {
                return null;
            }

            // If there are odd number of controls, the left side takes the extra one.
            int controlCount = controlSetXamlList.Count / 2;
            controlCount += controlSetXamlList.Count % 2 == 1 ? 1 : 0;

            // Combine the generated XAML strings of the check box control sets on the left side.
            string leftPanelXaml = string.Empty;
            for (int i = 0; i < controlCount; ++i)
            {
                leftPanelXaml += controlSetXamlList[i];
            }

            templateStackPanel.SetTemplateTagValue(TemplateTagNameControlSets, leftPanelXaml);
            childXaml += templateStackPanel.Compose(16);

            // Combine the generated XAML strings of the check box control sets on the right side.
            string rightPanelXaml = string.Empty;
            for (int i = controlCount; i < controlSetXamlList.Count; ++i)
            {
                rightPanelXaml += controlSetXamlList[i];
            }

            templateStackPanel.SetTemplateTagValue(TemplateTagNameControlSets, rightPanelXaml);
            childXaml += templateStackPanel.Compose(16);

            // Compose the group for the check boxes.
            template =
                this.InitAndGetTemplate(Templates.Group);

            template.SetTemplateTagValue(TemplateTagNameLabel, def.GetLabelForLanguage(this.language));
            template.SetTemplateTagValue(TemplateTagNameControlSets, childXaml);

            return template.Compose(level * 4);
        }

        /// <summary>
        /// Generate data model property with the given radio button group definition.
        /// </summary>
        /// <param name="def">The radio button group definition.</param>
        /// <param name="level">The level in the hierarchy.</param>
        /// <returns>The generated XAML from the given group definition.</returns>
        private string GenerateRadioButtonGroupProperty(UserDefinitionBase def, int level)
        {
            var castedUserDef = def as GroupDefinition;
            if (castedUserDef == null || castedUserDef.GroupType != "RadioButton")
            {
                return null;
            }

            if (def.Visible == false)
            {
                return null;
            }

            TemplateHelper template =
                this.InitAndGetTemplate(Templates.RadioButtonGroup);

            TemplateHelper templateAvailableItems =
                this.InitAndGetTemplate(Templates.RadioButtonGroupAvailableItems);

            TemplateHelper templateItem =
                this.InitAndGetTemplate(Templates.RadioButtonGroupKeyValuePairItem);

            string availableItemsText = string.Empty;

            string dataSrcName = "Switch" + def.Index;
            if (string.IsNullOrEmpty(this.FirstRadioButtonGroupPropertyName) == false && def.Index == 0)
            {
                dataSrcName = this.FirstRadioButtonGroupPropertyName;
            }

            // Determine the default value.
            uint duplicationTester = 0x00000000;
            foreach (UserDefinitionBase childDef in castedUserDef.Controls)
            {
                var radioButtonDef = childDef as RadioButtonDefinition;
                if (radioButtonDef == null || radioButtonDef.Visible == false)
                {
                    continue;
                }

                // A radio button group binds to a uint (32-bit), so the index has
                // to be between 0 to 31, each of them represents a bit in the uint.
                if (radioButtonDef.Index < 0 || radioButtonDef.Index >= 32)
                {
                    continue;
                }

                uint value = (uint)(0x00000001 << radioButtonDef.Index);
                if ((duplicationTester & value) != 0)
                {
                    // A check box with the same index has already been defined.
                    // Skip this one.
                    continue;
                }

                // Register this bit so we know a check box with this index is already
                // defined.
                duplicationTester |= value;

                templateItem.SetTemplateTagValue(TemplateTagNameLabel, radioButtonDef.GetLabelForLanguage(this.language));
                templateItem.SetTemplateTagValue(TemplateTagNameValue, value.ToString());

                availableItemsText += templateItem.Compose();
            }

            templateAvailableItems.SetTemplateTagValue(TemplateTagNameResourceKey, "AvailableItems_Switch" + def.Index);
            templateAvailableItems.SetTemplateTagValue(TemplateTagNameItems, availableItemsText);

            this.xamlResources += templateAvailableItems.Compose();

            template.SetTemplateTagValue(TemplateTagNameLabel, castedUserDef.GetLabelForLanguage(this.language));
            template.SetTemplateTagValue(TemplateTagNameResourceKey, "AvailableItems_Switch" + def.Index);
            template.SetTemplateTagValue(TemplateTagNameBindSource, dataSrcName);

            return template.Compose(level * 4);
        }

        /// <summary>
        /// Generate data model property with the given integer slider definition.
        /// </summary>
        /// <param name="def">The integer slider definition.</param>
        /// <param name="level">The level in the hierarchy.</param>
        /// <returns>The generated XAML from the given group definition.</returns>
        private string GenerateIntSliderProperty(UserDefinitionBase def, int level)
        {
            if (def.Visible == false)
            {
                return null;
            }

            var castedUserDef = def as IntSliderDefinition;
            if (castedUserDef == null)
            {
                return null;
            }

            TemplateHelper template = this.InitAndGetTemplate(Templates.Slider);

            template.SetTemplateTagValue(
                TemplateTagNameLabel,
                def.GetLabelForLanguage(this.language));

            template.SetTemplateTagValue(
                TemplateTagNameMinValue,
                castedUserDef.MinValue.ToString());

            template.SetTemplateTagValue(
                TemplateTagNameMaxValue,
                castedUserDef.MaxValue.ToString());

            template.SetTemplateTagValue(
                "DeltaLevel",
                castedUserDef.DeltaLevel.ToString());

            template.SetTemplateTagValue(
                TemplateTagNameBindSource,
                "IntParam" + def.Index);

            return template.Compose();
        }

        /// <summary>
        /// Generate data model property with the given floating point slider definition.
        /// </summary>
        /// <param name="def">The floating point slider definition.</param>
        /// <param name="level">The level in the hierarchy.</param>
        /// <returns>The generated XAML from the given group definition.</returns>
        private string GenerateFloatSliderProperty(UserDefinitionBase def, int level)
        {
            if (def.Visible == false)
            {
                return null;
            }

            var castedUserDef = def as FloatSliderDefinition;
            if (castedUserDef == null)
            {
                return null;
            }

            TemplateHelper template = this.InitAndGetTemplate(Templates.Slider);

            template.ResetTemplateTags();

            template.SetTemplateTagValue(
                TemplateTagNameLabel,
                def.GetLabelForLanguage(this.language));

            template.SetTemplateTagValue(
                TemplateTagNameMinValue,
                castedUserDef.MinValue.ToString());

            template.SetTemplateTagValue(
                TemplateTagNameMaxValue,
                castedUserDef.MaxValue.ToString());

            template.SetTemplateTagValue(
                "DeltaLevel",
                castedUserDef.DeltaLevel.ToString());

            template.SetTemplateTagValue(
                TemplateTagNameBindSource,
                "FloatParam" + def.Index);

            return template.Compose();
        }

        /// <summary>
        /// Initialize and get the specified template helper.
        /// </summary>
        /// <param name="template">The template.</param>
        /// <returns>The template helper.</returns>
        private TemplateHelper InitAndGetTemplate(Templates template)
        {
            TemplateHelper helper = TemplateHelpers[(int)template];

            helper.ResetTemplateTags();

            return helper;
        }
    }
}
