﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq.Expressions;
using System.Reflection;
using System.Xml.Serialization;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.ObsoleteUserDataConverter.ObsoleteUserData
{
    /// <summary>
    /// The base definition for .udd/.usd (user data/shader definition)
    /// file serialization.
    /// </summary>
    [Serializable]
    public abstract class UserDefinitionBase : IXmlDocSerializable
    {
        /// <summary>Default value for Index property.</summary>
        private const int DefaultIndexValue = 0;

        /// <summary>Default value for Label property.</summary>
        private const string DefaultLabelValue = "";

        /// <summary>Default value for LabelEN property.</summary>
        private const string DefaultLabelENValue = "";

        /// <summary>Default value for Visible property.</summary>
        private const bool DefaultVisibleValue = true;

        /// <summary>
        /// A static dictionary contains property info as key and the smart property's definition.
        /// </summary>
        private static Dictionary<PropertyInfo, SmartPropertyDef> smartPropertieDefinitions =
            new Dictionary<PropertyInfo, SmartPropertyDef>();

        /// <summary>
        /// A dictionary contains the property name as key and the smart property's data.
        /// </summary>
        private Dictionary<string, SmartPropertyData> instanceSmartProperties =
            new Dictionary<string, SmartPropertyData>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static UserDefinitionBase()
        {
            Type myType = typeof(UserDefinitionBase);

            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("Index"), DefaultIndexValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("Label"), DefaultLabelValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("LabelEN"), DefaultLabelENValue);
            UserDefinitionBase.RegisterSmartProperty(myType.GetProperty("Visible"), DefaultVisibleValue);
        }

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserDefinitionBase()
        {
            this.InitializeSmartProperties();
        }

        /// <summary>
        /// Get or set the index of the control in the parent.
        /// </summary>
        [XmlAttribute("Index")]
        public int Index
        {
            get { return this.GetSmartPropertyValue(() => this.Index); }
            set { this.SetSmartPropertyValue(() => this.Index, value); }
        }

        /// <summary>
        /// Get or set the label of the control.
        /// </summary>
        [XmlAttribute("Label")]
        public string Label
        {
            get { return this.GetSmartPropertyValue(() => this.Label); }
            set { this.SetSmartPropertyValue(() => this.Label, value); }
        }

        /// <summary>
        /// Get or set the label of the control (English version).
        /// </summary>
        [XmlAttribute("Label_EN")]
        public string LabelEN
        {
            get { return this.GetSmartPropertyValue(() => this.LabelEN); }
            set { this.SetSmartPropertyValue(() => this.LabelEN, value); }
        }

        /// <summary>
        /// Get or set the visibility of the control.
        /// </summary>
        [XmlAttribute("Visible")]
        public bool Visible
        {
            get { return this.GetSmartPropertyValue(() => this.Visible); }
            set { this.SetSmartPropertyValue(() => this.Visible, value); }
        }

        /// <summary>
        /// Get label for the specified language.
        /// </summary>
        /// <param name="language">The language.</param>
        /// <returns>The label for the language.</returns>
        public string GetLabelForLanguage(string language)
        {
            if (string.Equals(language, "en", StringComparison.InvariantCultureIgnoreCase) == true)
            {
                return this.LabelEN;
            }
            else
            {
                return this.Label;
            }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool ReadXml(XmlDocSerializationContext context)
        {
            if (this.HasAttribute(context, "Index") == true)
            {
                this.Index = this.ReadAttribute(context, "Index", DefaultIndexValue);
            }

            if (this.HasAttribute(context, "Label") == true)
            {
                this.Label = this.ReadAttribute(context, "Label", DefaultLabelValue);
            }

            if (this.HasAttribute(context, "Label_EN") == true)
            {
                this.LabelEN = this.ReadAttribute(context, "Label_EN", DefaultLabelENValue);
            }

            if (this.HasAttribute(context, "Visible") == true)
            {
                this.Visible = this.ReadAttribute(context, "Visible", DefaultVisibleValue);
            }

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public virtual bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }

        /// <summary>
        /// Check if the given property value is explicitly assigned.
        /// Usage:
        /// <code>
        /// if (def.IsPropertyAssigned(() => def.Visible) == true)
        /// {
        ///     // The value of property 'Visible' of 'def' is assigned.
        /// }
        /// </code>
        /// </summary>
        /// <typeparam name="T">The type of the property.</typeparam>
        /// <param name="propertyExp">The expression that contains the property.</param>
        /// <returns>True if the property value is assigned.</returns>
        public bool IsPropertyAssigned<T>(Expression<Func<T>> propertyExp)
        {
            string propertyName = (propertyExp.Body as MemberExpression).Member.Name;

            return this.IsPropertyAssigned(propertyName);
        }

        /// <summary>
        /// Check if the given property value is explicitly assigned.
        /// Usage:
        /// <code>
        /// if (def.IsPropertyAssigned("Visible") == true)
        /// {
        ///     // The value of property 'Visible' of 'def' is assigned.
        /// }
        /// </code>
        /// </summary>
        /// <param name="propertyName">The name of the property.</param>
        /// <returns>True if the property value is assigned.</returns>
        public bool IsPropertyAssigned(string propertyName)
        {
            SmartPropertyData data;
            if (this.instanceSmartProperties.TryGetValue(propertyName, out data) == false)
            {
                throw new ArgumentException("The property '" + propertyName + "' isn't registered as smart property.");
            }

            return data.IsAssigned;
        }

        /// <summary>
        /// Apply template to the definition.
        /// </summary>
        /// <param name="template">The template.</param>
        public virtual void ApplyTemplate(TemplateDefinition template)
        {
            template.ApplyTemplateValue(this, "Label");
            template.ApplyTemplateValue(this, "LabelEN");
            template.ApplyTemplateValue(this, "Visible");
        }

        /// <summary>
        /// This method should be called after deserialized from Xml
        /// so that the definition can prepare its data before it can
        /// be used.
        /// </summary>
        public virtual void PostLoad()
        {
            // The method does not do anything by default.
        }

        /// <summary>
        /// Register a smart property that has the knowledge about
        /// whether it's assigned and it's default value.
        /// </summary>
        /// <param name="propertyInfo">The property info.</param>
        /// <param name="defaultValue">The default value.</param>
        protected static void RegisterSmartProperty(
            PropertyInfo propertyInfo,
            object defaultValue)
        {
            smartPropertieDefinitions.Add(
                propertyInfo,
                new SmartPropertyDef(propertyInfo, defaultValue));
        }

        /// <summary>
        /// Get smart property value.
        /// Usage:
        /// <code>return this.GetSmartPropertyValue(() => this.Visible);</code>
        /// </summary>
        /// <typeparam name="T">The type of the property.</typeparam>
        /// <param name="propertyExp">The expression that contains the property.</param>
        /// <returns>The property value.</returns>
        protected T GetSmartPropertyValue<T>(Expression<Func<T>> propertyExp)
        {
            string propertyName = (propertyExp.Body as MemberExpression).Member.Name;

            SmartPropertyData data;
            if (this.instanceSmartProperties.TryGetValue(propertyName, out data) == false)
            {
                throw new ArgumentException("The property '" + propertyName + "' isn't registered as smart property.");
            }

            return (T)data.Value;
        }

        /// <summary>
        /// Set smart property value.
        /// Usage:
        /// <code>return this.SetSmartPropertyValue(() => this.Visible, true);</code>
        /// </summary>
        /// <typeparam name="T">The type of the property.</typeparam>
        /// <param name="propertyExp">The expression that contains the property.</param>
        /// <param name="value">The value to set to the property.</param>
        protected void SetSmartPropertyValue<T>(Expression<Func<T>> propertyExp, T value)
        {
            string propertyName = (propertyExp.Body as MemberExpression).Member.Name;

            SmartPropertyData data;
            if (this.instanceSmartProperties.TryGetValue(propertyName, out data) == false)
            {
                throw new ArgumentException("The property '" + propertyName + "' isn't registered as smart property.");
            }

            data.Value = value;
        }

        /// <summary>
        /// Initialize all the smart properties of the caller's type.
        /// </summary>
        private void InitializeSmartProperties()
        {
            Type myType = this.GetType();

            // Get all the properties in the type.
            PropertyInfo[] properties = myType.GetProperties();

            // Find matches in the registered smart properties,
            // and create smart property data for them.
            foreach (PropertyInfo info in properties)
            {
                var declaredPropertyInfo = info.DeclaringType.GetProperty(info.Name);

                SmartPropertyDef def;
                if (smartPropertieDefinitions.TryGetValue(declaredPropertyInfo, out def) == true)
                {
                    this.instanceSmartProperties[def.PropertyInfo.Name] =
                        new SmartPropertyData(def.PropertyInfo);
                }
            }
        }

        /// <summary>
        /// The definition of a smart property.
        /// </summary>
        private class SmartPropertyDef
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="propertyInfo">The property info.</param>
            /// <param name="defaultValue">The default value of the property.</param>
            public SmartPropertyDef(PropertyInfo propertyInfo, object defaultValue)
            {
                if (propertyInfo == null)
                {
                    throw new ArgumentNullException("propertyInfo");
                }

                if (TypeConversionUtility.CanConvert(propertyInfo.PropertyType, defaultValue) == false)
                {
                    throw new ArgumentException("The default value cannot be assigned to the property.");
                }

                this.PropertyInfo = propertyInfo;
                this.DefaultValue = defaultValue;
            }

            /// <summary>
            /// Get the property info.
            /// </summary>
            public PropertyInfo PropertyInfo { get; private set; }

            /// <summary>
            /// Get the default value of the property.
            /// </summary>
            public object DefaultValue { get; private set; }
        }

        /// <summary>
        /// Stores information and value of a smart property of control definitions.
        /// </summary>
        private class SmartPropertyData
        {
            /// <summary>The property value.</summary>
            private object propertyValue = null;

            /// <summary>
            /// Constructor
            /// </summary>
            /// <param name="propertyInfo">The property info.</param>
            public SmartPropertyData(PropertyInfo propertyInfo)
            {
                if (propertyInfo == null)
                {
                    throw new ArgumentNullException("propertyInfo");
                }

                this.PropertyInfo = propertyInfo;
                this.IsAssigned = false;
            }

            /// <summary>
            /// Get the property info.
            /// </summary>
            public PropertyInfo PropertyInfo { get; private set; }

            /// <summary>
            /// Get the flag indicating whether the property has been assigned.
            /// </summary>
            public bool IsAssigned { get; private set; }

            /// <summary>
            /// Get or set the value of the property.
            /// </summary>
            public object Value
            {
                get
                {
                    if (this.IsAssigned == false)
                    {
                        return this.DefaultValue;
                    }
                    else
                    {
                        return this.propertyValue;
                    }
                }

                set
                {
                    this.IsAssigned = true;

                    if (TypeConversionUtility.CanConvert(this.PropertyInfo.PropertyType, value) == false)
                    {
                        throw new ArgumentException("The value cannot be assigned to the property.");
                    }

                    this.propertyValue = value;
                }
            }

            /// <summary>
            /// Get the default value of the smart property.
            /// </summary>
            public object DefaultValue
            {
                get
                {
                    SmartPropertyDef def;
                    if (UserDefinitionBase.smartPropertieDefinitions.TryGetValue(
                        this.PropertyInfo,
                        out def) == false)
                    {
                        throw new InvalidOperationException("The property is not registered.");
                    }

                    return def.DefaultValue;
                }
            }
        }
    }
}
