﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.ToolDevelopmentKit
{
    using System;
    using System.Collections;
    using System.Diagnostics;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    /// デバッグビルド時に条件が満たされることを保証する際に利用するクラスです。
    /// リリースビルド時にも条件を評価するには Ensure クラスを利用します。
    /// </summary>
    public static class Assertion
    {
        /// <summary>
        /// 引数に関するアサートを送出します。
        /// </summary>
        public static class Argument
        {
            /// <summary>
            /// 引数が true であるか判定し、true でない場合はアサートします。
            /// </summary>
            /// <param name="status">引数です。</param>
            [Conditional("DEBUG")]
            public static void True(bool status)
            {
                Debug.Assert(status, "invalid argument assertion.");
            }

            /// <summary>
            /// 引数がnullでは無いか判断し、不正な場合はアサートします。
            /// </summary>
            /// <param name="argument">引数です。</param>
            [Conditional("DEBUG")]
            public static void NotNull(object argument)
            {
                Debug.Assert(argument != null, "argument null assertion.");
            }

            /// <summary>
            /// 文字列引数が妥当か判断し、不正な場合はアサートします。
            /// </summary>
            /// <param name="stringArg">文字列引数です。</param>
            [Conditional("DEBUG")]
            public static void StringNotEmpty(string stringArg)
            {
                Assertion.Argument.NotNull(stringArg);
                Assertion.Argument.True(stringArg != string.Empty);
            }

            /// <summary>
            /// 行列引数が空の行列ではないか判断し、不正な場合はアサートします。
            /// </summary>
            /// <param name="argument">行列引数です。</param>
            [Conditional("DEBUG")]
            public static void ArrayNotEmpty(Array argument)
            {
                Assertion.Argument.EnumerableNotEmpty(argument);
            }

            /// <summary>
            /// 列挙可能型が空ではないか判断し、不正な場合はアサートします。
            /// </summary>
            /// <param name="argument">行列引数です。</param>
            [Conditional("DEBUG")]
            public static void EnumerableNotEmpty(IEnumerable argument)
            {
                Assertion.Argument.NotNull(argument);
                Assertion.Argument.True(EnumerableUtility.NotEmpty(argument));
            }
        }

        /// <summary>
        /// 操作に関するアサートを送出します。
        /// </summary>
        public static class Operation
        {
            /// <summary>
            /// 操作が true であるか判定し、true でない場合はアサートします。
            /// </summary>
            /// <param name="status">状態です。</param>
            [Conditional("DEBUG")]
            public static void True(bool status)
            {
                True(status, "invalid operation assertion.");
            }

            /// <summary>
            /// 操作が true であるか判定し、true でない場合はアサートします。
            /// </summary>
            /// <param name="status">状態です。</param>
            /// <param name="debugMessage">
            /// デバック用文字列です。
            /// ユーザに表示するメッセージではないのでローカライズの必要はありません。
            /// </param>
            [Conditional("DEBUG")]
            public static void True(bool status, string debugMessage)
            {
                if (status == false)
                {
                    Debug.Assert(status, debugMessage);
                }
            }

            /// <summary>
            /// オブジェクトがnullであるか判定し、nullでない場合はアサートします。
            /// </summary>
            /// <param name="referenceObject">オブジェクトです。</param>
            [Conditional("DEBUG")]
            public static void ObjectNull(object referenceObject)
            {
                Assertion.Operation.True(referenceObject == null);
            }

            /// <summary>
            /// オブジェクトがnullで無いか判定し、nullの場合はアサートします。
            /// </summary>
            /// <param name="referenceObject">オブジェクトです。</param>
            [Conditional("DEBUG")]
            public static void ObjectNotNull(object referenceObject)
            {
                Assertion.Operation.True(referenceObject != null);
            }

            /// <summary>
            /// 除算の分母が0.0fであるか判定し、0.0fの場合はアサートします。
            /// </summary>
            /// <param name="divider">除算の分母です。</param>
            [Conditional("DEBUG")]
            public static void DividerNotZero(float divider)
            {
                if (FloatUtility.NearlyEqual(divider, 0f))
                {
                    Debug.Assert(false, "DivideByZero!");
                }
            }
        }
    }
}
