﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    using System;

    /// <summary>
    /// 配列用ユーティリティです。
    /// </summary>
    public static class ArrayUtility
    {
        //-----------------------------------------------------------------
        // 状態
        //-----------------------------------------------------------------

        /// <summary>
        /// リストがnullか空の状態か判定します。
        /// </summary>
        /// <typeparam name="TItem">リストの要素型です。</typeparam>
        /// <param name="source">リストです。</param>
        /// <returns>リストがnullか空なら true を返します。</returns>
        public static bool IsNullOrEmpty<TItem>(TItem[] source)
        {
            return source == null || source.Length == 0;
        }

        /// <summary>
        /// リストの要素がすべて同じ値か判定します。
        /// </summary>
        /// <typeparam name="TItem">リストの要素型です。</typeparam>
        /// <param name="source">リストです。</param>
        /// <returns>すべて同じ値なら true を返します。</returns>
        public static bool Unified<TItem>(TItem[] source)
        {
            Ensure.Argument.ArrayNotEmpty(source);

            TItem firstValue = source[0];

            foreach (var value in source)
            {
                if (!value.Equals(firstValue))
                {
                    return false;
                }
            }

            return true;
        }

        //-----------------------------------------------------------------
        // 移動、コピー
        //-----------------------------------------------------------------

        /// <summary>
        /// リストの内容をコピーします。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元のリストです。</param>
        /// <param name="destination">コピー先のリストです。</param>
        public static void Copy<TItem>(TItem[] source, TItem[] destination)
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.NotNull(destination);

            Ensure.Argument.True(source.Length == destination.Length);

            for (int i = 0; i < source.Length; i++)
            {
                destination[i] = source[i];
            }
        }

        /// <summary>
        /// リストを作成し、内容をコピーします。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元のリストです。</param>
        /// <returns>新しいリストです。</returns>
        public static TItem[] CreateAndCopy<TItem>(TItem[] source)
        {
            Ensure.Argument.NotNull(source);
            TItem[] newArray = new TItem[source.Length];
            Copy(source, newArray);

            return newArray;
        }

        /// <summary>
        /// リストが存在すればリストを作成し、内容をコピーします。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元のリストです。</param>
        /// <returns>新しいリストです。</returns>
        public static TItem[] CreateAndCopyIfExist<TItem>(TItem[] source)
        {
            return (source != null) ? CreateAndCopy(source) : null;
        }

        /// <summary>
        /// リストの内容を複製してコピーします。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元のリストです。</param>
        /// <param name="destination">コピー先のリストです。</param>
        public static void CloneCopy<TItem>(TItem[] source, TItem[] destination)
            where TItem : class, ICloneable
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.NotNull(destination);

            Ensure.Argument.True(source.Length == destination.Length);

            for (int i = 0; i < source.Length; i++)
            {
                if (source[i] != null)
                {
                    destination[i] = source[i].Clone() as TItem;
                }
                else
                {
                    destination[i] = null;
                }
            }
        }

        /// <summary>
        /// リストが存在すればリストを作成し、内容をコピーします。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">コピー元のリストです。</param>
        /// <returns>新しいリストです。</returns>
        public static TItem[] CloneCopyIfExsist<TItem>(TItem[] source)
            where TItem : class, ICloneable
        {
            if (source == null)
            {
                return null;
            }
            else
            {
                TItem[] destination = new TItem[source.Length];
                CloneCopy<TItem>(source, destination);

                return destination;
            }
        }

        /// <summary>
        /// リストの内容を全てセットします。
        /// </summary>
        /// <remarks>
        /// sourceとdestinationの要素は必ず同数、NotNullでなくてはなりません。
        /// </remarks>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="source">セット元のリストです。</param>
        /// <param name="destination">セット先のリストです。</param>
        public static void SetAll<TItem>(TItem[] source, TItem[] destination)
            where TItem : class, ISettable
        {
            Ensure.Argument.NotNull(source);
            Ensure.Argument.NotNull(destination);

            Ensure.Argument.True(source.Length == destination.Length);

            for (int i = 0; i < source.Length; i++)
            {
                if (source[i] != null)
                {
                    destination[i].Set(source[i]);
                }
                else
                {
                    destination[i] = null;
                }
            }
        }

        /// <summary>
        /// リストの値をすべて特定の値に設定します。
        /// </summary>
        /// <typeparam name="TItem">リストのテンプレート型です。</typeparam>
        /// <param name="value">設定する値です。</param>
        /// <param name="destination">セット先のリストです。</param>
        public static void FillAll<TItem>(TItem value, TItem[] destination)
        {
            Ensure.Argument.NotNull(destination);

            for (int i = 0; i < destination.Length; i++)
            {
                destination[i] = value;
            }
        }

        /// <summary>
        /// 配列を連結します。
        /// </summary>
        /// <typeparam name="TItem">連結する配列の型です。</typeparam>
        /// <param name="value1">前に連結する元配列です。</param>
        /// <param name="value2">後ろに連結する配列です。</param>
        /// <returns>連結した配列です。</returns>
        public static TItem[] Concatenate<TItem>(TItem[] value1, TItem[] value2)
        {
            Ensure.Argument.NotNull(value1);
            Ensure.Argument.NotNull(value2);

            TItem[] result = new TItem[value1.Length + value2.Length];
            value1.CopyTo(result, 0);
            value2.CopyTo(result, value1.Length);

            return result;
        }

        /// <summary>
        /// 部分配列を生成します。
        /// </summary>
        /// <typeparam name="TItem">配列の型です。</typeparam>
        /// <param name="src">元配列です。</param>
        /// <param name="startIndex">開始インデックスです。</param>
        /// <param name="count">部分配列のデータ数です。</param>
        /// <returns>部分配列です。</returns>
        public static TItem[] SubArray<TItem>(TItem[] src, int startIndex, int count)
        {
            Ensure.Argument.NotNull(src);
            Ensure.Argument.True(startIndex < src.Length);
            Ensure.Argument.True(startIndex + count <= src.Length);

            TItem[] result = new TItem[count];

            Array.Copy(src, startIndex, result, 0, count);

            return result;
        }

        /// <summary>
        /// 指定インデックスから後ろの部分配列を生成します。
        /// </summary>
        /// <typeparam name="TItem">配列の型です。</typeparam>
        /// <param name="src">元配列です。</param>
        /// <param name="startIndex">開始インデックスです。</param>
        /// <returns>部分配列です。</returns>
        public static TItem[] SubArray<TItem>(TItem[] src, int startIndex)
        {
            Ensure.Argument.NotNull(src);
            Ensure.Argument.True(startIndex < src.Length);

            int count = src.Length - startIndex;
            TItem[] result = new TItem[count];

            Array.Copy(src, startIndex, result, 0, count);

            return result;
        }

        /// <summary>
        /// 配列を違う型にキャストしてコピーします。
        /// </summary>
        /// <typeparam name="TSrc">変換元の型です。</typeparam>
        /// <typeparam name="TDst">変換先の型です。</typeparam>
        /// <param name="src">変換元の配列です。</param>
        /// <returns>キャストした配列を返します。</returns>
        public static TDst[] Copy<TSrc, TDst>(TSrc[] src)
        {
            Ensure.Argument.NotNull(src);

            TDst[] result = new TDst[src.Length];
            src.CopyTo(result, 0);

            return result;
        }

        //-----------------------------------------------------------------
        // 比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 配列を比較します。
        /// </summary>
        /// <typeparam name="TItem">配列要素の型パラメータです。</typeparam>
        /// <param name="lhs">左辺配列です。</param>
        /// <param name="rhs">右辺配列です。</param>
        /// <returns>左辺配列と右辺配列が等値であれば true を返します。</returns>
        public static bool Equals<TItem>(TItem[] lhs, TItem[] rhs)
        {
            if (lhs == null)
            {
                return rhs == null;
            }

            if (rhs == null)
            {
                return false;
            }

            int count = lhs.Length;
            if (rhs.Length != count)
            {
                return false;
            }

            for (int i = 0; i < count; i++)
            {
                TItem lhsItem = lhs[i];
                TItem rhsItem = rhs[i];
                if (lhsItem == null)
                {
                    if (rhsItem != null)
                    {
                        return false;
                    }
                }
                else if (!lhsItem.Equals(rhsItem))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// ハッシュコードを取得します。
        /// </summary>
        /// <typeparam name="TItem">配列要素の型パラメータです。</typeparam>
        /// <param name="array">ハッシュコードを計算する配列です。</param>
        /// <returns>配列のハッシュコードです。</returns>
        public static int GetHashCode<TItem>(TItem[] array)
        {
            if (array == null)
            {
                return 0;
            }

            int hashCode = typeof(TItem).GetHashCode();
            foreach (TItem item in array)
            {
                hashCode ^= item.GetHashCode();
            }

            return hashCode;
        }
    }
}
