﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit
{
    using System.Diagnostics;

    /// <summary>
    /// 行列の３次元の平行移動に関する演算関数です。
    /// </summary>
    /// <typeparam name="TMatrix">テンプレート行列型です。</typeparam>
    internal static class Matrix3DTransformFunction<TMatrix> where TMatrix : IMatrix, new()
    {
        /// <summary>
        /// Transform に対応する行列を生成します。
        /// </summary>
        /// <param name="mtx">設定する行列です。</param>
        /// <param name="trfm"> SRTです。</param>
        public static void SetTransform(TMatrix mtx, Transform trfm)
        {
            Ensure.Argument.NotNull(mtx);

            DebugMatrixRowColumnCount(mtx);

            Matrix3DScaleRotateFunction<TMatrix>.SetRotateEuler(mtx, trfm.Rotate);
            for (int i = 0; i < 3; i++)
            {
                mtx[i, 0] *= trfm.Scale.X;
                mtx[i, 1] *= trfm.Scale.Y;
                mtx[i, 2] *= trfm.Scale.Z;
            }

            mtx[0, 3] = trfm.Translate.X;
            mtx[1, 3] = trfm.Translate.Y;
            mtx[2, 3] = trfm.Translate.Z;
        }

        /// <summary>
        /// 行列からSRTを取得します。
        /// </summary>
        /// <param name="source">行列です。</param>
        /// <param name="trfm">設定するSRTです。</param>
        public static void GetTransform(TMatrix source, Transform trfm)
        {
            Ensure.Argument.NotNull(source);

            TMatrix mtx = new TMatrix();
            mtx.Set(source);
            for (int iaxis = 0; iaxis < 3; ++iaxis)
            {
                Vector3 f = new Vector3(mtx[0, iaxis], mtx[1, iaxis], mtx[2, iaxis]);
                trfm.Scale[iaxis] = f.Length;
                if (trfm.Scale[iaxis] != 0)
                {
                    if (iaxis == 2)
                    {
                        Vector3 ax = new Vector3(mtx[0, 0], mtx[1, 0], mtx[2, 0]);
                        Vector3 ay = new Vector3(mtx[0, 1], mtx[1, 1], mtx[2, 1]);
                        Vector3 cz = Vector3.Cross(ax, ay);
                        if (Vector3.Dot(cz, f) < 0.0F)
                        {
                            trfm.Scale[iaxis] = -trfm.Scale[iaxis];
                        }
                    }

                    float d = 1.0F / trfm.Scale[iaxis];
                    mtx[0, iaxis] *= d;
                    mtx[1, iaxis] *= d;
                    mtx[2, iaxis] *= d;
                }
            }

            Matrix3DScaleRotateFunction<TMatrix>.GetRotateEuler(mtx, trfm.Rotate);

            trfm.Translate[0] = mtx[0, 3];
            trfm.Translate[1] = mtx[1, 3];
            trfm.Translate[2] = mtx[2, 3];
        }

        /// <summary>
        /// 行列の行数と列数が演算に対して適切であることを確認します。
        /// </summary>
        /// <param name="mtx">対象の行列です。</param>
        [Conditional("DEBUG")]
        private static void DebugMatrixRowColumnCount(TMatrix mtx)
        {
            Debug.Assert(mtx.RowCount >= 3, "Unexpected case!");
            Debug.Assert(mtx.ColumnCount >= 4, "Unexpected case!");
        }
    }
}
