﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit.Xml
{
    using System;
    using System.Xml.Serialization;
    using NintendoWare.ToolDevelopmentKit;
    using NintendoWare.ToolDevelopmentKit.Conversion;

    /// <summary>
    /// 3 次元のベクトルのシリアライズ用クラスです。
    /// </summary>
    [CoupledType(typeof(Vector3))]
    public class Vector3Xml : IStringSettable, ISettable
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public Vector3Xml()
        {
        }

        /// <summary>
        /// X,Y,Z座標を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="x">X座標です。</param>
        /// <param name="y">Y座標です。</param>
        /// <param name="z">Z座標です。</param>
        public Vector3Xml(float x, float y, float z)
        {
            this.Set(x, y, z);
        }

        /// <summary>
        /// X 成分を取得または設定します。
        /// </summary>
        [XmlElement("x")]
        public float X { get; set; }

        /// <summary>
        /// Y 成分を取得または設定します。
        /// </summary>
        [XmlElement("y")]
        public float Y { get; set; }

        /// <summary>
        /// Z 成分を取得または設定します。
        /// </summary>
        [XmlElement("z")]
        public float Z { get; set; }

        //-----------------------------------------------------------------
        // 設定
        //-----------------------------------------------------------------

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source as Vector3Xml);
        }

        /// <summary>
        /// Vector3Xml を設定します。
        /// </summary>
        /// <param name="source">設定する Vector3Xml です。</param>
        public void Set(Vector3Xml source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source.X, source.Y, source.Z);
        }

        /// <summary>
        /// X,Y,Z座標を指定の値に設定するコンストラクタです。
        /// </summary>
        /// <param name="x">X座標です。</param>
        /// <param name="y">Y座標です。</param>
        /// <param name="z">Z座標です。</param>
        public void Set(float x, float y, float z)
        {
            this.X = x;
            this.Y = y;
            this.Z = z;
        }

        /// <summary>
        /// 状態を設定します。
        /// </summary>
        /// <param name="source">文字列です。</param>
        public void Set(string source)
        {
            Ensure.Argument.StringNotEmpty(source);
            string[] elements = source.Split(" ".ToCharArray());
            Ensure.Argument.True(elements.Length == 3);

            this.X = float.Parse(elements[0]);
            this.Y = float.Parse(elements[1]);
            this.Z = float.Parse(elements[2]);
        }

        /// <summary>
        /// 現在のオブジェクトを表す文字列を返します。
        /// </summary>
        /// <returns>現在のオブジェクトを表す文字列です。</returns>
        public override string ToString()
        {
            return string.Format("{0} {1} {2}", this.X, this.Y, this.Z);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return
                this.X.GetHashCode() ^
                this.Y.GetHashCode() ^
                this.Z.GetHashCode();
        }

        //-----------------------------------------------------------------
        // 比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// [sealedクラスならprivateとします。]
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            return this.Equals(other as Vector3Xml);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// [sealedクラスならprivateとします。]
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        protected bool Equals(Vector3Xml other)
        {
            Ensure.Argument.NotNull(other);
            return this.X == other.X && this.Y == other.Y && this.Z == other.Z;
        }
    }
}
