﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit.Xml.Complex
{
    using System.Collections.Generic;
    using NintendoWare.ToolDevelopmentKit.Collections;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// 複合 XML ドキュメントです。
    /// ルート直下の要素毎に入出力処理を切り替えられる XML ドキュメントです。
    /// </summary>
    public class XmlComplexDocument : ObservableObject, IXmlComplexDocument
    {
        private string rootElementName;
        private IList<IXmlContent> contents = new ObservableInstanceList<IXmlContent>();

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="rootElementName">ルート要素の名前です。</param>
        public XmlComplexDocument(string rootElementName)
        {
            Ensure.Argument.NotNull(rootElementName);

            this.Initialize(rootElementName);
        }

        /// <summary>
        /// コピーコンストラクタです。
        /// </summary>
        /// <param name="source">コピー元の実体です。</param>
        public XmlComplexDocument(XmlComplexDocument source)
        {
            Ensure.Argument.NotNull(source);

            this.Set(source);
        }

        /// <summary>
        /// ルート要素の名前を取得または設定します。
        /// </summary>
        public string RootElementName
        {
            get
            {
                return this.rootElementName;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.rootElementName = value;
                OnPropertyChanged("RootElementName");
            }
        }

        /// <summary>
        /// コンテンツリストを取得または設定します。
        /// </summary>
        public IList<IXmlContent> Contents
        {
            get
            {
                return this.contents;
            }

            set
            {
                Ensure.Argument.NotNull(value);

                this.contents = value;
                OnPropertyChanged("Contents");
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            return new XmlComplexDocument(this);
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        public void Set(object source)
        {
            this.Set(source is IXmlComplexDocument);
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="obj">比較対象です。</param>
        /// <returns>比較結果です。</returns>
        public override bool Equals(object obj)
        {
            return CheckClassEqualityUtil<IXmlComplexDocument>.Equals(this, obj, this.Equals);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            int result = this.RootElementName.GetHashCode();

            foreach (IXmlContent content in this.Contents)
            {
                result ^= content.GetHashCode();
            }

            return result;
        }

        /// <summary>
        /// オブジェクトを設定します。
        /// </summary>
        /// <param name="source">設定するオブジェクトです。</param>
        protected void Set(IXmlComplexDocument source)
        {
            Ensure.Argument.NotNull(source);

            this.Initialize(source.RootElementName);

            foreach (IXmlContent content in source.Contents)
            {
                this.Contents.Add(content.Clone() as IXmlContent);
            }
        }

        /// <summary>
        /// オブジェクトを初期化します。
        /// </summary>
        /// <param name="rootElementName">ルート要素の名前です。</param>
        protected void Initialize(string rootElementName)
        {
            Ensure.Argument.NotNull(rootElementName);

            this.rootElementName = rootElementName;

            this.Contents.Clear();
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="obj">比較対象です。</param>
        /// <returns>比較結果です。</returns>
        protected bool Equals(IXmlComplexDocument obj)
        {
            if (obj == null)
            {
                return false;
            }

            if (this.RootElementName != obj.RootElementName)
            {
                return false;
            }

            if (this.Contents.Count != obj.Contents.Count)
            {
                return false;
            }

            for (int i = 0; i < this.Contents.Count; i++)
            {
                if (!this.Contents[i].Equals(obj.Contents[i]))
                {
                    return false;
                }
            }

            return true;
        }
    }
}
