﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit.Logs
{
    using System.Collections.Generic;
    using NintendoWare.ToolDevelopmentKit.Collections;

    /// <summary>
    /// ログを収集するログコレクタです。
    /// </summary>
    public class LogCollector : ILogCollector
    {
        //-----------------------------------------------------------------
        private readonly IList<ILog> fatal = new ObservableInstanceList<ILog>();
        private readonly IList<ILog> error = new ObservableInstanceList<ILog>();
        private readonly IList<ILog> warn = new ObservableInstanceList<ILog>();
        private readonly IList<ILog> info = new ObservableInstanceList<ILog>();
        private readonly IList<ILog> debug = new ObservableInstanceList<ILog>();

        private ILogger logger;

        //-----------------------------------------------------------------
        // オブジェクトの生成
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        public LogCollector()
        {
        }

        //-----------------------------------------------------------------
        // プロパティの取得または設定
        //-----------------------------------------------------------------

        /// <summary>
        /// ログ収集対象のロガーを取得または設定します。
        /// </summary>
        public ILogger Logger
        {
            get
            {
                return this.logger;
            }

            set
            {
                if (this.logger != null)
                {
                    this.logger.Fatal.LogListener -= this.AddFatalLog;
                    this.logger.Error.LogListener -= this.AddErrorLog;
                    this.logger.Warn.LogListener -= this.AddWarnLog;
                    this.logger.Info.LogListener -= this.AddInfoLog;
                    this.logger.Debug.LogListener -= this.AddDebugLog;
                }

                this.logger = value;

                if (this.logger != null)
                {
                    this.logger.Fatal.LogListener += this.AddFatalLog;
                    this.logger.Error.LogListener += this.AddErrorLog;
                    this.logger.Warn.LogListener += this.AddWarnLog;
                    this.logger.Info.LogListener += this.AddInfoLog;
                    this.logger.Debug.LogListener += this.AddDebugLog;
                }
            }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 致命的な障害ログのログリストです。
        /// </summary>
        public IList<ILog> Fatal
        {
            get
            {
                return this.fatal;
            }
        }

        /// <summary>
        /// エラーログのログリストです。
        /// </summary>
        public IList<ILog> Error
        {
            get
            {
                return this.error;
            }
        }

        /// <summary>
        /// 警告ログのログリストです。
        /// </summary>
        public IList<ILog> Warn
        {
            get
            {
                return this.warn;
            }
        }

        /// <summary>
        /// 情報ログのログリストです。
        /// </summary>
        public IList<ILog> Info
        {
            get
            {
                return this.info;
            }
        }

        /// <summary>
        /// デバッグ用ログのログリストです。
        /// </summary>
        public IList<ILog> Debug
        {
            get
            {
                return this.debug;
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public object Clone()
        {
            LogCollector collector = new LogCollector();
            foreach (ILog log in this.Fatal)
            {
                collector.Fatal.Add((ILog)log.Clone());
            }

            foreach (ILog log in this.Error)
            {
                collector.Error.Add((ILog)log.Clone());
            }

            foreach (ILog log in this.Warn)
            {
                collector.Warn.Add((ILog)log.Clone());
            }

            foreach (ILog log in this.Info)
            {
                collector.Info.Add((ILog)log.Clone());
            }

            foreach (ILog log in this.Debug)
            {
                collector.Debug.Add((ILog)log.Clone());
            }

            return collector;
        }

        //-----------------------------------------------------------------
        // 収集したログへのアクセス
        //-----------------------------------------------------------------

        /// <summary>
        /// 収集したログをクリアします。
        /// </summary>
        public void Clear()
        {
            this.Fatal.Clear();
            this.Error.Clear();
            this.Warn.Clear();
            this.Info.Clear();
            this.Debug.Clear();
        }

        /// <summary>
        /// ログを持っているかどうかを返します。
        /// </summary>
        /// <returns>ログを持っている場合に true を返します。</returns>
        public bool HasLog()
        {
            return
                (this.Fatal.Count > 0) ||
                (this.Error.Count > 0) ||
                (this.Warn.Count > 0) ||
                (this.Info.Count > 0) ||
                (this.Debug.Count > 0);
        }

        /// <summary>
        /// 致命的な障害かエラーがあるかどうかを返します。
        /// </summary>
        /// <returns>致命的な障害かエラーがある場合に true を返します。</returns>
        public bool HasFatalOrErrorLog()
        {
            return
                (this.Fatal.Count > 0) ||
                (this.Error.Count > 0);
        }

        /// <summary>
        /// 致命的な障害かエラーか警告があるかどうかを返します。
        /// </summary>
        /// <returns>致命的な障害かエラーか警告がある場合に true を返します。</returns>
        public bool HasFatalOrErrorOrWarnLog()
        {
            return
                (this.Fatal.Count > 0) ||
                (this.Error.Count > 0) ||
                (this.Warn.Count > 0);
        }

        //-----------------------------------------------------------------
        // 同値比較
        //-----------------------------------------------------------------

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        public override bool Equals(object other)
        {
            if (other == this)
            {
                return true;
            }

            if ((other == null) || (other.GetType() != GetType()))
            {
                return false;
            }

            return this.Equals(other as LogCollector);
        }

        /// <summary>
        /// ハッシュ値を取得します。
        /// </summary>
        /// <returns>ハッシュ値です。</returns>
        public override int GetHashCode()
        {
            return
                this.Fatal.GetHashCode() ^
                this.Error.GetHashCode() ^
                this.Warn.GetHashCode() ^
                this.Info.GetHashCode() ^
                this.Debug.GetHashCode();
        }

        /// <summary>
        /// 等値であるかどうか比較します。
        /// </summary>
        /// <param name="other">比較対象です。</param>
        /// <returns>等値であれば true を返します。</returns>
        protected bool Equals(LogCollector other)
        {
            Ensure.Argument.NotNull(other);
            return
                this.Fatal.Equals(other.Fatal) &&
                this.Error.Equals(other.Error) &&
                this.Warn.Equals(other.Warn) &&
                this.Info.Equals(other.Info) &&
                this.Debug.Equals(other.Debug);
        }

        // 致命的な障害ログの追加
        private void AddFatalLog(ILog log)
        {
            this.Fatal.Add(log);
        }

        // エラーログの追加
        private void AddErrorLog(ILog log)
        {
            this.Error.Add(log);
        }

        // 警告ログの追加
        private void AddWarnLog(ILog log)
        {
            this.Warn.Add(log);
        }

        // 情報ログの追加
        private void AddInfoLog(ILog log)
        {
            this.Info.Add(log);
        }

        // デバッグ用ログの追加
        private void AddDebugLog(ILog log)
        {
            this.Debug.Add(log);
        }
    }
}
