﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.ToolDevelopmentKit.ComponentModel
{
    using System;
    using System.ComponentModel;
    using System.Linq.Expressions;

    /// <summary>
    /// 通知機能付きオブジェクトです。
    /// </summary>
    [Serializable]
    public class ObservableObject : INotifyPropertyChanged
    {
        /// <summary>
        /// プロパティ変更イベント
        /// </summary>
        [NonSerialized]
        private PropertyChangedEventHandler propertyChanged;

        /// <summary>
        /// プロパティ変更イベント
        /// </summary>
        public event PropertyChangedEventHandler PropertyChanged
        {
            add { propertyChanged += value; }
            remove { propertyChanged -= value; }
        }

        /// <summary>
        /// プロパティ変更イベントの発動です。
        /// </summary>
        /// <param name="args">プロパティ変更イベントの引数です。</param>
        protected virtual void OnPropertyChanged(PropertyChangedEventArgs args)
        {
            if (this.propertyChanged != null)
            {
                this.propertyChanged(this, args);
            }
        }

        /// <summary>
        /// プロパティ変更イベントの発動です。
        /// </summary>
        /// <param name="propertyName">変更したプロパティの名前です。</param>
        protected virtual void OnPropertyChanged(string propertyName)
        {
            this.OnPropertyChanged(new PropertyChangedEventArgs(propertyName));
        }

        /// <summary>
        /// プロパティ変更イベントの発動です。
        /// </summary>
        /// <typeparam name="TProperty">プロパティの型です。</typeparam>
        /// <param name="propertyNameExpression">変更したプロパティの名前の式です。</param>
        protected virtual void OnPropertyChanged<TProperty>(
            Expression<Func<TProperty>> propertyNameExpression)
        {
            Ensure.Argument.NotNull(propertyNameExpression);

            // TODO:無駄な部分があるので、最適化する必要あり。
            MemberExpression memberExpression =
                propertyNameExpression.Body as MemberExpression;
            Ensure.Operation.ObjectNotNull(memberExpression);

            ConstantExpression senderExpression =
                memberExpression.Expression as ConstantExpression;
            Ensure.Operation.ObjectNotNull(senderExpression);

            this.OnPropertyChanged(memberExpression.Member.Name);
        }
    }
}
