﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.ToolDevelopmentKit.Collections
{
    using System.Collections;
    using System.Collections.Generic;
    using System.Collections.Specialized;
    using System.ComponentModel;
    using NintendoWare.ToolDevelopmentKit.ComponentModel;

    /// <summary>
    /// 通知機能付き辞書コレクションのデコレータです。
    /// </summary>
    /// <typeparam name="TKey">ディクショナリー内のキーのテンプレート型です。</typeparam>
    /// <typeparam name="TValue">ディクショナリー内の値のテンプレート型です。</typeparam>
    public abstract class DictionaryDecorator<TKey, TValue> :
        ObservableObject, IDictionary<TKey, TValue>, INotifyCollectionChanged
    {
        //-----------------------------------------------------------------
        // フィールド
        //-----------------------------------------------------------------
        private IDictionary<TKey, TValue> dictionary;

        //-----------------------------------------------------------------
        // コンストラクタ
        //-----------------------------------------------------------------

        /// <summary>
        /// デフォルトコンストラクタです。
        /// </summary>
        /// <param name="dictionary">デコレート対象のIDictionaryを持つオブジェクトです。</param>
        public DictionaryDecorator(IDictionary<TKey, TValue> dictionary)
        {
            this.dictionary = dictionary;
            INotifyCollectionChanged observableDictionary =
                this.dictionary as INotifyCollectionChanged;
            if (observableDictionary != null)
            {
                observableDictionary.CollectionChanged +=
                    delegate(object sender, NotifyCollectionChangedEventArgs e)
                    {
                        if (this.CollectionChanged != null)
                        {
                            this.CollectionChanged(this, e);
                        }
                    };
            }

            INotifyPropertyChanged notifyPropertyDictionary =
                this.dictionary as INotifyPropertyChanged;
            if (notifyPropertyDictionary != null)
            {
                notifyPropertyDictionary.PropertyChanged +=
                    delegate(object sender, PropertyChangedEventArgs e)
                    {
                        OnPropertyChanged(e);
                    };
            }
        }

        //-----------------------------------------------------------------
        // public イベント
        //-----------------------------------------------------------------

        /// <summary>
        /// コレクション変更イベントです。
        /// </summary>
        public event NotifyCollectionChangedEventHandler CollectionChanged;

        //-----------------------------------------------------------------
        // public プロパティ
        //-----------------------------------------------------------------

        /// <summary>
        /// 格納されている要素の数を取得します。
        /// </summary>
        /// <returns>格納されている要素の数です。</returns>
        public virtual int Count
        {
            get { return this.dictionary.Count; }
        }

        /// <summary>
        /// 読み取り専用かどうかを示す値を取得します。
        /// </summary>
        /// <returns>読み取り専用の場合は trueです。それ以外の場合は false です。</returns>
        public virtual bool IsReadOnly
        {
            get { return false; }
        }

        /// <summary>
        /// 設定されているすべてのキーを取得します。
        /// </summary>
        /// <returns>設定されているすべてのキーのコレクションです。</returns>
        public virtual ICollection<TKey> Keys
        {
            get { return this.dictionary.Keys; }
        }

        /// <summary>
        /// 設定されているすべての値を取得します。
        /// </summary>
        /// <returns>設定されているすべての値のコレクションです。</returns>
        public virtual ICollection<TValue> Values
        {
            get { return this.dictionary.Values; }
        }

        /// <summary>
        /// 指定したキーを持つ要素を取得または設定します。
        /// </summary>
        /// <param name="key">取得または設定する要素のキーです。</param>
        /// <returns>指定したキーを持つ要素です。</returns>
        public virtual TValue this[TKey key]
        {
            get { return this.dictionary[key]; }
            set { this.dictionary[key] = value; }
        }

        //-----------------------------------------------------------------
        // public メソッド
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したキーおよび値を持つ要素を追加します。
        /// </summary>
        /// <param name="key">追加する要素のキーです。</param>
        /// <param name="value">追加する要素の値です。</param>
        public virtual void Add(TKey key, TValue value)
        {
            this.dictionary.Add(key, value);
        }

        /// <summary>
        /// すべての要素を削除します。
        /// </summary>
        public virtual void Clear()
        {
            this.dictionary.Clear();
        }

        /// <summary>
        /// 特定のキーと値のペアが格納されているかどうかを判断します。
        /// </summary>
        /// <param name="keyValuePair">検索されるキーと値のペアです。</param>
        /// <returns>存在する場合は true です。それ以外の場合は false です。</returns>
        public virtual bool Contains(KeyValuePair<TKey, TValue> keyValuePair)
        {
            return this.dictionary.Contains(keyValuePair);
        }

        /// <summary>
        /// 指定したキーの要素が格納されているかどうかを確認します。
        /// </summary>
        /// <param name="key">検索されるキーです。</param>
        /// <returns>存在する場合は true です。それ以外の場合は false です。</returns>
        public virtual bool ContainsKey(TKey key)
        {
            return this.dictionary.ContainsKey(key);
        }

        /// <summary>
        /// 要素を配列にコピーします。配列の特定のインデックスからコピーが開始されます。
        /// </summary>
        /// <param name="array">要素がコピーされる 1 次元のキーと値のペアの配列です。</param>
        /// <param name="arrayIndex">
        /// コピーの開始位置となる array の 0 から始まるインデックス番号です。
        /// </param>
        public virtual void CopyTo(KeyValuePair<TKey, TValue>[] array, int arrayIndex)
        {
            this.dictionary.CopyTo(array, arrayIndex);
        }

        /// <summary>
        /// 指定したキーを持つ値を削除します。
        /// </summary>
        /// <param name="key">削除する要素のキーです。</param>
        /// <returns>
        /// 要素が見つかり正常に削除された場合は true です。それ以外の場合は false です。
        /// </returns>
        public virtual bool Remove(TKey key)
        {
            return this.dictionary.Remove(key);
        }

        /// <summary>
        /// 指定したキーに関連付けられている値を取得します。
        /// </summary>
        /// <param name="key">取得する値のキーです。</param>
        /// <param name="value">指定したキーに関連付けられている値が格納されます。</param>
        /// <returns>存在する場合は true です。それ以外の場合は false です。</returns>
        public virtual bool TryGetValue(TKey key, out TValue value)
        {
            return this.dictionary.TryGetValue(key, out value);
        }

        //-----------------------------------------------------------------
        // その他 インターフェイス実装
        //-----------------------------------------------------------------

        /// <summary>
        /// 指定したキーおよび値を持つ要素を追加します。
        /// </summary>
        /// <param name="keyValuePair">追加する要素のキーと値のペアです。</param>
        void ICollection<KeyValuePair<TKey, TValue>>.Add(KeyValuePair<TKey, TValue> keyValuePair)
        {
            this.dictionary.Add(keyValuePair);
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>コレクションを反復処理するために使用できる列挙子です。</returns>
        IEnumerator<KeyValuePair<TKey, TValue>>
            IEnumerable<KeyValuePair<TKey, TValue>>.GetEnumerator()
        {
            return (IEnumerator<KeyValuePair<TKey, TValue>>)this.dictionary.GetEnumerator();
        }

        /// <summary>
        /// コレクションを反復処理する列挙子を返します。
        /// </summary>
        /// <returns>コレクションを反復処理するために使用できる列挙子です。</returns>
        IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            return (IEnumerator)this.dictionary.GetEnumerator();
        }

        /// <summary>
        /// 指定したキーを持つ値を削除します。
        /// </summary>
        /// <param name="keyValuePair">削除する要素のキーと値のペアです。</param>
        /// <returns>
        /// 要素が見つかり正常に削除された場合は true です。それ以外の場合は false です。
        /// </returns>
        bool ICollection<KeyValuePair<TKey, TValue>>.Remove(
            KeyValuePair<TKey, TValue> keyValuePair)
        {
            return this.dictionary.Remove(keyValuePair);
        }
    }
}
