﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NWCore.Utility
{
    /// <summary>
    /// Class for computing CRC32 hash value.
    /// </summary>
    public class CRC32
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public CRC32()
        {
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize CRC32 lookup table.
        /// </summary>
        /// <returns>True on success.</returns>
        public bool Initialize()
        {
            uint dwPolynomial = 0xEDB88320;
            uint i, j;

            uint dwCrc;

            for ( i=0;i<256;++i )
            {
                dwCrc = i;
                for ( j=8;j>0;--j )
                {
                    if ( (dwCrc & 1)>0 )
                        dwCrc = (dwCrc >> 1) ^ dwPolynomial;
                    else
                        dwCrc >>= 1;
                }

                m_CRC32LUT[i] = dwCrc;
            }

            return true;
        }

        #endregion

        #region Compute CRC32 hash value

        /// <summary>Compute CRC32 value from streams of bytes</summary>
        /// <param name="buffer">Byte array to compute CRC32 value</param>
        /// <returns>CRC32 value of the passed in byte array</returns>
        public uint ComputeCRC32( byte[] buffer )
        {
            if ( buffer.Length<=0 || buffer==null )
                return 0;

            uint dwCRC32 = 0xFFFFFFFF;

            int i;
            uint iIndex = 0;
            for ( i=0;i<buffer.Length;++i )
            {
                iIndex = (((uint)(buffer[i]) ^ (dwCRC32))) & 0x000000FF;
                dwCRC32 = ((dwCRC32) >> 8) ^ m_CRC32LUT[iIndex];
            }

            dwCRC32 = ~dwCRC32;

            return dwCRC32;
        }


        /// <summary>
        /// Compute CRC32 value from streams of bytes beginning from the end of
        /// the buffer.
        /// </summary>
        /// <param name="buffer">Byte array to compute CRC32 value</param>
        /// <returns>CRC32 value of the passed in byte array</returns>
        public uint ComputeCRC32Reverse( byte[] buffer )
        {
            if ( buffer.Length<=0 || buffer==null )
                return 0;

            uint dwCRC32 = 0xFFFFFFFF;

            int i;
            uint iIndex = 0;
            for ( i=(buffer.Length-1);i>=0;--i )
            {
                iIndex = (((uint)(buffer[i]) ^ (dwCRC32))) & 0x000000FF;
                dwCRC32 = ((dwCRC32) >> 8) ^ m_CRC32LUT[iIndex];
            }

            dwCRC32 = ~dwCRC32;

            return dwCRC32;
        }


        /// <summary>
        /// Compute CRC32 value from string
        /// </summary>
        /// <param name="str">String to compute CRC32 value</param>
        /// <returns>CRC32 value of the passed in byte array</returns>
        public uint ComputeCRC32Str(string str)
        {
            if ( string.IsNullOrEmpty(str)==true )
                return 0;

            uint dwCRC32 = 0xFFFFFFFF;

            int i;
            uint iIndex = 0;
            uint iByteValue;
            for ( i=0;i<str.Length;++i )
            {
                // Least significant byte
                iByteValue = (((uint)str[i]) & 0x00FF);
                iIndex = (iByteValue ^ dwCRC32) & 0x000000FF;
                dwCRC32 = ((dwCRC32) >> 8) ^ m_CRC32LUT[iIndex];

                // Most significant byte
                iByteValue = (((uint)str[i]) >> 8);
                iIndex = (iByteValue ^ dwCRC32) & 0x000000FF;
                dwCRC32 = ((dwCRC32) >> 8) ^ m_CRC32LUT[iIndex];
            }

            dwCRC32 = ~dwCRC32;

            return dwCRC32;
        }


        /// <summary>
        /// Compute CRC32 value from string beginning from the end of
        /// the buffer.
        /// </summary>
        /// <param name="str">String to compute CRC32 value</param>
        /// <returns>CRC32 value of the passed in byte array</returns>
        public uint ComputeCRC32StrReverse( string str )
        {
            if ( string.IsNullOrEmpty(str)==true )
                return 0;

            uint dwCRC32 = 0xFFFFFFFF;

            int i;
            uint iIndex = 0;
            uint iByteValue;
            for ( i=str.Length;i>=0;--i )
            {
                // Most significant byte
                iByteValue = (((uint)str[i]) >> 8);
                iIndex = (iByteValue ^ dwCRC32) & 0x000000FF;
                dwCRC32 = ((dwCRC32) >> 8) ^ m_CRC32LUT[iIndex];

                // Least significant byte
                iByteValue = (((uint)str[i]) & 0x00FF);
                iIndex = (iByteValue ^ dwCRC32) & 0x000000FF;
                dwCRC32 = ((dwCRC32) >> 8) ^ m_CRC32LUT[iIndex];
            }

            dwCRC32 = ~dwCRC32;

            return dwCRC32;
        }

        #endregion

        #region Member variables

        private uint[] m_CRC32LUT = new uint[256];

        #endregion
    }
}
