﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Xml;
using System.Xml.Serialization;

namespace NWCore.Remake.Updater
{

    /// <summary>
    /// 現行版EffectMakerからRemake版EffectMakerの中間ファイルに変換を行うクラスです.
    /// </summary>
    public class PrevUpdater
    {
        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public PrevUpdater()
        {
            /* DO_NOTHING */
        }

        /// <summary>
        /// プレビューデータを新しい形式に変換します.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="path">出力ファイルパス.</param>
        /// <returns></returns>
        public bool Convert(NWCore.Serializer.GameConfigDataXml oldPrev, string path)
        {
            EffectMaker.DataModel.DataModels.PreviewData newPrev = new EffectMaker.DataModel.DataModels.PreviewData();

            this.ConvertPreview(oldPrev, newPrev);

            try
            {
                using (TextWriter writer = new StreamWriter(path, false, Encoding.UTF8))
                {
                    // シリアライザー生成.
                    var serializer = new XmlSerializer(typeof(EffectMaker.DataModel.DataModels.PreviewData));

                    // シリアライズ！
                    serializer.Serialize(writer, newPrev);
                }
            }
            catch
            {
                return false;
            }

            return true;
        }

        #region Preview

        /// <summary>
        /// プレビューデータを新しい形式に変換します.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="newPrev">新型式.</param>
        private void ConvertPreview(NWCore.Serializer.GameConfigDataXml oldPrev, EffectMaker.DataModel.DataModels.PreviewData newPrev)
        {
            // 基本設定.
            this.ConvertPreviewBasicSettingsTab(oldPrev, newPrev);

            // SRT制御.
            this.ConvertPreviewSRTControlTab(oldPrev, newPrev);

            // 自動移動.
            this.ConvertPreviewAutoMovementTab(oldPrev, newPrev);

            // カラー.
            this.ConvertPreviewColorTab(oldPrev, newPrev);

            // パーティクル放出.
            this.ConvertPreviewParticleEmissionTab(oldPrev, newPrev);

            // パーティクルスケール.
            this.ConvertPreviewParticleScaleTab(oldPrev, newPrev);

            // パーティクル制御.
            this.ConvertPreviewParticleControlTab(oldPrev, newPrev);
        }

        /// <summary>
        /// 【プレビュー】基本設定タブの変換処理です.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="newPrev">新型式.</param>
        private void ConvertPreviewBasicSettingsTab(NWCore.Serializer.GameConfigDataXml oldPrev, EffectMaker.DataModel.DataModels.PreviewData newPrev)
        {
            //********************************
            //  ファイル管理.
            //********************************
            // プレビュー名.
            newPrev.Name = oldPrev.Name;

            // エミッタセット名.
            newPrev.EsetName = oldPrev.EmitterSetName;

            // ファイルパス.
            newPrev.FilePath = oldPrev.EmitterSetPath;

            // コメント.
            newPrev.BasicSettingData.PreviewBasicFileData.Comment = oldPrev.Comment;

            //********************************
            //  再生管理.
            //********************************
            // 再生開始フレーム.
            newPrev.BasicSettingData.PreviewBasicPlaybackData.PreviewStartFrame = oldPrev.PlaybackStartFrame;

            // 強制開始フレーム.
            newPrev.BasicSettingData.PreviewBasicPlaybackData.PreviewEndFrame = oldPrev.ForceStartFrame;
        }

        /// <summary>
        /// 【プレビュー】SRT制御タブの変換処理です.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="newPrev">新型式.</param>
        private void ConvertPreviewSRTControlTab(NWCore.Serializer.GameConfigDataXml oldPrev, EffectMaker.DataModel.DataModels.PreviewData newPrev)
        {
            //********************************
            //  行列設定.
            //********************************
            // 行列設定タイプ.
            newPrev.PreviewMatrixData.PreviewMatrixMatrixSettingsData.MatrixType = (int)oldPrev.MartixSetType;

            //********************************
            //  モデル連携.
            //********************************
            // マトリクス適用モード.
            newPrev.PreviewMatrixData.PreviewMatrixModelRelationData.MatrixApplyMode = (int)oldPrev.MtxApplyMode;

            // 接続モデル.
            newPrev.PreviewMatrixData.PreviewMatrixModelRelationData.PreviewModelIndex = oldPrev.PreviewModelIndex;

            // 接続ボーン.
            newPrev.PreviewMatrixData.PreviewMatrixModelRelationData.BoneIndex = oldPrev.BoneIndex;

            //********************************
            //  オフセット行列.
            //********************************
            // スケール.
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetScale.X = oldPrev.OffsetScale.X;
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetScale.Y = oldPrev.OffsetScale.Y;
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetScale.Z = oldPrev.OffsetScale.Z;

            // エミッタビルボード.
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.EmitterBillboard = oldPrev.EmitterBillboard;

            // 回転X
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetRotationX = oldPrev.OffsetRotateX;

            // 回転Y
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetRotationY = oldPrev.OffsetRotateY;

            // 回転Z
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetRotationZ = oldPrev.OffsetRotateZ;

            // 座標.
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetPosition.X = oldPrev.OffsetPosition.X;
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetPosition.Y = oldPrev.OffsetPosition.Y;
            newPrev.PreviewMatrixData.PreviewMatrixOffsetMatrixData.OffsetPosition.Z = oldPrev.OffsetPosition.Z;
        }

        /// <summary>
        /// 【プレビュー】SRT制御タブの変換処理です.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="newPrev"></param>
        private void ConvertPreviewAutoMovementTab(NWCore.Serializer.GameConfigDataXml oldPrev, EffectMaker.DataModel.DataModels.PreviewData newPrev)
        {
            NWCore.Serializer.UserFunctionXML oldData = null;
            foreach (var udd in oldPrev.UserFunctionList)
            {
                if (udd.UserFuncName == "PreviewAutomaticTransform" && udd.UserFuncID == 6)
                {
                    oldData = udd;
                    break;
                }
            }

            if (oldData == null)
            { return; }


            foreach (var field in oldData.Fields)
            {
                //********************************
                //  自動移動.
                //********************************

                if (field.Name == "AutoMoveTypeValue")
                {
                    // 自動移動タイプ.
                    newPrev.AutoMoveData.MoveType = (field.Value as int?).GetValueOrDefault();
                }

                if (field.Name == "AutoMoveRadiusValue")
                {
                    // XZ移動半径.
                    newPrev.AutoMoveData.MoveRadiusXZ = (field.Value as float?).GetValueOrDefault();
                }

                if (field.Name == "AutoMoveRotVelValue")
                {
                    // XZ移動速度.
                    newPrev.AutoMoveData.MoveSpeedXZ = (field.Value as float?).GetValueOrDefault();
                }

                if (field.Name == "AutoYSwingValue")
                {
                    // Y揺れ振幅.
                    newPrev.AutoMoveData.ShakeAmplitudeY = (field.Value as float?).GetValueOrDefault();
                }

                if (field.Name == "AutoYSwingSpeedValue")
                {
                    // Y揺れ速度.
                    newPrev.AutoMoveData.ShakeSpeedY = (field.Value as float?).GetValueOrDefault();
                }

                if (field.Name == "AutoZRollSpeedValue")
                {
                    // Zロール速度.
                    newPrev.AutoMoveData.RollSpeedZ = (field.Value as float?).GetValueOrDefault();
                }
            }
        }

        /// <summary>
        /// 【プレビュー】カラータブの変換処理です.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="newPrev">新型式.</param>
        private void ConvertPreviewColorTab(NWCore.Serializer.GameConfigDataXml oldPrev, EffectMaker.DataModel.DataModels.PreviewData newPrev)
        {
            NWCore.Serializer.UserFunctionXML oldData = null;
            foreach (var udd in oldPrev.UserFunctionList)
            {
                if (udd.UserFuncName == "ColorFunction" && udd.UserFuncID == 2)
                {
                    oldData = udd;
                    break;
                }
            }

            if (oldData == null)
            { return; }

            foreach (var field in oldData.Fields)
            {
                //********************************
                //  カラー.
                //********************************
                if (field.Name == "ColorEditValue")
                {
                    float[] value = field.Value as float[];
                    // カラー(EmitterSet::SetAlpha/SetColor)
                    newPrev.ColorData.Color.X = value[0];
                    newPrev.ColorData.Color.Y = value[1];
                    newPrev.ColorData.Color.Z = value[2];
                    newPrev.ColorData.Color.W = value[3];
                }
            }
        }

        /// <summary>
        /// 【プレビュー】パーティクル放出タブの変換処理です.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="newPrev">新型式.</param>
        private void ConvertPreviewParticleEmissionTab(NWCore.Serializer.GameConfigDataXml oldPrev, EffectMaker.DataModel.DataModels.PreviewData newPrev)
        {
            NWCore.Serializer.UserFunctionXML oldData = null;
            foreach (var udd in oldPrev.UserFunctionList)
            {
                if (udd.UserFuncName == "ParticleEmissionFunction" && udd.UserFuncID == 8)
                {
                    oldData = udd;
                    break;
                }
            }

            if (oldData == null)
            { return; }

            foreach (var field in oldData.Fields)
            {
                //********************************
                //  パーティクル放出.
                //********************************
                if (field.Name == "EmissionRatio")
                {
                    // 放出レート(EmitterController::SetEmissionRatio)
                    newPrev.ParticleEmissionData.EmissionRate = (field.Value as float?).GetValueOrDefault();
                }

                if (field.Name == "EmissionInterval")
                {
                    // 放出間隔(EmitterController::SetEmissionInterval)
                    newPrev.ParticleEmissionData.EmissionInterval = (field.Value as float?).GetValueOrDefault();
                }
            }
        }

        /// <summary>
        /// 【プレビュー】パーティクルスケールタブの変換処理です.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="newPrev">新型式.</param>
        private void ConvertPreviewParticleScaleTab(NWCore.Serializer.GameConfigDataXml oldPrev, EffectMaker.DataModel.DataModels.PreviewData newPrev)
        {
            NWCore.Serializer.UserFunctionXML oldData = null;
            foreach (var udd in oldPrev.UserFunctionList)
            {
                if (udd.UserFuncName == "ParticleScaleFunction" && udd.UserFuncID == 5)
                {
                    oldData = udd;
                    break;
                }
            }

            if (oldData == null)
            { return; }

            foreach (var field in oldData.Fields)
            {
                //********************************
                //  パーティクルスケール.
                //********************************
                if (field.Name == "ParticleScale")
                {
                    float[] value = field.Value as float[];
                    // パーティクルの大きさ(EmitterSet::SetParticleScale)
                    newPrev.ParticleScaleData.ParticleScale.X = value[0];
                    newPrev.ParticleScaleData.ParticleScale.Y = value[1];
                }

                if (field.Name == "EmittionScale")
                {
                    float[] value = field.Value as float[];
                    // 放出時のみ作用するパーティクルの大きさ(EmitterSet::SetEmissionParticleScale)
                    newPrev.ParticleScaleData.EmissionParticleScale.X = value[0];
                    newPrev.ParticleScaleData.EmissionParticleScale.Y = value[1];
                }

                //********************************
                //  エミッタ形状スケール.
                //********************************
                if (field.Name == "Scale")
                {
                    float[] value = field.Value as float[];
                    // エミッタ形状スケール(EmitterSet::SetEmitterScale)
                    newPrev.ParticleScaleData.EmitterScale.X = value[0];
                    newPrev.ParticleScaleData.EmitterScale.Y = value[1];
                    newPrev.ParticleScaleData.EmitterScale.Z = value[2];
                }
            }
        }

        /// <summary>
        /// 【プレビュー】パーティクル制御タブ.
        /// </summary>
        /// <param name="oldPrev">旧型式.</param>
        /// <param name="newPrev">新型式.</param>
        private void ConvertPreviewParticleControlTab(NWCore.Serializer.GameConfigDataXml oldPrev, EffectMaker.DataModel.DataModels.PreviewData newPrev)
        {
            NWCore.Serializer.UserFunctionXML oldData = null;
            foreach (var udd in oldPrev.UserFunctionList)
            {
                if (udd.UserFuncName == "ParticleVelocityFunction" && udd.UserFuncID == 7)
                {
                    oldData = udd;
                    break;
                }
            }

            if (oldData == null)
            { return; }

            foreach (var field in oldData.Fields)
            {
                //********************************
                //  ライフ.
                //********************************
                if (field.Name == "SetLife")
                {
                    // ライフスケール(EmitterSet::SetLife)
                    newPrev.ParticleControlData.LifeScale = (field.Value as float?).GetValueOrDefault();
                }

                //********************************
                //  初速.
                //********************************
                if (field.Name == "AllDirectionalVelScale")
                {
                    // 全方向初速スケール(EmitterSet::SetAllDirectionalVel)
                    newPrev.ParticleEmissionData.SetAllDirectionalVelScale = (field.Value as float?).GetValueOrDefault();
                }

                if (field.Name == "DirectionalVelScale")
                {
                    // 指定方向初速スケール(EmitterSet::SetDirectionalVel)
                    newPrev.ParticleEmissionData.SetDirectionalVelScale = (field.Value as float?).GetValueOrDefault();
                }

                if (field.Name == "SetRandomVel")
                {
                    // 初速のランダムスケール(EmitterSet::SetRandomVel)
                    newPrev.ParticleEmissionData.SetRandomVelScale = (field.Value as float?).GetValueOrDefault();
                }

                if (field.Name == "SetAddVel")
                {
                    float[] value = field.Value as float[];

                    // 速度に加算するワールドベクトル(EmitterSet:SetAddVel)
                    newPrev.ParticleEmissionData.SetAddVelValue.X = value[0];
                    newPrev.ParticleEmissionData.SetAddVelValue.Y = value[1];
                    newPrev.ParticleEmissionData.SetAddVelValue.Z = value[2];
                }

                //********************************
                //  エミッタカラー.
                //********************************
                // 有効.

                // エミッタカラー0(EmitterCtrl::SetEmitterColor0)
                if (field.Name == "EmitterColor0")
                {
                    float[] color = field.Value as float[];
                    newPrev.ColorData.EmitterColor0.X = color[0];
                    newPrev.ColorData.EmitterColor0.Y = color[1];
                    newPrev.ColorData.EmitterColor0.Z = color[2];
                }

                // エミッタカラー1(EmitterCtrl::SetEmitterColor1)
                if (field.Name == "EmitterColor1")
                {
                    float[] color = field.Value as float[];
                    newPrev.ColorData.EmitterColor1.X = color[0];
                    newPrev.ColorData.EmitterColor1.Y = color[1];
                    newPrev.ColorData.EmitterColor1.Z = color[2];
                }
            }
        }

        #endregion
    }
}
