﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Resources;
using System.Text;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// Class containing utility methods related to resources.
    /// </summary>
    public static class ResourceUtility
    {
        /// <summary>
        /// Cache of ResourceManager classes, stored per assembly, for fast access purpose.
        /// </summary>
        private static Dictionary<string, ResourceManager> resourceManagers =
            new Dictionary<string, ResourceManager>();

        /// <summary>
        /// リソース情報からリソース読み込み
        /// </summary>
        /// <param name="resourceInfo">リソース情報</param>
        /// <returns>Returns the resource content as a stream.</returns>
        public static Stream Load(ResourceInfo resourceInfo)
        {
            return Load(Assembly.GetCallingAssembly(), resourceInfo);
        }

        /// <summary>
        /// リソース情報からリソース読み込み
        /// </summary>
        /// <param name="assembly">The assembly that contains the embedded resource.</param>
        /// <param name="resourceInfo">リソース情報</param>
        /// <returns>Returns the resource content as a stream.</returns>
        public static Stream Load(Assembly assembly, ResourceInfo resourceInfo)
        {
            switch (resourceInfo.Kind)
            {
                case ResourceInfo.ResourceKind.Embedded:
                    return LoadEmbedded(assembly, resourceInfo.ResourceName);

                case ResourceInfo.ResourceKind.External:
                    return LoadExternal(assembly, resourceInfo.ResourceName);
            }

            throw new NotImplementedException();
        }

        /// <summary>
        /// Load an embedded resource by its name.
        /// </summary>
        /// <param name="relativeName">The relative name of the resource.</param>
        /// <returns>Returns the resource content as a stream.</returns>
        public static Stream LoadEmbedded(string relativeName)
        {
            return LoadEmbedded(Assembly.GetCallingAssembly(), relativeName);
        }

        /// <summary>
        /// Load an embedded resource from within an assembly, by its name.
        /// </summary>
        /// <param name="assembly">The assembly that contains the embedded resource.</param>
        /// <param name="relativeName">The relative name of the resource.</param>
        /// <returns>Returns the resource content as a stream.</returns>
        public static Stream LoadEmbedded(Assembly assembly, string relativeName)
        {
            if (assembly == null)
            {
                throw new ArgumentNullException("assembly");
            }

            if (string.IsNullOrWhiteSpace(relativeName))
            {
                throw new ArgumentException("Invalid 'relativeName' argument.", "relativeName");
            }

            string fullResourceName = string.Format(
                "{0}.Resources.{1}",
                assembly.GetName().Name,
                relativeName);

            var stream = assembly.GetManifestResourceStream(fullResourceName);

            if (stream == null)
            {
                var format =
                    "Impossible to find resource named '{0}' in resources of assembly '{1}'.";
                var msg = string.Format(format, relativeName, assembly.GetName().Name);
                throw new FileNotFoundException(msg);
            }

            return stream;
        }

        /// <summary>
        /// 外部リソースを読み込む
        /// </summary>
        /// <param name="filepath">ファイルパス</param>
        /// <returns>Returns the resource content as a stream.</returns>
        public static Stream LoadExternal(string filepath)
        {
            return LoadExternal(Assembly.GetCallingAssembly(), filepath);
        }

        /// <summary>
        /// 外部リソースを読み込む
        /// </summary>
        /// <param name="assembly">The assembly that contains the embedded resource.</param>
        /// <param name="filepath">ファイルパス</param>
        /// <returns>Returns the resource content as a stream.</returns>
        public static Stream LoadExternal(Assembly assembly, string filepath)
        {
            var fullFilepath = Path.Combine(Path.GetDirectoryName(Assembly.GetEntryAssembly().Location), filepath);

            if (File.Exists(fullFilepath) == false)
            {
                ////return null;

                // 空
                return new MemoryStream();
            }
            else
            {
                return new FileStream(fullFilepath, FileMode.Open, FileAccess.Read);
            }
        }

        /// <summary>
        /// Get a string value from a resource identifier.
        /// </summary>
        /// <param name="assembly">The assembly containing the resources.</param>
        /// <param name="resourceId">The identifier of the resource.</param>
        /// <param name="value">The resource value.</param>
        /// <returns>Returns true if the resource could be found, false otherwise.</returns>
        public static bool GetStringValue(Assembly assembly, string resourceId, out string value)
        {
            value = null;

            if (assembly == null)
            {
                throw new ArgumentNullException("assembly");
            }

            string assemblyName = assembly.GetName().Name;

            ResourceManager resourceManager;

            if (resourceManagers.TryGetValue(assemblyName, out resourceManager) == false)
            {
                resourceManager = FindAssemblyResourceManager(assembly);

                if (resourceManager == null)
                {
                    return false;
                }

                resourceManagers.Add(assemblyName, resourceManager);
            }

            value = resourceManager.GetString(resourceId);

            return value != null;
        }

        /// <summary>
        /// Retrieves a ResourceManager instance from with an Assembly.
        /// </summary>
        /// <param name="assembly">The assembly that contains the ResourceManager.</param>
        /// <returns>Returns an instance of ResourceManager,
        /// or null if it failed to retrieve it.</returns>
        public static ResourceManager FindAssemblyResourceManager(Assembly assembly)
        {
            var resourcesType = assembly.GetTypes().FirstOrDefault(t => t.Name == "Resources");

            if (resourcesType == null)
            {
                return null;
            }

            var resourceManagerProperty = resourcesType.GetProperty(
                "ResourceManager",
                BindingFlags.NonPublic | BindingFlags.Static);

            if (resourceManagerProperty == null)
            {
                return null;
            }

            var method = resourceManagerProperty.GetGetMethod(true);
            if (method == null)
            {
                return null;
            }

            return method.Invoke(null, null) as ResourceManager;
        }
    }

    /// <summary>
    /// リソース情報
    /// </summary>
    public class ResourceInfo
    {
        /// <summary>
        /// リソースの種類
        /// </summary>
        public enum ResourceKind
        {
            /// <summary>
            /// 組み込みリソース
            /// </summary>
            Embedded,

            /// <summary>
            /// 外部リソース
            /// </summary>
            External
        }

        /// <summary>
        /// リソース名
        /// </summary>
        public string ResourceName { get; set; }

        /// <summary>
        /// リソースの種類
        /// </summary>
        public ResourceKind Kind { get; set; }
    }
}
