﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Text;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// Utility class with methods related to exception.
    /// </summary>
    public static class ExceptionUtility
    {
        /// <summary>
        /// Convert an exception to string format.
        /// </summary>
        /// <param name="ex">The exception to make a string of it.</param>
        /// <returns>Returns the string representation of the exception.</returns>
        public static string Dump(this Exception ex)
        {
            if (ex == null)
            {
                throw new ArgumentNullException("ex");
            }

            return DumpExceptionRecursive(ex, 0);
        }

        /// <summary>
        /// Convert an exception to string format.
        /// </summary>
        /// <param name="ex">The exception to make a string of it.</param>
        /// <param name="level">The level of text indentation.</param>
        /// <returns>Returns the string representation of the exception.</returns>
        private static string DumpExceptionRecursive(Exception ex, int level)
        {
            var sb = new StringBuilder();

            var indent = new string(' ', level * 4);

            sb.AppendLine(string.Format("{0}Type: {1}", indent, ex.GetType().FullName));
            sb.AppendLine(string.Format("{0}Message: {1}", indent, ex.Message));

            if (ex.Data.Keys.Count > 0)
            {
                sb.AppendLine(string.Format("{0}Data:", indent));
                foreach (object key in ex.Data.Keys)
                {
                    sb.AppendLine(string.Format(
                        "{0}  Key: {1}, Value: {2}",
                        indent,
                        key,
                        ex.Data[key]));
                }
            }

            if (ex is ReflectionTypeLoadException)
            {
                Exception[] exceptions = ((ReflectionTypeLoadException)ex).LoaderExceptions;
                foreach (Exception loaderException in exceptions)
                {
                    sb.AppendLine(DumpExceptionRecursive(loaderException, level + 1));
                }
            }
            else if (ex is AggregateException)
            {
                IEnumerable<Exception> exceptions = ((AggregateException)ex).InnerExceptions;
                foreach (var innerException in exceptions)
                {
                    sb.AppendLine(DumpExceptionRecursive(innerException, level + 1));
                }
            }

            if (ex.StackTrace != null)
            {
                sb.AppendLine(string.Format("{0}Stack:", indent));
                DumpStackTrace(level, ex.StackTrace, sb);
            }

            if (ex.InnerException != null)
            {
                sb.AppendLine();
                sb.AppendLine(DumpExceptionRecursive(ex.InnerException, level + 1));
            }

            return sb.ToString();
        }

        /// <summary>
        /// Write the stack trace to the StringBuilder.
        /// </summary>
        /// <param name="level">The level of text indentation.</param>
        /// <param name="stackTrace">The stack trace to write.</param>
        /// <param name="sb">The StringBuilder where to write the stack trace.</param>
        private static void DumpStackTrace(int level, string stackTrace, StringBuilder sb)
        {
            var indent = new string(' ', 2 + (level * 4));

            foreach (var line in stackTrace.Split('\n').Select(l => l.Trim()))
            {
                sb.AppendLine(string.Format("{0}{1}", indent, line));
            }
        }
    }
}
