﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// Identifies the type of information that will be written to
    /// the minidump file by the MiniDumpWriteDump function.
    /// </summary>
    [Flags]
    public enum DumpType
    {
        /// <summary>
        /// Include just the information necessary to capture stack traces for all existing
        /// threads in a process.
        /// </summary>
        Normal = 0x00000000,

        /// <summary>
        /// Include the data sections from all loaded modules.
        /// This results in the inclusion of global variables,
        /// which can make the minidump file significantly larger.
        /// For per-module control, use the ModuleWriteDataSeg enumeration value
        /// from MODULE_WRITE_FLAGS.
        /// </summary>
        WithDataSegs = 0x00000001,

        /// <summary>
        /// Include all accessible memory in the process.
        /// The raw memory data is included at the end,
        /// so that the initial structures can be mapped directly without the raw memory
        /// information. This option can result in a very large file.
        /// </summary>
        WithFullMemory = 0x00000002,

        /// <summary>
        /// Include high-level information about the operating system handles
        /// that are active when the minidump is made.
        /// </summary>
        WithHandleData = 0x00000004,

        /// <summary>
        /// Stack and backing store memory written to the minidump file should
        /// be filtered to remove all but the pointer values necessary
        /// to reconstruct a stack trace.
        /// </summary>
        FilterMemory = 0x00000008,

        /// <summary>
        /// Stack and backing store memory should be scanned for pointer references to
        /// modules in the module list.
        /// If a module is referenced by stack or backing store memory,
        /// the ModuleWriteFlags member of the MINIDUMP_CALLBACK_OUTPUT
        /// structure is set to ModuleReferencedByMemory.
        /// </summary>
        ScanMemory = 0x00000010,

        /// <summary>
        /// Include information from the list of modules that were recently unloaded,
        /// if this information is maintained by the operating system.
        /// Windows Server 2003 and Windows XP:
        ///   The operating system does not maintain information for unloaded modules until
        ///   Windows Server 2003 with SP1 and Windows XP with SP2.
        /// DbgHelp 5.1:
        ///   This value is not supported.
        /// </summary>
        WithUnloadedModules = 0x00000020,

        /// <summary>
        /// Include pages with data referenced by locals or other stack memory.
        /// This option can increase the size of the minidump file significantly.
        /// DbgHelp 5.1:
        ///   This value is not supported.
        /// </summary>
        WithIndirectlyReferencedMemory = 0x00000040,

        /// <summary>
        /// Filter module paths for information such as user names or important directories.
        /// This option may prevent the system from locating the image file and should
        /// be used only in special situations.
        /// DbgHelp 5.1:
        ///   This value is not supported.
        /// </summary>
        FilterModulePaths = 0x00000080,

        /// <summary>
        /// Include complete per-process and per-thread information from the operating system.
        /// DbgHelp 5.1:
        ///   This value is not supported.
        /// </summary>
        WithProcessThreadData = 0x00000100,

        /// <summary>
        /// Scan the virtual address space for PAGE_READWRITE memory to be included.
        /// DbgHelp 5.1:
        ///   This value is not supported.
        /// </summary>
        WithPrivateReadWriteMemory = 0x00000200,

        /// <summary>
        /// Reduce the data that is dumped by eliminating memory regions that are not essential
        /// to meet criteria specified for the dump.
        /// This can avoid dumping memory that may contain data that is private to the user.
        /// However, it is not a guarantee that no private information will be present.
        /// DbgHelp 6.1 and earlier:
        ///   This value is not supported.
        /// </summary>
        WithoutOptionalData = 0x00000400,

        /// <summary>
        /// Include memory region information. For more information,
        /// see MINIDUMP_MEMORY_INFO_LIST.
        /// DbgHelp 6.1 and earlier:
        ///   This value is not supported.
        /// </summary>
        WithFullMemoryInfo = 0x00000800,

        /// <summary>
        /// Include thread state information.
        /// For more information, see MINIDUMP_THREAD_INFO_LIST.
        /// DbgHelp 6.1 and earlier:
        ///   This value is not supported.
        /// </summary>
        WithThreadInfo = 0x00001000,

        /// <summary>
        /// Include all code and code-related sections from loaded
        /// modules to capture executable content.
        /// For per-module control, use the ModuleWriteCodeSegs enumeration
        /// value from MODULE_WRITE_FLAGS.
        /// DbgHelp 6.1 and earlier:
        ///   This value is not supported.
        /// </summary>
        WithCodeSegs = 0x00002000,

        /// <summary>
        /// Turns off secondary auxiliary-supported memory gathering.
        /// </summary>
        WithoutManagedState = 0x00004000,
    }

    /// <summary>
    /// Tell whether to grab exception info or not.
    /// </summary>
    public enum ExceptionInfo
    {
        /// <summary>
        /// Do not grab exception info.
        /// </summary>
        None,

        /// <summary>
        /// Grab exception info.
        /// </summary>
        Present
    }

    /// <summary>
    /// Contains the exception information written to the
    /// minidump file by the MiniDumpWriteDump function.
    /// </summary>
    [StructLayout(LayoutKind.Sequential, Pack = 4)]
    //// Pack=4 is important! So it works also for x64!
    public struct MiniDumpExceptionInformation
    {
        /// <summary>
        /// The identifier of the thread throwing the exception.
        /// </summary>
        public uint ThreadId;

        /// <summary>
        /// A pointer to an EXCEPTION_POINTERS structure specifying a
        /// computer-independent description of the exception and
        /// the processor context at the time of the exception.
        /// </summary>
        public IntPtr ExceptionPointers;

        /// <summary>
        /// Determines where to get the memory regions pointed to by the ExceptionPointers member.
        /// Set to TRUE if the memory resides in the process being debugged
        /// (the target process of the debugger).
        /// Otherwise, set to FALSE if the memory resides in the address space of the calling
        /// program (the debugger process).
        /// If you are accessing local memory (in the calling process) you should
        /// not set this member to TRUE.
        /// </summary>
        [MarshalAs(UnmanagedType.Bool)]
        public bool ClientPointers;
    }

    /// <summary>
    /// Utility class containing methods for core dump.
    /// </summary>
    public static class CoreDumpUtility
    {
        /// <summary>
        /// Write the core dump to a file, with full memory information.
        /// </summary>
        /// <param name="filename">The file to write the core dump to.</param>
        /// <returns>Returns true if success, false otherwise.</returns>
        public static bool Dump(string filename)
        {
            return Dump(filename, DumpType.WithFullMemory);
        }

        /// <summary>
        /// Write the core dump to a file.
        /// </summary>
        /// <param name="filename">The file to write the core dump to.</param>
        /// <param name="dumpType">The type of core dump to write.</param>
        /// <returns>Returns true if success, false otherwise.</returns>
        public static bool Dump(string filename, DumpType dumpType)
        {
            using (var fs = new FileStream(
                filename,
                FileMode.Create,
                FileAccess.Write,
                FileShare.Write))
            {
                return Write(fs.SafeFileHandle, dumpType, ExceptionInfo.Present);
            }
        }

        // Overload requiring MiniDumpExceptionInformation
        [DllImport(
            "dbghelp.dll",
            EntryPoint = "MiniDumpWriteDump",
            CallingConvention = CallingConvention.StdCall,
            CharSet = CharSet.Unicode,
            ExactSpelling = true,
            SetLastError = true)]
        private static extern bool MiniDumpWriteDump(
            IntPtr processHandle,
            uint processId,
            SafeHandle fileHandle,
            uint dumpType,
            ref MiniDumpExceptionInformation expParam,
            IntPtr userStreamParam,
            IntPtr callbackParam);

        // Overload supporting MiniDumpExceptionInformation == NULL
        [DllImport(
            "dbghelp.dll",
            EntryPoint = "MiniDumpWriteDump",
            CallingConvention = CallingConvention.StdCall,
            CharSet = CharSet.Unicode,
            ExactSpelling = true,
            SetLastError = true)]
        private static extern bool MiniDumpWriteDump(
            IntPtr processHandle,
            uint processId,
            SafeHandle fileHandle,
            uint dumpType,
            IntPtr expParam,
            IntPtr userStreamParam,
            IntPtr callbackParam);

        /// <summary>
        /// Gets te current unmanaged thread identifier.
        /// </summary>
        /// <returns>Returns the unmanaged thread identifier.</returns>
        [DllImport("kernel32.dll", EntryPoint = "GetCurrentThreadId", ExactSpelling = true)]
        private static extern uint GetCurrentThreadId();

        /// <summary>
        /// Writes user-mode minidump information to the specified file.
        /// </summary>
        /// <param name="fileHandle">A handle to the file in which
        /// the information is to be written.</param>
        /// <param name="dumpType">The type of information to be generated.
        /// This parameter can be one or more of the
        /// values from the MINIDUMP_TYPE enumeration.</param>
        /// <param name="exceptionInfo">Tell whether to grab exception information or not.</param>
        /// <returns>If the function succeeds, the return value is TRUE;
        /// otherwise, the return value is FALSE.
        /// To retrieve extended error information, call GetLastError.
        /// Note that the last error will be an HRESULT value.
        /// If the operation is canceled,
        /// the last error code is HRESULT_FROM_WIN32(ERROR_CANCELLED).</returns>
        private static bool Write(
            SafeHandle fileHandle,
            DumpType dumpType,
            ExceptionInfo exceptionInfo)
        {
            var currentProcess = Process.GetCurrentProcess();

            var currentProcessHandle = currentProcess.Handle;
            var currentProcessId = (uint)currentProcess.Id;

            var exp = new MiniDumpExceptionInformation
            {
                ThreadId = GetCurrentThreadId(),
                ClientPointers = false,
                ExceptionPointers = IntPtr.Zero,
            };

            if (exceptionInfo == ExceptionInfo.Present)
            {
                exp.ExceptionPointers = Marshal.GetExceptionPointers();
            }

            if (exp.ExceptionPointers == IntPtr.Zero)
            {
                return MiniDumpWriteDump(
                    currentProcessHandle,
                    currentProcessId,
                    fileHandle,
                    (uint)dumpType,
                    IntPtr.Zero,
                    IntPtr.Zero,
                    IntPtr.Zero);
            }

            return MiniDumpWriteDump(
                currentProcessHandle,
                currentProcessId,
                fileHandle,
                (uint)dumpType,
                ref exp,
                IntPtr.Zero,
                IntPtr.Zero);
        }
    }
}
