﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// Convert data to byte array.
    /// </summary>
    public static class BinaryConversionUtility
    {
        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public static byte[] Convert(object value)
        {
            bool flipEndian = true;

            byte[] result = null;
            if (value is bool)
            {
                flipEndian = false;
                result = BitConverter.GetBytes((bool)value);
            }
            else if (value is int)
            {
                result = BitConverter.GetBytes((int)value);
            }
            else if (value is uint)
            {
                result = BitConverter.GetBytes((uint)value);
            }
            else if (value is float)
            {
                result = BitConverter.GetBytes((float)value);
            }
            else if (value is double)
            {
                result = BitConverter.GetBytes((double)value);
            }
            else if (value is long)
            {
                result = BitConverter.GetBytes((long)value);
            }
            else if (value is ulong)
            {
                result = BitConverter.GetBytes((ulong)value);
            }
            else if (value is short)
            {
                result = BitConverter.GetBytes((short)value);
            }
            else if (value is ushort)
            {
                result = BitConverter.GetBytes((ushort)value);
            }
            else if (value is byte)
            {
                flipEndian = false;
                result = new byte[] { (byte)value };
            }
            else if (value is Guid)
            {
                flipEndian = false;
                result = ((Guid)value).ToByteArray();
            }
            else if (value is char)
            {
                flipEndian = false;
                result = new byte[] { (byte)(char)value };
            }
            else if (value is string)
            {
                // Not supported for now.
                flipEndian = false;
                return null;
            }
            else if (value is IEnumerable)
            {
                flipEndian = false;
                List<byte> tmpList = new List<byte>();
                foreach (object element in (IEnumerable)value)
                {
                    byte[] elementBytes = Convert(element);
                    if (elementBytes == null)
                    {
                        // Failed, just return null.
                        return null;
                    }

                    tmpList.AddRange(elementBytes);
                }

                result = tmpList.ToArray();
            }
            else
            {
                return null;
            }

            // Flip byte order.
            if (flipEndian == true)
            {
                FlipEndian(ref result);
            }

            return result;
        }

        /// <summary>
        /// Utility method for flipping byte endian.
        /// </summary>
        /// <param name="data">The data to flip.</param>
        public static void FlipEndian(ref byte[] data)
        {
            int length = data.Length;
            int helfLength = length / 2;
            for (int i = 0; i < helfLength; ++i)
            {
                byte tmp = data[i];
                data[i] = data[length - i - 1];
                data[length - i - 1] = tmp;
            }
        }
    }
}
