﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using EffectMaker.Foundation.Extensions;

namespace EffectMaker.Foundation.Core
{
    /// <summary>
    /// Comparer that sort by type, but do not modify order of items of the same type.
    /// </summary>
    /// <typeparam name="T">Type of items to compare.</typeparam>
    public class TypeOrderingComparer<T> : IComparer<T>
    {
        /// <summary>
        /// Stores the types order.
        /// </summary>
        private Type[] typeOrder;

        /// <summary>
        /// Initializes the TypeOrderingComparer instance.
        /// </summary>
        /// <param name="typeOrder">Types that define the type ordering.</param>
        public TypeOrderingComparer(Type[] typeOrder)
        {
            this.typeOrder = typeOrder;
        }

        /// <summary>
        /// Compares instances by type.
        /// </summary>
        /// <param name="x">First instance to compare.</param>
        /// <param name="y">Second instance to compare.</param>
        /// <returns>Returns -1 or +1 according to difference of input types.
        /// Returns 0 if instances are of the same type or common base type
        /// defined in the type order array.</returns>
        public int Compare(T x, T y)
        {
            if (x == null || y == null || this.typeOrder == null)
            {
                return 0;
            }

            Type typeA = x.GetType();
            Type typeB = y.GetType();

            int indexA = this.typeOrder.FindIndex(t => t.IsAssignableFrom(typeA));
            int indexB = this.typeOrder.FindIndex(t => t.IsAssignableFrom(typeB));

            return indexA.CompareTo(indexB);
        }
    }
}
