﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections.Concurrent;
using System.Collections.Generic;

namespace EffectMaker.Foundation.Collections.Generic
{
    /// <summary>
    /// LRUキャッシュ
    /// </summary>
    /// <typeparam name="TKey">キーの型</typeparam>
    /// <typeparam name="TValue">値の型</typeparam>
    public class LruCache<TKey, TValue>
    {
        /// <summary>
        /// キー値保存用
        /// </summary>
        private readonly LinkedList<KeyValuePair> keyValueStorage;

        /// <summary>
        /// キー値検索用
        /// </summary>
        private readonly Dictionary<TKey, LinkedListNode<KeyValuePair>> keyValueLookup;

        /// <summary>
        /// 同期オブジェクト
        /// </summary>
        private readonly object syncObject;

        /// <summary>
        /// 最大キャパシティ
        /// </summary>
        private readonly int maxCapacity;

        /// <summary>
        /// 現在のサイズ
        /// </summary>
        private int currentSize;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="maxCapacity">最大キャパシティ</param>
        /// <param name="comparer">キーの比較時に使用する IEqualityComparer 実装</param>
        public LruCache(int maxCapacity, IEqualityComparer<TKey> comparer = null)
        {
            this.keyValueStorage = new LinkedList<KeyValuePair>();
            this.keyValueLookup = (comparer != null) ? new Dictionary<TKey, LinkedListNode<KeyValuePair>>(comparer) : new Dictionary<TKey, LinkedListNode<KeyValuePair>>();
            this.syncObject = new object();

            this.maxCapacity = maxCapacity;
            this.currentSize = 0;
        }

        /// <summary>
        /// クリア
        /// </summary>
        public void Clear()
        {
            lock (this.syncObject)
            {
                foreach (var valueNode in this.keyValueStorage)
                {
                    this.OnDisposingValue(valueNode.Key, valueNode.Value);
                }

                this.keyValueStorage.Clear();
                this.keyValueLookup.Clear();
            }
        }

        /// <summary>
        /// 値取得
        /// </summary>
        /// <param name="key">値のキー</param>
        /// <returns>値</returns>
        public TValue Get(TKey key)
        {
            lock (this.syncObject)
            {
                LinkedListNode<KeyValuePair> listNode;

                if (this.keyValueLookup.TryGetValue(key, out listNode))
                {
                    this.keyValueStorage.Remove(listNode);
                    this.keyValueStorage.AddFirst(listNode);

                    return listNode.Value.Value;
                }

                return default(TValue);
            }
        }

        /// <summary>
        /// 値追加
        /// </summary>
        /// <param name="key">値のキー</param>
        /// <param name="value">値</param>
        public void Add(TKey key, TValue value)
        {
            lock (this.syncObject)
            {
                LinkedListNode<KeyValuePair> listNode;

                if (this.keyValueLookup.TryGetValue(key, out listNode))
                {
                    this.currentSize -= this.GetValueSize(listNode.Value.Value);

                    this.keyValueStorage.Remove(listNode);
                    this.keyValueStorage.AddFirst(listNode);

                    listNode.Value.Value = value;

                    this.currentSize += this.GetValueSize(listNode.Value.Value);
                }
                else
                {
                    var keyValue = new KeyValuePair { Key = key, Value = value };

                    listNode = this.keyValueStorage.AddFirst(keyValue);

                    this.keyValueLookup.Add(key, listNode);

                    this.currentSize += this.GetValueSize(listNode.Value.Value);
                }

                while (this.currentSize > this.maxCapacity)
                {
                    var valueNode = this.keyValueStorage.Last;

                    this.keyValueStorage.RemoveLast();
                    this.keyValueLookup.Remove(valueNode.Value.Key);

                    this.currentSize -= this.GetValueSize(valueNode.Value.Value);

                    this.OnDisposingValue(valueNode.Value.Key, valueNode.Value.Value);
                }
            }
        }

        /// <summary>
        /// 値削除
        /// </summary>
        /// <param name="key">値のキー</param>
        public void Remove(TKey key)
        {
            lock (this.syncObject)
            {
                LinkedListNode<KeyValuePair> listNode;

                if (!this.keyValueLookup.TryGetValue(key, out listNode))
                {
                    return;
                }

                this.OnDisposingValue(listNode.Value.Key, listNode.Value.Value);

                this.currentSize -= this.GetValueSize(listNode.Value.Value);

                this.keyValueStorage.Remove(listNode);
                this.keyValueLookup.Remove(key);
            }
        }

        /// <summary>
        /// 値サイズを得る
        /// </summary>
        /// <param name="value">値</param>
        /// <returns>値サイズ</returns>
        protected virtual int GetValueSize(TValue value)
        {
            return 1;
        }

        /// <summary>
        /// 値開放時処理
        /// </summary>
        /// <param name="key">値のキー</param>
        /// <param name="value">値</param>
        protected virtual void OnDisposingValue(TKey key, TValue value)
        {
        }

        /// <summary>
        /// キー値ペア
        /// </summary>
        private class KeyValuePair
        {
            /// <summary>
            /// キー
            /// </summary>
            public TKey Key { get; set; }

            /// <summary>
            /// 値
            /// </summary>
            public TValue Value { get; set; }
        }
    }
}
