﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.DataModel.RandomColor
{
    /// <summary>
    /// Class to store random colors.
    /// </summary>
    public class RandomColorTable : IXmlSerializable,
                                    ISettable,
                                    ICloneable,
                                    IEquatable<object>,
                                    IEnumerable<RandomColorData>
    {
        /// <summary>Constant value for the random color count.</summary>
        public const int RandomColorCount = 8;

        /// <summary>
        /// GetHashCode()で使用するハッシュテーブル.
        /// </summary>
        private static readonly int[] HashTable = { 0x77073096, 0x6e0e612c, 0x690951ba, 0x076dc419 };

        /// <summary>The random colors.</summary>
        private readonly RandomColorData[] randomColors = new RandomColorData[RandomColorCount];

        /// <summary>The flag indicating whether to ignore NotifyRandomColorSelected calls.</summary>
        private bool ignoreNotifyRandomColorSelected = false;

        /// <summary>The flag indicating whether multi-selection is enabled.</summary>
        private bool isMultiSelect = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public RandomColorTable()
        {
            for (int i = 0; i < RandomColorCount; ++i)
            {
                this.randomColors[i] = new RandomColorData()
                {
                    Owner = this
                };
            }
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source random color table to copy from.</param>
        public RandomColorTable(RandomColorTable src) :
            this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Get the count of the random colors.
        /// </summary>
        public int Count
        {
            get { return RandomColorCount; }
        }

        /// <summary>
        /// Get or set the flag indicating whether multi-selection is enabled.
        /// </summary>
        public bool MultiSelect
        {
            get
            {
                return this.isMultiSelect;
            }

            set
            {
                if (this.isMultiSelect != value)
                {
                    this.isMultiSelect = value;
                    if (value == false)
                    {
                        // Single selection is set, deselect all random colors except
                        // for the first selection.
                        this.ignoreNotifyRandomColorSelected = true;

                        bool firstSelectionFound = false;
                        foreach (RandomColorData data in this.randomColors)
                        {
                            if (data.IsSelected == true)
                            {
                                if (firstSelectionFound == false)
                                {
                                    firstSelectionFound = true;
                                }
                                else
                                {
                                    data.IsSelected = false;
                                }
                            }
                        }

                        this.ignoreNotifyRandomColorSelected = false;
                    }
                }
            }
        }

        /// <summary>
        /// Get the first selected random color in the table.
        /// </summary>
        public RandomColorData SelectedRandomColor
        {
            get
            {
                return this.randomColors.FirstOrDefault(data => data.IsSelected);
            }
        }

        /// <summary>
        /// Get the first selected random color index in the table.
        /// </summary>
        public int SelectedRandomColorIndex
        {
            get
            {
                for (var i = 0; i != RandomColorCount; ++i)
                {
                    if (this.randomColors[i].IsSelected)
                    {
                        return i;
                    }
                }

                return -1;
            }
        }

        /// <summary>
        /// Enumerate all the selected random colors.
        /// </summary>
        public IEnumerable<RandomColorData> SelectedRandomColors
        {
            get
            {
                return this.randomColors.Where(data => data.IsSelected);
            }
        }

        /// <summary>
        /// Get or set the random color data at the specified index.
        /// </summary>
        /// <param name="index">Index to the random color data.</param>
        /// <returns>The random color data.</returns>
        public RandomColorData this[int index]
        {
            get
            {
                if (index < 0 || index >= RandomColorCount)
                {
                    throw new IndexOutOfRangeException();
                }

                return this.randomColors[index];
            }

            set
            {
                if (index < 0 || index >= RandomColorCount)
                {
                    throw new IndexOutOfRangeException();
                }

                this.randomColors[index].Set(value);
            }
        }

        /// <summary>
        /// Get enumerator for the random colors.
        /// </summary>
        /// <returns>The random color data enumerator.</returns>
        public IEnumerator<RandomColorData> GetEnumerator()
        {
            foreach (RandomColorData data in this.randomColors)
            {
                yield return data;
            }
        }

        /// <summary>
        /// Get non-generic enumerator.
        /// </summary>
        /// <returns>The enumerator.</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return this.GetEnumerator();
        }

        /// <summary>
        /// Set a batch of random colors.
        /// </summary>
        /// <param name="items">The random colors to set.</param>
        public void SetRandomColors(IEnumerable<RandomColorData> items)
        {
            int index = 0;
            foreach (RandomColorData data in items)
            {
                this.randomColors[index].Set(data);

                ++index;
                if (index >= RandomColorCount)
                {
                    break;
                }
            }
        }

        /// <summary>
        /// Compare with the given object and check if they are equal.
        /// </summary>
        /// <param name="other">The object to compare with.</param>
        /// <returns>True if the instances are equal.</returns>
        public override bool Equals(object other)
        {
            var otherTable = other as RandomColorTable;
            if (otherTable == null)
            {
                return false;
            }

            for (int i = 0; i < RandomColorCount; ++i)
            {
                if (this[i].Equals(otherTable[i]) == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Get hash code of this random color table.
        /// </summary>
        /// <returns>The hash code.</returns>
        public override int GetHashCode()
        {
            int num = 42;
            unchecked
            {
                for (int i = 0; i < this.Count; ++i)
                {
                    int hash = this.randomColors[i].GetHashCode();
                    num = (num * 37) + hash;
                    num ^= hash >> 16;
                    num ^= HashTable[(hash >> 30) & 3];
                }
            }

            return num;
        }

        /// <summary>
        /// Set data.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public bool Set(object src)
        {
            var srcTable = src as RandomColorTable;
            if (srcTable == null)
            {
                return false;
            }

            this.isMultiSelect = srcTable.isMultiSelect;

            for (int i = 0; i < RandomColorCount; ++i)
            {
                this.randomColors[i].Set(srcTable.randomColors[i]);
            }

            return true;
        }

        /// <summary>
        /// Clone this random color table.
        /// </summary>
        /// <returns>The create random color table.</returns>
        public object Clone()
        {
            return new RandomColorTable(this);
        }

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml(XmlReader reader)
        {
            IXmlLineInfo lineInfo = (IXmlLineInfo)reader;

            // Read multi-selection flag.
            bool tmpMultiSelect;
            string strMultiSelect = reader.GetAttribute("MultiSelect");
            if (string.IsNullOrEmpty(strMultiSelect) == true ||
                bool.TryParse(strMultiSelect, out tmpMultiSelect) == false)
            {
                tmpMultiSelect = false;
            }

            this.isMultiSelect = tmpMultiSelect;

            // early break if there is no child to process.
            if (reader.IsEmptyElement)
            {
                return;
            }

            this.ignoreNotifyRandomColorSelected = true;

            // Read random colors.
            int index = 0;
            bool firstSelectionFound = false;
            string myLocalName = reader.LocalName;
            while (reader.Read())
            {
                if (reader.NodeType == XmlNodeType.Element &&
                    reader.Name == "RandomColor" &&
                    index < RandomColorCount)
                {
                    // Read the random color data.
                    this.randomColors[index].ReadXml(reader);

                    // If single selection is set, deselect all key frames
                    // except the first selection.
                    if (tmpMultiSelect == false &&
                        this.randomColors[index].IsSelected == true)
                    {
                        if (firstSelectionFound == false)
                        {
                            firstSelectionFound = true;
                        }
                        else
                        {
                            this.randomColors[index].IsSelected = false;
                        }
                    }

                    ++index;
                }
                else if (reader.NodeType == XmlNodeType.EndElement &&
                    reader.LocalName == myLocalName)
                {
                    // This is the end tag of the random color table element.
                    // Read once more, so the next element can be read correctly.
                    reader.Read();
                    break;
                }
            }

            // If there are less random colors in the Xml, reset the rest.
            for (int i = index; i < RandomColorCount; ++i)
            {
                this.randomColors[i].Reset();
            }

            this.ignoreNotifyRandomColorSelected = false;
        }

        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml(XmlWriter writer)
        {
            writer.WriteAttributeString("MultiSelect", this.isMultiSelect.ToString());

            foreach (RandomColorData data in this.randomColors)
            {
                writer.WriteStartElement("RandomColor");
                data.WriteXml(writer);
                writer.WriteEndElement();
            }
        }

        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        /// <summary>
        /// Notify random color data being selected.
        /// </summary>
        /// <param name="selectedItem">The selected random color data.</param>
        internal void NotifyRandomColorSelected(RandomColorData selectedItem)
        {
            if (this.ignoreNotifyRandomColorSelected == true)
            {
                return;
            }

            this.ignoreNotifyRandomColorSelected = true;

            if (this.MultiSelect == false)
            {
                foreach (RandomColorData data in this.randomColors)
                {
                    if (object.ReferenceEquals(data, selectedItem) == false)
                    {
                        data.IsSelected = false;
                    }
                }
            }

            this.ignoreNotifyRandomColorSelected = false;
        }
    }
}
