﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.DataModel.RandomColor
{
    /// <summary>
    /// Class to store a random color data.
    /// </summary>
    public class RandomColorData : IXmlSerializable,
                                   ICloneable,
                                   ISettable,
                                   IEquatable<object>
    {
        /// <summary>The color for the random color data.</summary>
        private readonly ColorRgba color = new ColorRgba(1.0f, 1.0f, 1.0f, 1.0f);

        /// <summary>The flag indicating whether this random color is enabled.</summary>
        private bool enabled = false;

        /// <summary>The flag indicating whether this key frame is selected.</summary>
        private bool isSelected = false;

        /// <summary>The tuple object to help up with generating hash code.</summary>
        private Tuple<bool, float, float, float, float> hashHelper = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public RandomColorData()
        {
            this.UpdateHashHelper();
        }

        /// <summary>
        /// Value setting constructor.
        /// </summary>
        /// <param name="enabled">The flag indicating whether the random color is enabled.</param>
        /// <param name="color">The color for the random color data.</param>
        public RandomColorData(bool enabled, ColorRgba color)
        {
            this.enabled = enabled;
            this.color.Set(color);

            this.UpdateHashHelper();
        }

        /// <summary>
        /// Value setting constructor.
        /// </summary>
        /// <param name="enabled">The flag indicating whether the random color is enabled.</param>
        /// <param name="r">The R color value of the random color data.</param>
        /// <param name="g">The G color value of the random color data.</param>
        /// <param name="b">The B color value of the random color data.</param>
        /// <param name="a">The A color value of the random color data.</param>
        public RandomColorData(bool enabled, float r, float g, float b, float a)
        {
            this.enabled = enabled;
            this.color.Set(r, g, b, a);

            this.UpdateHashHelper();
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">The source random color data to copy from.</param>
        public RandomColorData(RandomColorData src)
        {
            this.Set(src);
        }

        /// <summary>
        /// Get or set the owner of the random color data.
        /// </summary>
        public RandomColorTable Owner { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the random color data is selected or not.
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    if (value == true && this.Owner != null)
                    {
                        // Tell the owner that the random color data is selected
                        // so that the owner can handle single selection correctly.
                        this.Owner.NotifyRandomColorSelected(this);
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether this random color is enabled.
        /// </summary>
        public bool Enabled
        {
            get
            {
                return this.enabled;
            }

            set
            {
                this.enabled = value;
                this.UpdateHashHelper();
            }
        }

        /// <summary>
        /// Get or set the color for the random color data.
        /// </summary>
        public ColorRgba Color
        {
            get
            {
                return this.color.Clone() as ColorRgba;
            }

            set
            {
                this.color.Set(value);
                this.UpdateHashHelper();
            }
        }

        /// <summary>
        /// Compare with the given object and check if they are equal.
        /// </summary>
        /// <param name="other">The object to compare with.</param>
        /// <returns>True if the instances are equal.</returns>
        public override bool Equals(object other)
        {
            var otherData = other as RandomColorData;
            if (otherData == null)
            {
                return false;
            }

            return (this.enabled == otherData.enabled) && (this.color.Equals(otherData.color) == true);
        }

        /// <summary>
        /// Get hash code of this key frame data.
        /// </summary>
        /// <returns>The hash code according to the frame number and value.</returns>
        public override int GetHashCode()
        {
            return this.hashHelper.GetHashCode();
        }

        /// <summary>
        /// Reset the random color data to disabled and white color.
        /// </summary>
        public void Reset()
        {
            this.IsSelected = false;
            this.enabled = false;
            this.color.Set(1.0f, 1.0f, 1.0f, 1.0f);

            this.UpdateHashHelper();
        }

        /// <summary>
        /// Set data.
        /// </summary>
        /// <param name="src">The source object.</param>
        /// <returns>True on success.</returns>
        public bool Set(object src)
        {
            var srcData = src as RandomColorData;
            if (srcData == null)
            {
                return false;
            }

            this.IsSelected = srcData.IsSelected;
            this.enabled = srcData.enabled;
            this.color.Set(srcData.color);

            this.UpdateHashHelper();

            return true;
        }

        /// <summary>
        /// Clone this key frame data.
        /// </summary>
        /// <returns>The create key frame data.</returns>
        public object Clone()
        {
            return new RandomColorData(this);
        }

        /// <summary>
        /// Generates an object from its XML representation.
        /// </summary>
        /// <param name="reader">The XmlReader stream from which the object is deserialized.</param>
        public void ReadXml(XmlReader reader)
        {
            IXmlLineInfo lineInfo = (IXmlLineInfo)reader;

            bool tmpSelected;
            string strSelected = reader.GetAttribute("IsSelected");
            if (string.IsNullOrEmpty(strSelected) == false &&
                bool.TryParse(strSelected, out tmpSelected) == true)
            {
                this.isSelected = tmpSelected;
            }
            else
            {
                this.isSelected = false;
            }

            bool tmpEnabled;
            string strEnabled = reader.GetAttribute("Enabled");
            if (string.IsNullOrEmpty(strEnabled) == true ||
                bool.TryParse(strEnabled, out tmpEnabled) == false)
            {
                throw new XmlException(
                                        "The XML node must contain an boolean 'Enabled' attribute.",
                                        null,
                                        lineInfo.LineNumber,
                                        lineInfo.LinePosition);
            }

            this.enabled = tmpEnabled;

            string strColor = reader.GetAttribute("Color");
            if (string.IsNullOrEmpty(strColor) == true)
            {
                throw new XmlException(
                                        "The XML node must contain a 'Color' attribute with 4 comma-separated floating point values.",
                                        null,
                                        lineInfo.LineNumber,
                                        lineInfo.LinePosition);
            }

            // Split the comma-separated string.
            string[] tokens = strColor.Split(',');

            for (int i = 0; (i < this.color.Count) && (i < tokens.Length); ++i)
            {
                // Parse the elements and store them.
                float value;
                if (float.TryParse(tokens[i].Trim(), out value) == false)
                {
                    throw new XmlException(
                                            "The XML node must contain a 'Color' attribute with 4 comma-separated floating point values.",
                                            null,
                                            lineInfo.LineNumber,
                                            lineInfo.LinePosition);
                }

                this.color[i] = value;
            }

            this.UpdateHashHelper();
        }

        /// <summary>
        /// Converts an object into its XML representation.
        /// </summary>
        /// <param name="writer">The XmlWriter stream to which the object is serialized.</param>
        public void WriteXml(XmlWriter writer)
        {
            writer.WriteAttributeString("IsSelected", this.isSelected.ToString());
            writer.WriteAttributeString("Enabled", this.enabled.ToString());
            writer.WriteAttributeString("Color", this.color.ToString());
        }

        /// <summary>
        /// This method is reserved and should not be used.
        /// When implementing the IXmlSerializable interface,
        /// you should return null from this method, and
        /// instead, if specifying a custom schema is required,
        /// apply the XmlSchemaProviderAttribute to the class.
        /// </summary>
        /// <returns>
        /// An XmlSchema that describes the XML representation
        /// of the object that is produced by the WriteXml
        /// method and consumed by the ReadXml method.
        /// </returns>
        public XmlSchema GetSchema()
        {
            return null;
        }

        /// <summary>
        /// Update values to the hash helper.
        /// </summary>
        private void UpdateHashHelper()
        {
            this.hashHelper = Tuple.Create(
                this.enabled,
                this.color.R,
                this.color.G,
                this.color.B,
                this.color.A);
        }
    }
}
