﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0
{
    /// <summary>
    /// シンプルエミッタデータ
    /// </summary>
    public class ComplexEmitterData : BaseDataModel<ComplexEmitterData>
    {
        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public ComplexEmitterData()
        {
            this.EmitterVelDir = new Vector3();     // エミッタ速度の方向
            this.GravityDir = new Vector3();        // 重力の向き
            this.SpreadVec = new Vector3();         // 拡散ベクトル
            this.VolumeRadius = new Vector3();      // ボリューム半径
            this.UVScroll = new Vector2();          // ＵＶスクロール

            this.TexRes0 = new TextureRes();
            this.TexRes1 = new TextureRes();
            this.TexRes2 = new TextureRes();

            this.FluctuationFlag = new FluctuationFlag();

            this.FieldFlag = new FieldMask();
            this.FieldFlag.Flag = (uint)0;

            this.TexPatData0 = new TexPatData();
            this.TexPatData1 = new TexPatData();
            this.TexPatData2 = new TexPatData();

            this.AnimEditData = new AnimEditData();
            this.ChildFlag = new ChildFlag();
            this.ChildAssistData = new ChildAssistData();
            this.ChildData = new ChildData();
            this.ChildTexPatData = new TexPatData();
            this.FieldCollisionData = new FieldCollisionData();
            this.FieldConvergenceData = new FieldConvergenceData();
            this.FieldMagnetData = new FieldMagnetData();
            this.FieldPosAddData = new FieldPosAddData();
            this.FieldRandomData = new FieldRandomData();
            this.FieldCurlNoiseData = new FieldCurlNoiseData();
            this.VolumeRadius = new Vector3();
            this.StripeData = new StripeData();
            this.FluctuationData = new FluctuationData();
            this.FieldSpinData = new FieldSpinData();
            this.UVScroll         = new Vector2( 0.0f, 0.0f );
            this.UVScrollInit     = new Vector2( 0.0f, 0.0f );
            this.UVScrollInitRand = new Vector2( 0.0f, 0.0f );
            this.UVScale          = new Vector2( 0.0f, 0.0f );
            this.UVScaleInit      = new Vector2( 1.0f, 1.0f );
            this.UVScaleInitRand  = new Vector2( 0.0f, 0.0f );
            this.SubTexUVScroll         = new Vector2(0.0f, 0.0f);
            this.SubTexUVScrollInit     = new Vector2(0.0f, 0.0f);
            this.SubTexUVScrollInitRand = new Vector2(0.0f, 0.0f);
            this.SubTexUVScale          = new Vector2(0.0f, 0.0f);
            this.SubTexUVScaleInit      = new Vector2(1.0f, 1.0f);
            this.SubTexUVScaleInitRand  = new Vector2(0.0f, 0.0f);
            this.Tex2UVScroll         = new Vector2( 0.0f, 0.0f );
            this.Tex2UVScrollInit     = new Vector2( 0.0f, 0.0f );
            this.Tex2UVScrollInitRand = new Vector2( 0.0f, 0.0f );
            this.Tex2UVScale          = new Vector2( 0.0f, 0.0f );
            this.Tex2UVScaleInit      = new Vector2( 1.0f, 1.0f );
            this.Tex2UVScaleInitRand  = new Vector2( 0.0f, 0.0f );

            this.UserDataGroups = new EmitterUserData[9];
            for ( int i=0;i<this.UserDataGroups.Length;++i )
            {
                this.UserDataGroups[i] = new EmitterUserData();
                this.UserDataGroups[i].Index = i - 1;
            }

            this.UserShaderParamGroups = new UserShaderParameter[8];
            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
            {
                this.UserShaderParamGroups[i] = new UserShaderParameter();
                this.UserShaderParamGroups[i].Index = i;
            }

            this.AnimTableList = new EmitterAnimTableList();

            this.EmitterColor0 = new RgbaColor( 1.0f, 1.0f, 1.0f, 1.0f );
            this.EmitterColor1 = new RgbaColor( 1.0f, 1.0f, 1.0f, 1.0f );

            this.EmitterFormScale = new Vector3();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public ComplexEmitterData(ComplexEmitterData source)
            : this()
        {
            this.FluctuationFlag = new FluctuationFlag();
            this.FieldFlag = new FieldMask();

            this.Set(source);
        }

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.VolumeType = VolumeType.Point;
            this.PrimitiveEmitterFilePath = string.Empty;
            this.VolumePrimEmitType = VolumePrimitiveEmitTypes.Normal;

            this.IsCompactEmitter = false;

            this.DispParticle = true;

            this.EmitRate = 1.0f;
            this.VolumeLatitudeEnabled = false;
            this.VolumeLatitudeDir = VolumeLatitudeDirection.PlusY;
            this.VolumeLatitude = 180.0f;
            this.VolumeSweepParam = (uint)0xffffffff;
            this.VolumeSweepStart = 0;
            this.VolumeSweepStartRandom = false;
            this.VolumeSurfacePosRand = 0.0f;
            this.VolumeHollow = 0;
            this.VolumeTblIndex = 0;
            this.Volume64TblIndex = 0;

            this.EmitDistEnabled = false;
            this.EmitDistUnit = 1.0f;
            this.EmitDistMin = 0.0f;
            this.EmitDistMax = 10.0f;
            this.EmitDistMargin = 0.0f;

            this.GravityVolume = 0;
            this.GravityDir.Set(0, -1, 0);

            this.FluctuationFlag.ApplyAlpha = true;
            this.FluctuationFlag.ApplyScale = true;

            this.ChildFlag.EmitterFollow = true;
            this.ChildFlag.Alpha0Inherit = true;
            this.ChildFlag.Alpha1Inherit = true;
            this.ChildFlag.ScaleInherit = true;
            this.ChildFlag.RotateInherit = true;

            this.FieldFlag.Flag = (uint)0;
            this.StripeFlag = 0;

            this.TexPatData0.Initialize();
            this.TexPatData1.Initialize();
            this.TexPatData2.Initialize();
            this.AnimEditData.Initialize();
            this.ChildAssistData.Initialize();
            this.ChildData.Initialize();
            this.ChildTexPatData.Initialize();

            this.FieldRandomData.Initialize();
            this.FieldMagnetData.Initialize();
            this.FieldSpinData.Initialize();
            this.FieldCollisionData.Initialize();
            this.FieldCurlNoiseData.Initialize();
            this.FieldConvergenceData.Initialize();
            this.FieldPosAddData.Initialize();


            this.FluctuationData.Initialize();
            this.StripeData.Initialize();

            this.AnimTableList.Initialize();

            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
                this.UserShaderParamGroups[i].Initialize();

            this.AirRegist = 1.0f;
            this.DynamicsRandom = 0.0f;
            this.LifeStep = 10;
            this.LineCenter = 0.0f;

            this.UseSoftParticle = false;
            this.SoftParticleProcessingMethod = SoftParticleProcessingMethod.Fragment;
            this.SoftParticleProcessingSettings = SoftParticleProcessingSettings.Disable;
            this.SoftParticleFadeDistance = 0.0f;
            this.SoftParticleAlphaOffset = 0.0f;
            this.SoftParticleVolumeParam = 0.0f;
            this.SoftParticleSampleParam = 0.5f;
            this.DecalEnabled = false;
            this.DecalAlphaMin = 1.0f;
            this.NearAlphaEnabled = false;
            this.NearAlphaMin = 10.0f;
            this.NearAlphaMax = 30.0f;
            this.FresnelAlphaType = FresnelAlphaTypes.Disable;
            this.FresnelAlphaMin = 1.0f;
            this.FresnelAlphaMax = 0.0f;
            this.ShaderType = ShaderType.Particle;
            this.ShaderSetting = 0;
            this.ShaderParam0 = 0.0f;
            this.ShaderParam1 = 0.0f;
            this.AnimShaderParamInit = 1.0f;
            this.AnimShaderParam2 = 1.0f;
            this.AnimShaderParam3 = 1.0f;
            this.AnimShaderTime2 = 25;
            this.AnimShaderTime3 = 75;
            this.UI_ApplyAlphaToRefraction = true;

            this.ZBufATestType = ZBufATestType.Normal;
            this.DisplaySide = DisplaySideTypes.Both;

            this.UserShaderCompileDef1 = NWKernel.DefaultUserShaderCompileDef1;
            this.UserShaderCompileDef2 = NWKernel.DefaultUserShaderCompileDef2;

            this.MeshType = MeshType.Particle;
            this.TextureTrimming = false;
            this.ParticleType  = BillboardType.Billboard;
            this.StripeType    = StripeType.Stripe;
            this.BillboardPrimitiveFileSource = string.Empty;
            this.PrimitiveScaleType = NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0.PrimitiveScaleTypes.UseScaleX;

            this.EmitterVelDir.Set(0, 1, 0);
            this.EmitterVelDirAngle = 0.0f;
            this.YAxisDiffusionVel = 0.0f;

            this.InitPosRand = 0.0f;

            float worldScale = s_WorldScaleRange / 100.0f;
            {
                this.FigureVelocity = 0.1f * 1.0f * worldScale; // *velScale;
                this.VolumeRadius.Set(1.0f * worldScale, 1.0f * worldScale, 1.0f * worldScale); // *posScale;
            }

            this.UVShiftAnimMode      = 0;
            this.UVRot                = 0.0f;
            this.UVRotInit            = 0.0f;
            this.UVRotInitRand        = 0.0f;

            this.UVScroll.Set( 0.0f, 0.0f );
            this.UVScrollInit.Set( 0.0f, 0.0f );
            this.UVScrollInitRand.Set( 0.0f, 0.0f );
            this.UVScale.Set( 0.0f, 0.0f );
            this.UVScaleInit.Set( 1.0f, 1.0f );
            this.UVScaleInitRand.Set( 0.0f, 0.0f );

            this.SubTexUVShiftAnimMode = 0;
            this.SubTexUVRot = 0.0f;
            this.SubTexUVRotInit = 0.0f;
            this.SubTexUVRotInitRand = 0.0f;

            this.SubTexUVScroll.Set(0.0f, 0.0f);
            this.SubTexUVScrollInit.Set(0.0f, 0.0f);
            this.SubTexUVScrollInitRand.Set(0.0f, 0.0f);
            this.SubTexUVScale.Set(0.0f, 0.0f);
            this.SubTexUVScaleInit.Set(1.0f, 1.0f);
            this.SubTexUVScaleInitRand.Set(0.0f, 0.0f);

            this.Tex2UVShiftAnimMode = 0;
            this.Tex2UVRot = 0.0f;
            this.Tex2UVRotInit = 0.0f;
            this.Tex2UVRotInitRand = 0.0f;

            this.Tex2UVScroll.Set(0.0f, 0.0f);
            this.Tex2UVScrollInit.Set(0.0f, 0.0f);
            this.Tex2UVScrollInitRand.Set(0.0f, 0.0f);
            this.Tex2UVScale.Set(0.0f, 0.0f);
            this.Tex2UVScaleInit.Set(1.0f, 1.0f);
            this.Tex2UVScaleInitRand.Set(0.0f, 0.0f);

            this.UserCallbackID = -1;
            for ( int i=0;i<this.UserDataGroups.Length;++i )
                this.UserDataGroups[i].Initialize();

            this.EmitterColor0.R = 1.0f;
            this.EmitterColor0.G = 1.0f;
            this.EmitterColor0.B = 1.0f;
            this.EmitterColor0.A = 1.0f;

            this.EmitterColor1.R = 1.0f;
            this.EmitterColor1.G = 1.0f;
            this.EmitterColor1.B = 1.0f;
            this.EmitterColor1.A = 1.0f;

            this.EmitterAlpha = 1.0f;

            this.EmitterFormScale.Set( 1.0f, 1.0f, 1.0f );

            this.ParticleSortType = ParticleSortType.Ascending;
            //             this.ChildDataOffset = 9;

            this.ParticleShapeOffsetType = ParticleShapeOffsetTypes.DepthDir;
            this.BillboardZOffset = 0.0f;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(ComplexEmitterData source)
        {
            Ensure.Argument.NotNull(source);

            // SimpleEmitterData
            /*
             *  SimpleEmitterData用
             */
            this.Type = source.Type;
            this.Name = source.Name;

            this.IsCompactEmitter = source.IsCompactEmitter;

            this.DispParticle = source.DispParticle;
            this.EnableGpuParticle = source.EnableGpuParticle;

            this.EmitterVelDir.Set(source.EmitterVelDir);
            this.GravityDir.Set(source.GravityDir);
            this.SpreadVec.Set(source.SpreadVec);
            this.VolumeRadius.Set(source.VolumeRadius);
            this.UVScroll.Set(source.UVScroll);

            this.TexRes0.Set(source.TexRes0);
            this.TexRes1.Set(source.TexRes1);
            this.TexRes2.Set(source.TexRes2);

            this.GravityVolume = source.GravityVolume;
            this.AirRegist = source.AirRegist;
            this.MeshType = source.MeshType;
            this.TextureTrimming = source.TextureTrimming;
            this.ParticleType = source.ParticleType;
            this.StripeType = source.StripeType;
            this.BillboardPrimitiveFileSource = source.BillboardPrimitiveFileSource;
            this.PrimitiveScaleType = source.PrimitiveScaleType;
            this.ParticleShapeOffsetType = source.ParticleShapeOffsetType;
            this.BillboardZOffset = source.BillboardZOffset;
            this.BlendType = source.BlendType;
            this.EmitRate = source.EmitRate;
            this.EmitterVelDirAngle = source.EmitterVelDirAngle;
            this.YAxisDiffusionVel = source.YAxisDiffusionVel;
            this.EmitterVelocity = source.EmitterVelocity;
            this.InitPosRand = source.InitPosRand;
            this.FigureVelocity = source.FigureVelocity;
            this.IsWorldField = source.IsWorldField;
            this.DynamicsRandom = source.DynamicsRandom;
            this.LifeStep = source.LifeStep;
            this.LifeStepRnd = source.LifeStepRnd;
            this.EmitDistEnabled = source.EmitDistEnabled;
            this.EmitDistUnit = source.EmitDistUnit;
            this.EmitDistMin = source.EmitDistMin;
            this.EmitDistMax = source.EmitDistMax;
            this.EmitDistMargin = source.EmitDistMargin;
            this.StartFrame = source.StartFrame;
            this.VolumeLatitudeEnabled = source.VolumeLatitudeEnabled;
            this.VolumeLatitudeDir = source.VolumeLatitudeDir;
            this.VolumeLatitude = source.VolumeLatitude;
            this.VolumeSweepParam = source.VolumeSweepParam;
            this.VolumeSweepStart = source.VolumeSweepStart;
            this.VolumeSweepStartRandom = source.VolumeSweepStartRandom;
            this.VolumeSurfacePosRand = source.VolumeSurfacePosRand;
            this.VolumeHollow = source.VolumeHollow;
            this.VolumeTblIndex = source.VolumeTblIndex;
            this.Volume64TblIndex = source.Volume64TblIndex;
            this.LineCenter = source.LineCenter;
            this.VolumeType = source.VolumeType;
            this.PrimitiveEmitterFilePath = source.PrimitiveEmitterFilePath;
            this.VolumePrimEmitType = source.VolumePrimEmitType;
            this.ZBufATestType = source.ZBufATestType;
            this.DisplaySide = source.DisplaySide;

            /*
             *  Cafe拡張用
             */
            this.UseSoftParticle = source.UseSoftParticle;
            this.SoftParticleProcessingMethod = source.SoftParticleProcessingMethod;
            this.SoftParticleFadeDistance = source.SoftParticleFadeDistance;
            this.SoftParticleAlphaOffset = source.SoftParticleAlphaOffset;
            this.SoftParticleVolumeParam = source.SoftParticleVolumeParam;
            this.SoftParticleSampleParam = source.SoftParticleSampleParam;
            this.SoftParticleProcessingSettings = source.SoftParticleProcessingSettings;
            this.DecalEnabled = source.DecalEnabled;
            this.DecalAlphaMin = source.DecalAlphaMin;
            this.NearAlphaEnabled = source.NearAlphaEnabled;
            this.NearAlphaMin = source.NearAlphaMin;
            this.NearAlphaMax = source.NearAlphaMax;
            this.FresnelAlphaType = source.FresnelAlphaType;
            this.FresnelAlphaMin = source.FresnelAlphaMin;
            this.FresnelAlphaMax = source.FresnelAlphaMax;
            this.ShaderType = source.ShaderType;
            this.ShaderSetting = source.ShaderSetting;
            this.ShaderParam0 = source.ShaderParam0;
            this.ShaderParam1 = source.ShaderParam1;
            this.AnimShaderParamInit = source.AnimShaderParamInit;
            this.AnimShaderParam2 = source.AnimShaderParam2;
            this.AnimShaderParam3 = source.AnimShaderParam3;
            this.AnimShaderTime2 = source.AnimShaderTime2;
            this.AnimShaderTime3 = source.AnimShaderTime3;
            this.UI_ApplyAlphaToRefraction = source.UI_ApplyAlphaToRefraction;

            for ( int i=0;i<this.UserShaderParamGroups.Length;++i )
            {
                this.UserShaderParamGroups[i].Set( source.UserShaderParamGroups[i] );
            }

            this.UserShaderCompileDef1 = source.UserShaderCompileDef1;
            this.UserShaderCompileDef2 = source.UserShaderCompileDef2;

            this.ChildFlag.Set(source.ChildFlag);
            this.FieldFlag.Set(source.FieldFlag);
            this.FluctuationFlag.Set(source.FluctuationFlag);

            this.StripeFlag = source.StripeFlag;

            this.TexPatData0.Set(source.TexPatData0);
            this.TexPatData1.Set(source.TexPatData1);
            this.TexPatData2.Set(source.TexPatData2);

            this.AnimEditData.Set(source.AnimEditData);
            this.ChildFlag.Set(source.ChildFlag);
            this.ChildAssistData.Set(source.ChildAssistData);
            this.ChildData.Set(source.ChildData);
            this.ChildTexPatData.Set(source.ChildTexPatData);
            this.FieldCollisionData.Set(source.FieldCollisionData);
            this.FieldCurlNoiseData.Set(source.FieldCurlNoiseData);
            this.FieldConvergenceData.Set(source.FieldConvergenceData);
            this.FieldMagnetData.Set(source.FieldMagnetData);
            this.FieldPosAddData.Set(source.FieldPosAddData);
            this.FieldRandomData.Set(source.FieldRandomData);
            this.VolumeRadius.Set(source.VolumeRadius);
            this.StripeData.Set(source.StripeData);
            this.FluctuationData.Set(source.FluctuationData);
            this.FieldSpinData.Set(source.FieldSpinData);
            this.ParticleSortType = source.ParticleSortType;

            this.UVShiftAnimMode      = source.UVShiftAnimMode;
            this.UVRot                = source.UVRot;
            this.UVRotInit            = source.UVRotInit;
            this.UVRotInitRand        = source.UVRotInitRand;

            this.UVScroll.Set( source.UVScroll );
            this.UVScrollInit.Set( source.UVScrollInit );
            this.UVScrollInitRand.Set( source.UVScrollInitRand );
            this.UVScale.Set( source.UVScale );
            this.UVScaleInit.Set( source.UVScaleInit );
            this.UVScaleInitRand.Set( source.UVScaleInitRand );

            this.SubTexUVShiftAnimMode = source.SubTexUVShiftAnimMode;
            this.SubTexUVRot = source.SubTexUVRot;
            this.SubTexUVRotInit = source.SubTexUVRotInit;
            this.SubTexUVRotInitRand = source.SubTexUVRotInitRand;

            this.SubTexUVScroll.Set(source.SubTexUVScroll);
            this.SubTexUVScrollInit.Set(source.SubTexUVScrollInit);
            this.SubTexUVScrollInitRand.Set(source.SubTexUVScrollInitRand);
            this.SubTexUVScale.Set(source.SubTexUVScale);
            this.SubTexUVScaleInit.Set(source.SubTexUVScaleInit);
            this.SubTexUVScaleInitRand.Set(source.SubTexUVScaleInitRand);

            this.Tex2UVShiftAnimMode = source.Tex2UVShiftAnimMode;
            this.Tex2UVRot = source.Tex2UVRot;
            this.Tex2UVRotInit = source.Tex2UVRotInit;
            this.Tex2UVRotInitRand = source.Tex2UVRotInitRand;

            this.Tex2UVScroll.Set(source.Tex2UVScroll);
            this.Tex2UVScrollInit.Set(source.Tex2UVScrollInit);
            this.Tex2UVScrollInitRand.Set(source.Tex2UVScrollInitRand);
            this.Tex2UVScale.Set(source.Tex2UVScale);
            this.Tex2UVScaleInit.Set(source.Tex2UVScaleInit);
            this.Tex2UVScaleInitRand.Set(source.Tex2UVScaleInitRand);

            this.UserCallbackID = source.UserCallbackID;
            for ( int i=0;i<this.UserDataGroups.Length;++i )
            {
                this.UserDataGroups[i].Set( source.UserDataGroups[i] );
            }

            this.AnimTableList.Set(source.AnimTableList);

            this.EmitterColor0.Set( source.EmitterColor0 );
            this.EmitterColor1.Set( source.EmitterColor1 );

            this.EmitterAlpha = source.EmitterAlpha;

            this.EmitterFormScale.Set( source.EmitterFormScale );
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override ComplexEmitterData Clone()
        {
            return new ComplexEmitterData(this);
        }

        #endregion

        #region Properties - SimpleEmitter

        /// <summary>
        /// エミッタ・タイプ
        /// </summary>
        public EmitterType Type { get; set; }

        /// <summary>
        /// コンパクトエミッタ。
        /// </summary>
        public bool IsCompactEmitter { get; set; }

        /// <summary>
        /// アドレス解決後の名前
        /// </summary>
        public string Name { get; set; }

        /// <summary>
        /// User callback ID ( -1, 0-7 )
        /// </summary>
        public int UserCallbackID { get; set; }

        /// <summary>
        /// ユーザーデータ
        /// </summary>
        public EmitterUserData[] UserDataGroups { get; set; }

        /// <summary>
        /// パーティクルを描画する
        /// </summary>
        public bool DispParticle { get; set; }

        /// <summary>
        /// GPUパーティクルを利用する
        /// </summary>
        public bool EnableGpuParticle { get; set; }

        /// <summary>
        /// ワールド座標系でフィールドをかけるか？
        /// </summary>
        public bool IsWorldField { get; set; }

        /// <summary>
        ///  テーブルを使うボリューム形状のインデックス
        /// </summary>
        //TODO:ＧＵＩ接続されていない
        public int VolumeTblIndex { get; set; }

        /// <summary>
        ///  テーブルを使うボリューム形状のインデックス
        /// </summary>
        public int Volume64TblIndex { get; set; }

        /// <summary>
        /// ボリュームのタイプ
        /// </summary>
        public VolumeType VolumeType { get; set; }

        /// <summary>
        /// プリミティブエミッタのファイルパス。
        /// </summary>
        public string PrimitiveEmitterFilePath { get; set; }

        /// <summary>
        /// プリミティブエミッタの放出タイプ。
        /// </summary>
        public VolumePrimitiveEmitTypes VolumePrimEmitType { get; set; }

        /// <summary>
        /// ボリューム半径
        /// </summary>
        public Vector3 VolumeRadius { get; set; }

        /// <summary>
        /// 中心位置
        /// </summary>
        public float LineCenter { get; set; }

        /// <summary>
        /// 弧の開き方
        /// </summary>
        public bool VolumeLatitudeEnabled { get; set; }

        /// <summary>
        /// 軸
        /// </summary>
        public VolumeLatitudeDirection VolumeLatitudeDir { get; set; }

        /// <summary>
        /// 弧の幅（角度）
        /// </summary>
        public float VolumeLatitude { get; set; }

        /// <summary>
        /// 弧の幅（開始）
        /// </summary>
        public uint VolumeSweepStart { get; set; }

        /// <summary>
        /// 弧の開始角度ランダム
        /// </summary>
        public bool VolumeSweepStartRandom { get; set; }

        /// <summary>
        /// 円（等分割）のランダム位置
        /// </summary>
        public float VolumeSurfacePosRand { get; set; }

        /// <summary>
        /// 弧の計算に使う値（ボリューム形状によって異なります）
        /// </summary>
        public uint VolumeSweepParam { get; set; }

        /// <summary>
        /// 空洞率
        /// </summary>
        public int VolumeHollow { get; set; }

        /// <summary>
        /// ランダム位置
        /// </summary>
        public float InitPosRand { get; set; }

        /// <summary>
        /// エミッタ速度
        /// </summary>
        public float FigureVelocity { get; set; }

        /// <summary>
        /// エミッタ速度の方向
        /// </summary>
        public Vector3 EmitterVelDir { get; set; }

        /// <summary>
        /// 指定方向拡散角度
        /// </summary>
        public float EmitterVelDirAngle { get; set; }

        /// <summary>
        /// Ｙ軸拡散速度
        /// </summary>
        public float YAxisDiffusionVel { get; set; }

        /// <summary>
        /// エミッタ速度
        /// </summary>
        public float EmitterVelocity { get; set; }

        /// <summary>
        /// 拡散ベクトル
        /// </summary>
        public Vector3 SpreadVec { get; set; }

        /// <summary>
        /// 放出開始フレーム
        /// </summary>
        public int StartFrame { get; set; }

        /// <summary>
        /// 運動量ランダム
        /// </summary>
        public float DynamicsRandom { get; set; }

        /// <summary>
        /// ステップ値
        /// </summary>
        public int LifeStep { get; set; }

        /// <summary>
        /// ステップ値のランダム
        /// </summary>
        public int LifeStepRnd { get; set; }

        /// <summary>
        /// 放出レート（数）
        /// </summary>
        public float EmitRate { get; set; }

        /// <summary>
        /// 放出間隔タイプ
        /// </summary>
        public bool EmitDistEnabled { get; set; }

        /// <summary>
        /// 放出間隔
        /// </summary>
        public float EmitDistUnit { get; set; }

        /// <summary>
        /// １フレームあたりの最小放出個数。
        /// </summary>
        public float EmitDistMin { get; set; }

        /// <summary>
        /// １フレームあたりの最大放出個数。
        /// </summary>
        public float EmitDistMax { get; set; }

        /// <summary>
        /// 移動距離切り捨ての閾値。
        /// </summary>
        public float EmitDistMargin { get; set; }

        /// <summary>
        /// 空気抵抗
        /// </summary>
        public float AirRegist { get; set; }

        /// <summary>
        /// ブレンドのタイプ
        /// </summary>
        public BlendType BlendType { get; set; }

        /// <summary>
        /// Mesh type
        /// </summary>
        public MeshType MeshType { get; set; }

        /// <summary>
        /// テクスチャトリミング
        /// </summary>
        public bool TextureTrimming { get; set; }

        /// <summary>
        /// パーティクルタイプ
        /// </summary>
        public BillboardType ParticleType { get; set; }

        /// <summary>
        /// スケールの適用タイプ
        /// </summary>
        public PrimitiveScaleTypes PrimitiveScaleType { get; set; }

        /// <summary>
        /// ストライプタイプ
        /// </summary>
        public StripeType StripeType { get; set; }

        /// <summary>
        /// プリミティブファイルパス
        /// </summary>
        public string BillboardPrimitiveFileSource { get; set; }

        /// <summary>
        /// ビルボードのオフセットの処理方法
        /// </summary>
        public ParticleShapeOffsetTypes ParticleShapeOffsetType { get; set; }

        /// <summary>
        /// ビルボードのスクリーンZ方向へのオフセット
        /// </summary>
        public float BillboardZOffset { get; set; }

        /// <summary>
        /// Ｚバッファー、αテストタイプ
        /// </summary>
        public ZBufATestType ZBufATestType { get; set; }

        /// <summary>
        /// カリング
        /// </summary>
        public DisplaySideTypes DisplaySide { get; set; }

        /// <summary>
        /// 重量の大きさ
        /// </summary>
        public float GravityVolume { get; set; }

        /// <summary>
        /// 重力の向き
        /// </summary>
        public Vector3 GravityDir { get; set; }

        /// <summary>
        /// シェーダータイプ：ソフトパーティクルを利用する
        /// </summary>
        public bool UseSoftParticle { get; set; }

        /// <summary>
        /// ソフトパーティクル：処理方法
        /// </summary>
        public SoftParticleProcessingMethod SoftParticleProcessingMethod { get; set; }

        /// <summary>
        /// ソフトパーティクル：セッティング
        /// </summary>
        public SoftParticleProcessingSettings SoftParticleProcessingSettings
        {
            get
            {
                if ( UseSoftParticle == false )
                {
                    return SoftParticleProcessingSettings.Disable;
                }
                else
                {
                    if ( SoftParticleProcessingMethod == SoftParticleProcessingMethod.Fragment )
                        return SoftParticleProcessingSettings.Fragment;
                    else if ( SoftParticleProcessingMethod == SoftParticleProcessingMethod.Vertex )
                        return SoftParticleProcessingSettings.Vertex;
                    else
                        return SoftParticleProcessingSettings.LensFlare;
                }
            }
            set
            {
                if ( value==SoftParticleProcessingSettings.Disable )
                    UseSoftParticle = false;
                else
                {
                    if ( value == SoftParticleProcessingSettings.LensFlare )
                    {
                        UseSoftParticle = true;
                        SoftParticleProcessingMethod = SoftParticleProcessingMethod.LensFlare;
                    }
                    else if ( value == SoftParticleProcessingSettings.Vertex )
                    {
                        UseSoftParticle = true;
                        SoftParticleProcessingMethod = SoftParticleProcessingMethod.Vertex;
                    }
                    else if ( value == SoftParticleProcessingSettings.Fragment )
                    {
                        UseSoftParticle = true;
                        SoftParticleProcessingMethod = SoftParticleProcessingMethod.Fragment;
                    }
                }

            }
        }

        /// <summary>
        /// Fade distance for soft particle.
        /// </summary>
        public float SoftParticleFadeDistance { get; set; }

        /// <summary>
        /// Alpha offset for soft particle.
        /// </summary>
        public float SoftParticleAlphaOffset { get; set; }

        /// <summary>
        /// Volume parameter for the soft particle
        /// </summary>
        public float SoftParticleVolumeParam { get; set; }

        /// <summary>
        /// Sampling parameter for the soft particle
        /// </summary>
        public float SoftParticleSampleParam { get; set; }

        /// <summary>
        /// デカール：デカールを利用する？
        /// </summary>
        public bool DecalEnabled { get; set; }

        /// <summary>
        /// デカール：アルファが最小になる角度。
        /// </summary>
        public float DecalAlphaMin { get; set; }

        /// <summary>
        /// 近距離アルファ：近距離アルファを利用する？
        /// </summary>
        public bool NearAlphaEnabled { get; set; }

        /// <summary>
        /// 近距離アルファ：アルファが0になる距離。
        /// </summary>
        public float NearAlphaMin { get; set; }

        /// <summary>
        /// 近距離アルファ：アルファが1になる距離。
        /// </summary>
        public float NearAlphaMax { get; set; }

        /// <summary>
        /// フレネルアルファ：フレネルアルファの処理方法。
        /// </summary>
        public FresnelAlphaTypes FresnelAlphaType { get; set; }

        /// <summary>
        /// フレネルアルファ：最小値
        /// </summary>
        public float FresnelAlphaMin { get; set; }

        /// <summary>
        /// フレネルアルファ：最大値
        /// </summary>
        public float FresnelAlphaMax { get; set; }

        /// <summary>
        /// シェーダータイプ
        /// </summary>
        public ShaderType ShaderType { get; set; }

        /// <summary>
        /// シェーダカスタマイズのセッティング
        /// </summary>
        public uint ShaderSetting { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam0 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public float ShaderParam1 { get; set; }

        /// <summary>
        /// シェーダパラメータ
        /// </summary>
        public Vector2 ShaderParams01
        {
            get
            {
                return new Vector2(ShaderParam0, ShaderParam1);
            }
            set
            {
                ShaderParam0 = value.X;
                ShaderParam1 = value.Y;
            }
        }

        /// <summary>
        /// シェーダパラメータアニメの初期値。
        /// </summary>
        public float AnimShaderParamInit { get; set; }

        /// <summary>
        /// シェーダパラメータアニメの基準値。
        /// </summary>
        public float AnimShaderParam2 { get; set; }

        /// <summary>
        /// シェーダパラメータアニメの目標値。
        /// </summary>
        public float AnimShaderParam3 { get; set; }

        /// <summary>
        /// シェーダパラメータアニメの前半セクション時間。
        /// </summary>
        public int AnimShaderTime2 { get; set; }

        /// <summary>
        /// シェーダパラメータアニメの後半セクション時間。
        /// </summary>
        public int AnimShaderTime3 { get; set; }

        /// <summary>
        /// アルファ値を反映する。
        /// </summary>
        public bool UI_ApplyAlphaToRefraction { get; set; }

        /// <summary>
        /// ユーザーシェーダパラメータ
        /// </summary>
        public UserShaderParameter[] UserShaderParamGroups { get; set; }

        /// <summary>
        /// ユーザーシェーダコンパイル定義１
        /// </summary>
        public string UserShaderCompileDef1 { get; set; }

        /// <summary>
        /// ユーザーシェーダコンパイル定義２
        /// </summary>
        public string UserShaderCompileDef2 { get; set; }


        /// <summary>
        /// The list of animation tables.
        /// </summary>
        public EmitterAnimTableList AnimTableList { get; set; }


        /// <summary>
        /// グローバル乗算カラー０
        /// </summary>
        public RgbaColor EmitterColor0 { get; set; }


        /// <summary>
        /// グローバル乗算カラー１
        /// </summary>
        public RgbaColor EmitterColor1 { get; set; }


        /// <summary>
        /// グローバル乗算アルファ
        /// </summary>
        public float EmitterAlpha { get; set; }


        /// <summary>
        /// エミッタ形状スケール
        /// </summary>
        public Vector3 EmitterFormScale { get; set; }

        #endregion

        #region Properties - Texture 0

        /// <summary>
        /// テクスチャリソース０
        /// </summary>
        public TextureRes TexRes0 { get; set; }

        /// <summary>
        /// エミッタ：テクスチャ０用。
        /// </summary>
        public TexPatData TexPatData0 { get; set; }

        #endregion

        #region Properties - Texture 1

        /// <summary>
        /// テクスチャリソース１
        /// </summary>
        public TextureRes TexRes1 { get; set; }

        /// <summary>
        /// エミッタ：テクスチャ１用。
        /// </summary>
        public TexPatData TexPatData1 { get; set; }

        #endregion

        #region Properties - Texture 2

        /// <summary>
        /// テクスチャリソース２
        /// </summary>
        public TextureRes TexRes2 { get; set; }

        /// <summary>
        /// エミッタ：テクスチャ２用。
        /// </summary>
        public TexPatData TexPatData2 { get; set; }

        #endregion

        #region Properties - Texture 0 UV animation

        /// <summary>
        /// ＵＶアニメーション
        /// </summary>
        public int UVShiftAnimMode { get; set; }

        #region Bitmask access

        public bool TexScrollAnimEnabled
        {
            get { return (UVShiftAnimMode & 8) == 8; }
            set { UVShiftAnimMode = (value ? 8 : 0) | (UVShiftAnimMode & ~8); }
        }

        public bool TexScaleAnimEnabled
        {
            get { return (UVShiftAnimMode & 16) == 16; }
            set { UVShiftAnimMode = (value ? 16 : 0) | (UVShiftAnimMode & ~16); }
        }

        public bool TexRotateAnimEnabled
        {
            get { return (UVShiftAnimMode & 32) == 32; }
            set { UVShiftAnimMode = (value ? 32 : 0) | (UVShiftAnimMode & ~32); }
        }

        #endregion

        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2 UVScroll { get; set; }


        /// <summary>
        /// ＵＶスクロール初期値
        /// </summary>
        public Vector2 UVScrollInit { get; set; }


        /// <summary>
        /// ＵＶスクロール　ランダム
        /// </summary>
        public Vector2 UVScrollInitRand { get; set; }


        /// <summary>
        /// ＵＶスケール加算値
        /// </summary>
        public Vector2 UVScale { get; set; }


        /// <summary>
        /// ＵＶスケール初期値
        /// </summary>
        public Vector2 UVScaleInit { get; set; }


        /// <summary>
        /// ＵＶスケール　ランダム
        /// </summary>
        public Vector2 UVScaleInitRand { get; set; }


        /// <summary>
        /// ＵＶ回転加算値
        /// </summary>
        public float UVRot { get; set; }


        /// <summary>
        /// ＵＶ回転初期値
        /// </summary>
        public float UVRotInit { get; set; }


        /// <summary>
        /// ＵＶ回転　ランダム
        /// </summary>
        public float UVRotInitRand { get; set; }

        #endregion

        #region Properties - Texture 1 UV animation

        /// <summary>
        /// ＵＶアニメーション
        /// </summary>
        public int SubTexUVShiftAnimMode { get; set; }

        #region Bitmask access

        public bool SubTexScrollAnimEnabled
        {
            get { return (SubTexUVShiftAnimMode & 8) == 8; }
            set { SubTexUVShiftAnimMode = (value ? 8 : 0) | (SubTexUVShiftAnimMode & ~8); }
        }

        public bool SubTexScaleAnimEnabled
        {
            get { return (SubTexUVShiftAnimMode & 16) == 16; }
            set { SubTexUVShiftAnimMode = (value ? 16 : 0) | (SubTexUVShiftAnimMode & ~16); }
        }

        public bool SubTexRotateAnimEnabled
        {
            get { return (SubTexUVShiftAnimMode & 32) == 32; }
            set { SubTexUVShiftAnimMode = (value ? 32 : 0) | (SubTexUVShiftAnimMode & ~32); }
        }

        #endregion

        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2 SubTexUVScroll { get; set; }


        /// <summary>
        /// ＵＶスクロール初期値
        /// </summary>
        public Vector2 SubTexUVScrollInit { get; set; }


        /// <summary>
        /// ＵＶスクロール　ランダム
        /// </summary>
        public Vector2 SubTexUVScrollInitRand { get; set; }


        /// <summary>
        /// ＵＶスケール加算値
        /// </summary>
        public Vector2 SubTexUVScale { get; set; }


        /// <summary>
        /// ＵＶスケール初期値
        /// </summary>
        public Vector2 SubTexUVScaleInit { get; set; }


        /// <summary>
        /// ＵＶスケール　ランダム
        /// </summary>
        public Vector2 SubTexUVScaleInitRand { get; set; }


        /// <summary>
        /// ＵＶ回転加算値
        /// </summary>
        public float SubTexUVRot { get; set; }


        /// <summary>
        /// ＵＶ回転初期値
        /// </summary>
        public float SubTexUVRotInit { get; set; }


        /// <summary>
        /// ＵＶ回転　ランダム
        /// </summary>
        public float SubTexUVRotInitRand { get; set; }

        #endregion

        #region Properties - Texture 2 UV animation

        /// <summary>
        /// ＵＶアニメーション
        /// </summary>
        public int Tex2UVShiftAnimMode { get; set; }

        #region Bitmask access

        public bool Tex2ScrollAnimEnabled
        {
            get { return (Tex2UVShiftAnimMode & 8) == 8; }
            set { Tex2UVShiftAnimMode = (value ? 8 : 0) | (Tex2UVShiftAnimMode & ~8); }
        }

        public bool Tex2ScaleAnimEnabled
        {
            get { return (Tex2UVShiftAnimMode & 16) == 16; }
            set { Tex2UVShiftAnimMode = (value ? 16 : 0) | (Tex2UVShiftAnimMode & ~16); }
        }

        public bool Tex2RotateAnimEnabled
        {
            get { return (Tex2UVShiftAnimMode & 32) == 32; }
            set { Tex2UVShiftAnimMode = (value ? 32 : 0) | (Tex2UVShiftAnimMode & ~32); }
        }

        #endregion

        /// <summary>
        /// ＵＶスクロール加算値
        /// </summary>
        public Vector2 Tex2UVScroll { get; set; }


        /// <summary>
        /// ＵＶスクロール初期値
        /// </summary>
        public Vector2 Tex2UVScrollInit { get; set; }


        /// <summary>
        /// ＵＶスクロール　ランダム
        /// </summary>
        public Vector2 Tex2UVScrollInitRand { get; set; }


        /// <summary>
        /// ＵＶスケール加算値
        /// </summary>
        public Vector2 Tex2UVScale { get; set; }


        /// <summary>
        /// ＵＶスケール初期値
        /// </summary>
        public Vector2 Tex2UVScaleInit { get; set; }


        /// <summary>
        /// ＵＶスケール　ランダム
        /// </summary>
        public Vector2 Tex2UVScaleInitRand { get; set; }


        /// <summary>
        /// ＵＶ回転加算値
        /// </summary>
        public float Tex2UVRot { get; set; }


        /// <summary>
        /// ＵＶ回転初期値
        /// </summary>
        public float Tex2UVRotInit { get; set; }


        /// <summary>
        /// ＵＶ回転　ランダム
        /// </summary>
        public float Tex2UVRotInitRand { get; set; }

        #endregion

        #region Properties - ComplexEmitter

        /// <summary>
        /// チャイルド：テクスチャ用
        /// </summary>
        public TexPatData ChildTexPatData { get; set; }

        /// <summary>
        /// チャイルド用フラグ
        /// </summary>
        public ChildFlag ChildFlag { get; set; }

        /// <summary>
        /// フィールドを使用するか？
        /// </summary>
        public FieldMask FieldFlag { get; set; }

        /// <summary>
        /// 揺らぎ用フラグ
        /// </summary>
        public FluctuationFlag FluctuationFlag { get; set; }

        /// <summary>
        /// ストライプ用フラグ
        /// </summary>
        public int StripeFlag
        {
            get
            {
                if ( this.StripeData==null )
                    return 0;

                return ( this.StripeData.FollowEmitter==true ? (int)NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0.StripeFlag.EmitterCoord : 0 );
            }
            set
            {
                if ( this.StripeData==null )
                    return;

                this.StripeData.FollowEmitter = ( (value & (int)NWCore.DataModel.Major_1.Minor_9.Build_0.Revision_0.StripeFlag.EmitterCoord)!=0 );
            }
        }

        /// <summary>
        ///
        /// </summary>
        public AnimEditData AnimEditData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ChildAssistData ChildAssistData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public ChildData ChildData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldRandomData FieldRandomData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldMagnetData FieldMagnetData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldSpinData FieldSpinData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldCollisionData FieldCollisionData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldCurlNoiseData FieldCurlNoiseData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldConvergenceData FieldConvergenceData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FieldPosAddData FieldPosAddData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public FluctuationData FluctuationData { get; set; }

        /// <summary>
        ///
        /// </summary>
        public StripeData StripeData { get; set; }

        /// <summary>
        /// ソートオーダー
        /// </summary>
        public ParticleSortType ParticleSortType { get; set; }

        #endregion
    }
}
