﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_8.Build_0.Revision_0
{


    /// <summary>
    /// Class for storing the loaded texture information.
    /// </summary>
    public class TextureInfo
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public TextureInfo()
        {
            this.CopyTextureResources = true;
            Initialize();
        }

        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="source">Source texture information.</param>
        public TextureInfo( TextureInfo source ) :
            this()
        {
            this.Set(source);
        }

        #endregion

        #region Utility

        /// <summary>
        /// Initialize the data.
        /// </summary>
        public void Initialize()
        {
            this.FileName            = string.Empty;
            this.Width               = 0;
            this.Height              = 0;
            this.Depth               = 1;
            this.TileMode            = string.Empty;
            this.Swizzle             = 0;
            this.Alignment           = 0;
            this.Pitch               = 0;
            this.MipLevel            = 0;
            this.CompSel             = 0;
            this.MipOffset           = null;
            this.TrimSize            = null;
            this.OriginalDataFormat  = string.Empty;
            this.OriginalDataSize    = 0;
            this.OriginalImage       = null;
            this.NativeDataFormat    = string.Empty;
            this.NativeDataSize      = 0;
            this.NativeImage         = null;
            this.Linear              = null;
            this.IsNativeTextureType = false;
        }


        /// <summary>
        /// Set data.
        /// </summary>
        public void Set( TextureInfo info )
        {
            if ( info==null )
                return;

            bool bShouldCopyTextures = false;
            if ( (NWKernel.ShouldOverrideTextureCopyFlags==true &&
                  NWKernel.OverriddenTextureCopyFlag==true) ||
                 (NWKernel.ShouldOverrideTextureCopyFlags==false &&
                  this.CopyTextureResources==true) )
            {
                bShouldCopyTextures = true;
            }

            this.FileName            = info.FileName;
            this.Width               = info.Width;
            this.Height              = info.Height;
            this.Depth               = info.Depth;
            this.Swizzle             = info.Swizzle;
            this.Alignment           = info.Alignment;
            this.Pitch               = info.Pitch;
            this.MipLevel            = info.MipLevel;
            this.CompSel             = info.CompSel;
            this.OriginalDataSize    = info.OriginalDataSize;
            this.NativeDataSize      = info.NativeDataSize;
            this.Linear              = info.Linear;
            this.IsNativeTextureType = info.IsNativeTextureType;

            if ( string.IsNullOrEmpty(info.TileMode)==true )
                this.TileMode = string.Empty;
            else
                this.TileMode = string.Copy( info.TileMode );

            if ( string.IsNullOrEmpty(info.OriginalDataFormat)==true )
                this.OriginalDataFormat = string.Empty;
            else
                this.OriginalDataFormat = string.Copy( info.OriginalDataFormat );

            if ( string.IsNullOrEmpty(info.NativeDataFormat)==true )
                this.NativeDataFormat = string.Empty;
            else
                this.NativeDataFormat = string.Copy( info.NativeDataFormat );

            if ( info.MipOffset!=null )
            {
                try
                {
                    if ( bShouldCopyTextures==true )
                    {
                        this.MipOffset = new uint[info.MipOffset.Length];
                        Buffer.BlockCopy( info.MipOffset,
                                          0,
                                          this.MipOffset,
                                          0,
                                          info.MipOffset.Length * sizeof(uint) );
                    }
                    else
                    {
                        this.MipOffset = info.MipOffset;
                    }
                }
                catch ( Exception ex )
                {
                    string msg = "Failed copying texture mipmap offset array.\r\n" +
                                 this.FileName +
                                 "\r\n" + ex.Message + "\r\n" +
                                 NWKernel.OutputMemoryUsage();
                    NWKernel.OutputLog( LogLevels.Fatal, msg );
                }
            }

            if ( info.TrimSize!=null )
            {
                this.TrimSize = (uint[])info.TrimSize.Clone();
            }

            if ( info.OriginalImage!=null )
            {
                try
                {
                    if ( bShouldCopyTextures==true )
                    {
                        this.OriginalImage = new byte[info.OriginalImage.Length];
                        Buffer.BlockCopy( info.OriginalImage,
                                          0,
                                          this.OriginalImage,
                                          0,
                                          info.OriginalImage.Length * sizeof(byte) );
                    }
                    else
                    {
                        this.OriginalImage = info.OriginalImage;
                    }
                }
                catch ( Exception ex )
                {
                    string msg = "Failed copying original image of the texture.\r\n" +
                                 this.FileName +
                                 "\r\n" + ex.Message + "\r\n" +
                                 NWKernel.OutputMemoryUsage();
                    NWKernel.OutputLog( LogLevels.Fatal, msg );
                }
            }

            if ( info.NativeImage!=null )
            {
                try
                {
                    if ( bShouldCopyTextures==true )
                    {
                        this.NativeImage = new byte[info.NativeImage.Length];
                        Buffer.BlockCopy( info.NativeImage,
                                          0,
                                          this.NativeImage,
                                          0,
                                          info.NativeImage.Length * sizeof( byte ) );
                    }
                    else
                    {
                        this.NativeImage = info.NativeImage;
                    }
                }
                catch ( Exception ex )
                {
                    string msg = "Failed copying native image of the texture.\r\n" +
                                 this.FileName +
                                 "\r\n" + ex.Message + "\r\n" +
                                 NWKernel.OutputMemoryUsage();
                    NWKernel.OutputLog( LogLevels.Fatal, msg );
                }
            }
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public TextureInfo Clone()
        {
            return new TextureInfo( this );
        }

        #endregion

        #region Properties

        /// <summary>File name of the texture.</summary>
        public string FileName { get; set; }

        /// <summary>Width of the image.</summary>
        public uint   Width { get; set; }

        /// <summary>Height of the image.</summary>
        public uint   Height { get; set; }

        /// <summary>Depth.</summary>
        public int Depth { get; set; }

        /// <summary>Tile mode.</summary>
        public string TileMode { get; set; }

        /// <summary>Swizzle.</summary>
        public int   Swizzle { get; set; }

        /// <summary>Alignment.</summary>
        public uint   Alignment { get; set; }

        /// <summary>Pitch of the image.</summary>
        public uint   Pitch { get; set; }

        /// <summary>Mip level of the image.</summary>
        public uint   MipLevel { get; set; }

        /// <summary>Component selector.</summary>
        public uint   CompSel { get; set; }

        /// <summary>Mipmap offset.</summary>
        public uint[] MipOffset { get; set; }

        /// <summary>Trimming size.</summary>
        public uint[] TrimSize { get; set; }


        /// <summary>Original image format.</summary>
        public string OriginalDataFormat { get; set; }

        /// <summary>Original image data size.</summary>
        public int    OriginalDataSize { get; set; }

        /// <summary>Original image data.</summary>
        public byte[] OriginalImage { get; set; }


        /// <summary>Native image format.</summary>
        public string NativeDataFormat { get; set; }

        /// <summary>Native image data size.</summary>
        public int    NativeDataSize { get; set; }

        /// <summary>Native image data.</summary>
        public byte[] NativeImage { get; set; }

        /// <summary>Linear</summary>
        public bool[] Linear { get; set; }

        /// <summary>Get or set the flag indicating whether this texture format is native to the platform.</summary>
        public bool IsNativeTextureType { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether to copy the texture resources
        /// or just to copy the reference to the texture resources.
        /// </summary>
        public bool CopyTextureResources { get; set; }

        #endregion
    }


    /// <summary>
    /// プロジェクトヘッダー
    /// </summary>
    public sealed class TextureRes : BaseDataModel<TextureRes>
    {
        Bitmap[] m_colorBitmaps;
        Bitmap[] m_alphaBitmaps;

        #region Properties

        /// <summary>
        /// Color Bitmap
        /// </summary>
        public Bitmap[] ColorBitmap
        {
            get { return m_colorBitmaps; }
        }

        /// <summary>
        /// Alpha Bitmap
        /// </summary>
        public Bitmap[] AlphaBitmap
        {
            get { return m_alphaBitmaps; }
        }

        /// <summary>
        /// テクスチャー幅
        /// </summary>
        public int Width
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャー高さ
        /// </summary>
        public int Height
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャーフォーマット
        /// </summary>
        public int Format
        {
            get;
            set;
        }

        /// <summary>
        /// 名前（名前テーブル先頭からのオフセット
        /// </summary>
        public int NumMipLevel
        {
            get;
            set;
        }

        /// <summary>
        /// ラップモード（上位:U 下位：V）
        /// </summary>
        public int WrapMode
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャフィルタ
        /// </summary>
        public FilterMode FilterMode
        {
            get;
            set;
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public int DataSize
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャーのデータ位置（テクスチャーテーブルからのオフセット）
        /// </summary>
        public int DataPos
        {
            get;
            set;
        }

        /// <summary>
        /// // テクスチャーのハンドル
        /// </summary>
        public uint Handle
        {
            get;
            set;
        }

        /// <summary>
        /// Texture information for binary converter.
        /// </summary>
        public TextureInfo TextureInfo
        {
            get;
            set;
        }

        /// <summary>
        /// Get or set the flag indicating whether to copy the texture resources
        /// or just to copy the reference to the texture resources.
        /// </summary>
        public bool CopyTextureResources
        {
            get { return m_bCopyTextureResources; }
            set
            {
                m_bCopyTextureResources = value;
                if ( this.TextureInfo!=null )
                    this.TextureInfo.CopyTextureResources = value;
            }
        }

        #endregion

        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public TextureRes()
        {
            m_colorBitmaps = null;
            m_alphaBitmaps = null;

            this.TextureInfo = new TextureInfo();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public TextureRes(TextureRes source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// テクスチャーの設定
        /// </summary>
        /// <param name="colorBmp">RGB Bitmap</param>
        /// <param name="alphaBmp">Alpha Bitmap</param>
        public void SetTexture(
            Bitmap[] colorBmps,
            Bitmap[] alphaBmps,
            int mipmapLevel,
            int dataSize,
            int format)
        {
            m_colorBitmaps = colorBmps;
            m_alphaBitmaps = alphaBmps;

            if ( colorBmps==null )
            {
                this.Width       =  0;
                this.Height      =  0;
                this.Format      = -1;
                this.NumMipLevel =  0;
                this.WrapMode    =  0;
                this.DataSize    =  0;
                this.DataPos     =  0;
            }
            else if ( colorBmps.Length>0 )
            {
                this.Width = colorBmps[0].Width;
                this.Height = colorBmps[0].Height;
//TODO:Format はまだ未完成
                this.Format = format;
                this.NumMipLevel = mipmapLevel;
                this.WrapMode = 0;
                this.DataSize = dataSize;
                this.DataPos = 0;
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.Width       = 0;
            this.Height      = 0;
            this.Format      = 0;
            this.NumMipLevel = 0;
            this.WrapMode    = 0;
            this.DataSize    = 0;
            this.DataPos     = 0;
            this.Handle      = 0;

            this.TextureInfo.Initialize();
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(TextureRes source)
        {
            Ensure.Argument.NotNull(source);

            this.Width = source.Width;
            this.Height = source.Height;
            this.Format = source.Format;
            this.NumMipLevel = source.NumMipLevel;
            this.WrapMode = source.WrapMode;
            this.DataSize = source.DataSize;
            this.DataPos = source.DataPos;
            this.Handle = source.Handle;

            #region Texture image resources

            if ( (NWKernel.ShouldOverrideTextureCopyFlags==true &&
                  NWKernel.OverriddenTextureCopyFlag==true) ||
                 (NWKernel.ShouldOverrideTextureCopyFlags==false &&
                  this.CopyTextureResources==true) )
            {
                if ( source.m_colorBitmaps!=null )
                    this.m_colorBitmaps = source.m_colorBitmaps.Clone() as Bitmap[];
                else
                    this.m_colorBitmaps = null;

                if ( source.m_alphaBitmaps!=null )
                    this.m_alphaBitmaps = source.m_alphaBitmaps.Clone() as Bitmap[];
                else
                    this.m_alphaBitmaps = null;
            }
            else
            {
                this.m_colorBitmaps = source.m_colorBitmaps;
                this.m_alphaBitmaps = source.m_alphaBitmaps;
            }

            #endregion

            this.TextureInfo.Set( source.TextureInfo );
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override TextureRes Clone()
        {
            return new TextureRes(this);
        }

        #endregion

        #region Member variables

        private bool m_bCopyTextureResources = true;

        #endregion
    }
}
