﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_1.Minor_8.Build_0.Revision_0
{
    /// <summary>
    /// Class for storing user shader parameters.
    /// </summary>
    [Serializable]
    public class EmitterUserData : BaseDataModel<EmitterUserData>
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public EmitterUserData()
        {
            this.Index    = -1;
            this.BitFlags = new bool[16];
            this.ParamsI  = new uint[6];
            this.ParamsF  = new float[8];
        }


        /// <summary>
        /// Copy constructor.
        /// </summary>
        /// <param name="src">Source to copy from.</param>
        public EmitterUserData( EmitterUserData src ) :
            this()
        {
            this.Set( src );
        }


        /// <summary>
        /// Initialize.
        /// </summary>
        public override void Initialize()
        {
            int i;
            for ( i=0;i<this.BitFlags.Length;++i )
                this.BitFlags[i] = false;

            for ( i=0;i<this.ParamsI.Length;++i )
                this.ParamsI[i] = 0;

            for ( i=0;i<this.ParamsF.Length;++i )
                this.ParamsF[i] = 0.0f;
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// Set parameters from the given source.
        /// </summary>
        /// <param name="src">Source to set from.</param>
        public override void Set( EmitterUserData src )
        {
            Ensure.Argument.NotNull( src );

            this.Index = src.Index;

            int i;
            for ( i=0;i<this.BitFlags.Length;++i )
                this.BitFlags[i] = src.BitFlags[i];

            for ( i=0;i<this.ParamsI.Length;++i )
                this.ParamsI[i] = src.ParamsI[i];

            for ( i=0;i<this.ParamsF.Length;++i )
                this.ParamsF[i] = src.ParamsF[i];
        }


        /// <summary>
        /// Make a clone of this object.
        /// </summary>
        /// <returns>The created clone object.</returns>
        public override EmitterUserData Clone()
        {
            return new EmitterUserData( this );
        }

        #endregion

        #region Properties

        /// <summary>
        /// The index of the user data.
        /// </summary>
        public int Index { get; set; }


        /// <summary>
        /// ユーザーデータ
        /// </summary>
        public uint Data1
        {
            get
            {
                uint iValue = 0x00000000;

                iValue |= (this.ParamsI[0] & (uint)0x000000FF);
                iValue |= (this.ParamsI[1] & (uint)0x000000FF) << 8;

                uint iMask  = 0x00010000;
                foreach ( bool bFlag in this.BitFlags )
                {
                    if ( bFlag==true )
                        iValue |= iMask;

                    iMask = iMask << 1;
                }

                return iValue;
            }

            set
            {
                this.ParamsI[0] = (value & 0x000000FF);
                this.ParamsI[1] = (value & 0x0000FF00) >> 8;

                uint iMask = 0x00010000;
                for ( int i=0;i<this.BitFlags.Length;++i )
                {
                    this.BitFlags[i] = ((value & iMask)==0x00000000) ? false : true;
                    iMask            = iMask << 1;
                }
            }
        }


        /// <summary>
        /// ユーザーデータ
        /// </summary>
        public uint Data2
        {
            get
            {
                uint iValue = 0x00000000;

                iValue |= (this.ParamsI[2] & (uint)0x000000FF);
                iValue |= (this.ParamsI[3] & (uint)0x000000FF) << 8;
                iValue |= (this.ParamsI[4] & (uint)0x000000FF) << 16;
                iValue |= (this.ParamsI[5] & (uint)0x000000FF) << 24;

                return iValue;
            }

            set
            {
                this.ParamsI[2] = (value & 0x000000FF);
                this.ParamsI[3] = (value & 0x0000FF00) >> 8;
                this.ParamsI[4] = (value & 0x00FF0000) >> 16;
                this.ParamsI[5] = (value & 0xFF000000) >> 24;
            }
        }

        /// <summary>
        /// ユーザーデータフラグ
        /// </summary>
        [System.Xml.Serialization.XmlIgnore]
        public bool[] BitFlags { get; set; }

        /// <summary>
        /// ユーザーデータ整数パラメータ
        /// </summary>
        [System.Xml.Serialization.XmlIgnore]
        public uint[] ParamsI { get; set; }

        /// <summary>
        /// ユーザーデータ浮動小数点数パラメータ
        /// </summary>
        [System.Xml.Serialization.XmlArrayItem("Param")]
        public float[] ParamsF { get; set; }

        #endregion
    }


    /// <summary>
    /// Class for storing default user shader parameters.
    /// </summary>
    public class DefaultEmitterUserData
    {
        #region Construtors

        /// <summary>
        /// Default constructor.
        /// </summary>
        public DefaultEmitterUserData()
        {
            this.BitFlags = new bool[16];
            this.ParamsI  = new uint[6];
            this.ParamsF  = new float[8];

            int i;
            for ( i=0;i<this.BitFlags.Length;++i )
                this.BitFlags[i] = false;

            for ( i=0;i<this.ParamsI.Length;++i )
                this.ParamsI[i] = 0;

            for ( i=0;i<this.ParamsF.Length;++i )
                this.ParamsF[i] = 0.0f;
        }

        #endregion

        #region Properties

        /// <summary>
        /// ユーザーデータフラグ
        /// </summary>
        public bool[] BitFlags { get; set; }

        /// <summary>
        /// ユーザーデータ整数パラメータ
        /// </summary>
        public uint[] ParamsI { get; set; }

        /// <summary>
        /// ユーザーデータ浮動小数点数パラメータ
        /// </summary>
        public float[] ParamsF { get; set; }

        #endregion
    }
}
