﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using NintendoWare.ToolDevelopmentKit;

namespace NWCore.DataModel.Major_0.Minor_9.Build_0.Revision_0
{


    /// <summary>
    /// Class for storing the loaded texture information.
    /// </summary>
    public class TextureInfo
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public TextureInfo()
        {
            Initialize();
        }

        #endregion

        #region Utility

        /// <summary>
        /// Initialize the data.
        /// </summary>
        public void Initialize()
        {
            this.Width              = 0;
            this.Height             = 0;
            this.TileMode           = string.Empty;
            this.Swizzle            = 0;
            this.Alignment          = 0;
            this.Pitch              = 0;
            this.MipLevel           = 0;
            this.CompSel            = 0;
            this.MipOffset          = null;
            this.OriginalDataFormat = string.Empty;
            this.OriginalDataSize   = 0;
            this.OriginalImage      = null;
            this.NativeDataFormat   = string.Empty;
            this.NativeDataSize     = 0;
            this.NativeImage        = null;
        }


        /// <summary>
        /// Set data.
        /// </summary>
        public void Set( TextureInfo info )
        {
            if ( info==null )
                return;

            this.Width              = info.Width;
            this.Height             = info.Height;
            this.Swizzle            = info.Swizzle;
            this.Alignment          = info.Alignment;
            this.Pitch              = info.Pitch;
            this.MipLevel           = info.MipLevel;
            this.CompSel            = info.CompSel;
            this.OriginalDataSize   = info.OriginalDataSize;
            this.NativeDataSize     = info.NativeDataSize;

            if ( info.MipOffset!=null )
            {
                this.MipOffset = new uint[info.MipOffset.Length];
                for ( int i=0;i<info.MipOffset.Length;++i )
                    this.MipOffset[i] = info.MipOffset[i];
            }

            if ( info.OriginalImage!=null )
            {
                this.OriginalImage = new byte[info.OriginalImage.Length];
                for ( int i=0;i<info.OriginalImage.Length;++i )
                    this.OriginalImage[i] = info.OriginalImage[i];
            }

            if ( info.NativeImage!=null )
            {
                this.NativeImage = new byte[info.NativeImage.Length];
                for ( int i=0;i<info.NativeImage.Length;++i )
                    this.NativeImage[i] = info.NativeImage[i];
            }

            this.TileMode           = string.IsNullOrEmpty(info.TileMode)==true ? string.Empty :
                                                                                  string.Copy(info.TileMode);
            this.OriginalDataFormat = string.IsNullOrEmpty(info.OriginalDataFormat)==true ? string.Empty :
                                                                                            string.Copy(info.OriginalDataFormat);
            this.NativeDataFormat   = string.IsNullOrEmpty(info.NativeDataFormat)==true ? string.Empty :
                                                                                          string.Copy(info.NativeDataFormat);
        }

        #endregion

        #region Properties

        /// <summary>Width of the image.</summary>
        public uint   Width { get; set; }

        /// <summary>Height of the image.</summary>
        public uint   Height { get; set; }

        /// <summary>Tile mode.</summary>
        public string TileMode { get; set; }

        /// <summary>Swizzle.</summary>
        public uint   Swizzle { get; set; }

        /// <summary>Alignment.</summary>
        public uint   Alignment { get; set; }

        /// <summary>Pitch of the image.</summary>
        public uint   Pitch { get; set; }

        /// <summary>Mip level of the image.</summary>
        public uint   MipLevel { get; set; }

        /// <summary>Component selector.</summary>
        public uint   CompSel { get; set; }

        /// <summary>Mipmap offset.</summary>
        public uint[] MipOffset { get; set; }


        /// <summary>Original image format.</summary>
        public string OriginalDataFormat { get; set; }

        /// <summary>Original image data size.</summary>
        public int    OriginalDataSize { get; set; }

        /// <summary>Original image data.</summary>
        public byte[] OriginalImage { get; set; }


        /// <summary>Native image format.</summary>
        public string NativeDataFormat { get; set; }

        /// <summary>Native image data size.</summary>
        public int    NativeDataSize { get; set; }

        /// <summary>Native image data.</summary>
        public byte[] NativeImage { get; set; }

        #endregion
    }


    /// <summary>
    /// プロジェクトヘッダー
    /// </summary>
    public sealed class TextureRes : BaseDataModel<TextureRes>
    {
        Bitmap[] m_colorBitmaps;
        Bitmap[] m_alphaBitmaps;

        #region Properties

        /// <summary>
        /// Color Bitmap
        /// </summary>
        public Bitmap[] ColorBitmap
        {
            get { return m_colorBitmaps; }
        }

        /// <summary>
        /// Alpha Bitmap
        /// </summary>
        public Bitmap[] AlphaBitmap
        {
            get { return m_alphaBitmaps; }
        }

        /// <summary>
        /// テクスチャー幅
        /// </summary>
        public int Width
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャー高さ
        /// </summary>
        public int Height
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャーフォーマット
        /// </summary>
        public int Format
        {
            get;
            set;
        }

        /// <summary>
        /// 名前（名前テーブル先頭からのオフセット
        /// </summary>
        public int NumMipLevel
        {
            get;
            set;
        }

        /// <summary>
        /// ラップモード（上位:U 下位：V）
        /// </summary>
        public int WrapMode
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャフィルタ
        /// </summary>
        public FilterMode FilterMode
        {
            get;
            set;
        }

        /// <summary>
        /// データサイズ
        /// </summary>
        public int DataSize
        {
            get;
            set;
        }

        /// <summary>
        /// テクスチャーのデータ位置（テクスチャーテーブルからのオフセット）
        /// </summary>
        public int DataPos
        {
            get;
            set;
        }

        /// <summary>
        /// // テクスチャーのハンドル
        /// </summary>
        public uint Handle
        {
            get;
            set;
        }

        /// <summary>
        /// Texture information for binary converter.
        /// </summary>
        public TextureInfo TextureInfo
        {
            get;
            set;
        }

        #endregion

        #region Construtors

        /// <summary>
        /// デフォルトコンストラクタ
        /// </summary>
        public TextureRes()
        {
            m_colorBitmaps = null;
            m_alphaBitmaps = null;

            this.TextureInfo = new TextureInfo();
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        /// <param name="source">コピー元のベクトル</param>
        public TextureRes(TextureRes source)
            : this()
        {
            this.Set(source);
        }

        /// <summary>
        /// テクスチャーの設定
        /// </summary>
        /// <param name="colorBmp">RGB Bitmap</param>
        /// <param name="alphaBmp">Alpha Bitmap</param>
        public void SetTexture(
            Bitmap[] colorBmps,
            Bitmap[] alphaBmps,
            int mipmapLevel,
            int dataSize,
            int format)
        {
            m_colorBitmaps = colorBmps;
            m_alphaBitmaps = alphaBmps;

            if ( colorBmps==null )
            {
                this.Width       =  0;
                this.Height      =  0;
                this.Format      = -1;
                this.NumMipLevel =  0;
                this.WrapMode    =  0;
                this.DataSize    =  0;
                this.DataPos     =  0;
            }
            else if ( colorBmps.Length>0 )
            {
                this.Width = colorBmps[0].Width;
                this.Height = colorBmps[0].Height;
//TODO:Format はまだ未完成
                this.Format = format;
                this.NumMipLevel = mipmapLevel;
                this.WrapMode = 0;
                this.DataSize = dataSize;
                this.DataPos = 0;
            }
        }

        #endregion

        #region Utility Functions

        /// <summary>
        /// 初期化
        /// </summary>
        public override void Initialize()
        {
            this.TextureInfo.Initialize();
        }

        /// <summary>
        /// ベクトルを設定します。
        /// </summary>
        /// <param name="source">設定するベクトルです。</param>
        public override void Set(TextureRes source)
        {
            Ensure.Argument.NotNull(source);

            this.Width = source.Width;
            this.Height = source.Height;
            this.Format = source.Format;
            this.NumMipLevel = source.NumMipLevel;
            this.WrapMode = source.WrapMode;
            this.DataSize = source.DataSize;
            this.DataPos = source.DataPos;
            this.Handle = source.Handle;

            this.m_colorBitmaps =
                (source.m_colorBitmaps != null) ?
                    (source.m_colorBitmaps.Clone() as Bitmap[]) : null;

            this.m_alphaBitmaps =
                (source.m_alphaBitmaps != null) ?
                    (source.m_alphaBitmaps.Clone() as Bitmap[]) : null;

            this.TextureInfo.Set( source.TextureInfo );
        }

        /// <summary>
        /// オブジェクトを複製します。
        /// </summary>
        /// <returns>複製したオブジェクトです。</returns>
        public override TextureRes Clone()
        {
            return new TextureRes(this);
        }

        #endregion
    }
}
