﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Diagnostics;
using System.Drawing;
using System.Xml.Serialization;

using NintendoWare.ToolDevelopmentKit;
using NintendoWare.ToolDevelopmentKit.Xml;

using NWCore.Serializer;

namespace NWCore.DataModel
{
    /// <summary>
    /// カラーラベルテーブル
    /// </summary>
    public static class ColorLabelTable
    {
        static RgbaColor CreateColor( byte r,
                                      byte g,
                                      byte b )
        {
            return new RgbaColor( ((float)r / 255.0f),
                                  ((float)g / 255.0f),
                                  ((float)b / 255.0f),
                                  1.0f );
        }

        /// <summary>
        /// カラーテーブル
        /// </summary>
        public static readonly RgbaColor[] ColorTable = new RgbaColor[]
        {
            /*
             * http://www-sdd.zelda.nintendo.co.jp/project/nintendoware3/kagemai/html/user.cgi?project=nw3_eft&action=view_report&id=274#6
                無　= ラベルカラーなし
                赤 = 232,196,190
                橙 = 242,215,174
                黄 = 242,240,174
                緑 = 209,226,180
                青 = 208,219,236
                紫 = 219,210,229
                灰 = 224,224,224
             */
            CreateColor(255,255,255),
            CreateColor(232,196,190),
            CreateColor(242,215,174),
            CreateColor(242,240,174),
            CreateColor(209,226,180),
            CreateColor(208,219,236),
            CreateColor(219,210,229),
            CreateColor(224,224,224),
        };

        /// <summary>
        /// テーブルの要素数
        /// </summary>
        public static int Count
        {
            get { return ColorTable.Length; }
        }

        /// <summary>
        /// デフォルトのカラーインデックス
        /// </summary>
        public const int DefaultColorIndex = 0;

        /// <summary>
        /// デフォルトのカラー
        /// </summary>
        public static RgbaColor DefaultColor
        {
            get { return ColorTable[DefaultColorIndex]; }
        }

        /// <summary>
        /// カラーインデックス → カラーデータ
        /// </summary>
        /// <remarks>
        /// カラーテーブルを超えるインデックスを指定した場合は全てDefaultColorになります。
        /// </remarks>
        public static RgbaColor GetColor(int iColorIndex)
        {
            if ( iColorIndex>=0 &&
                 iColorIndex<ColorLabelTable.Count )
            {
                return ColorTable[iColorIndex];
            }

            return DefaultColor;
        }

        /// <summary>
        /// カラーインデックス → カラーデータ
        /// </summary>
        /// <remarks>
        /// カラーテーブルを超えるインデックスを指定した場合は全てDefaultColorになります。
        /// </remarks>
        public static Color GetIntColor( int iColorIndex )
        {
            if ( iColorIndex>=0 &&
                 iColorIndex<ColorLabelTable.Count )
            {
                return ConvertColor( ColorTable[iColorIndex] );
            }

            return ConvertColor( DefaultColor );
        }

        /// <summary>
        /// Clamp the given value to the specified range.
        /// </summary>
        /// <param name="value">The value.</param>
        /// <param name="min">The minimum value.</param>
        /// <param name="max">The maximum value.</param>
        /// <returns>The clamped value.</returns>
        public static float Clamp( float value,
                                   float min,
                                   float max )
        {
            return System.Math.Max( System.Math.Min( value, max ), min );
        }

        /// <summary>
        /// Convert NintendoWare.ToolDevelopmentKit.RgbaColor to System.Drawing.Color.
        /// </summary>
        /// <param name="src">The source color to convert from.</param>
        /// <returns>The converted color format.</returns>
        public static Color ConvertColor( RgbaColor src )
        {
            return Color.FromArgb( (byte)(Clamp(src.A, 0.0f, 1.0f) * 255.0f),
                                   (byte)(Clamp(src.R, 0.0f, 1.0f) * 255.0f),
                                   (byte)(Clamp(src.G, 0.0f, 1.0f) * 255.0f),
                                   (byte)(Clamp(src.B, 0.0f, 1.0f) * 255.0f) );
        }
    }
}

