﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;

namespace App.Utility
{
    /// <summary>
    /// Contains threading extension and helper methods.
    /// </summary>
    public static class ThreadingExtensions
    {
        /// <summary>
        /// Produce a value on a given thread and returns it back to the caller thread.
        /// </summary>
        /// <typeparam name="T">Type of the value to be produced.</typeparam>
        /// <param name="context">SyncronizationContext of the UI thread.</param>
        /// <param name="producer">The method that produce the value.</param>
        /// <returns>Returns a value produced on the UI thread.</returns>
        public static T SafelyProduce<T>(this SynchronizationContext context, Func<T> producer)
        {
            if (context == null)
                throw new ArgumentNullException("context");
            if (producer == null)
                throw new ArgumentNullException("producer");

            var localValue = default(T);

            context.Send(_ => localValue = producer(), null);

            return localValue;
        }

        /// <summary>
        /// Run a method synchronously on a different thread.
        /// </summary>
        /// <param name="context">The synchronization context of the thread on which
        /// to run the method.</param>
        /// <param name="action">Method to run on the different thread.</param>
        public static void Send(this SynchronizationContext context, Action action)
        {
            if (context == null)
                throw new ArgumentNullException("context");
            if (action == null)
                throw new ArgumentNullException("action");

            context.Send(_ => action(), null);
        }

        /// <summary>
        /// Run a method asynchronously on a different thread.
        /// </summary>
        /// <param name="context">The synchronization context of the thread on which
        /// to run the method.</param>
        /// <param name="action">Method to run on the different thread.</param>
        public static void Post(this SynchronizationContext context, Action action)
        {
            if (context == null)
                throw new ArgumentNullException("context");
            if (action == null)
                throw new ArgumentNullException("action");

            context.Post(_ => action(), null);
        }

        /// <summary>
        /// Run a method on a thread pool thread.
        /// </summary>
        /// <param name="action">Method to run.</param>
        /// <returns>Returns true if the task could be run, false otherwise.</returns>
        public static bool FireAndForget(Action action)
        {
            if (action == null)
                throw new ArgumentNullException("action");

            return ThreadPool.QueueUserWorkItem(_ => action());
        }

        /// <summary>
        /// Run a method on a thread pool thread.
        /// </summary>
        /// <typeparam name="T">Type of the parameter to pass to the threaded method.</typeparam>
        /// <param name="action">Method to run on a different thread.</param>
        /// <param name="state">Parameter to provide to the method.</param>
        /// <returns>Returns true if the task could be run, false otherwise.</returns>
        public static bool FireAndForget<T>(Action<T> action, T state)
        {
            if (action == null)
                throw new ArgumentNullException("action");

            return ThreadPool.QueueUserWorkItem(st => action((T)st), state);
        }
    }
}
