﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using App.Data;
using NW4F.DataModel;

namespace App.Utility
{
    /// <summary>
    /// テクスチャー・ユーティリティ
    /// </summary>
    public static class TextureUtility
    {
        /// <summary>
        /// デコーダーです。
        /// </summary>
        public static ITextureDecoder Decoder = new TextureCtrDecoderEM4F();

        /// <summary>
        /// テクスチャーミップマップサイズを取得
        /// </summary>
        /// <param name="texture">ITextureオブジェクトです</param>
        /// <returns>ミップマップサイズ</returns>
        public static int GetMipmapSize(ITexture texture)
        {
            if (texture != null)
            {
                IPixelBasedTexture pixelBase = texture as IPixelBasedTexture;
                return (pixelBase != null ? pixelBase.MipmapSize : 0);
            }
            return 0;
        }

        /// <summary>
        /// テクスチャーミップマップサイズの文字列情報を取得
        /// </summary>
        /// <param name="texture">ITextureオブジェクトです</param>
        /// <returns>ミップマップサイズ文字列情報</returns>
        public static string GetMipmapSizeInfo(ITexture texture)
        {
            if (texture != null)
            {
                IPixelBasedTexture pixelBase = texture as IPixelBasedTexture;
                return (pixelBase != null ? pixelBase.MipmapSize.ToString() : String.Empty);
            }
            return String.Empty;
        }

        /// <summary>
        /// テクスチャーサイズの文字列情報を取得
        /// </summary>
        /// <param name="texture">ITextureオブジェクトです</param>
        /// <returns>サイズ文字列情報</returns>
        public static string GetSizeInfo(ITexture texture)
        {
            if (texture != null)
            {
                Size size = TextureUtility.Decoder.GetSize(texture);
                return String.Format(res.Strings.TextureSizeStringFormat, size.Width, size.Height);
            }
            return String.Empty;
        }

        /// <summary>
        /// テクスチャの実体を取得します。
        /// </summary>
        /// <returns>テクスチャの実体を返します。</returns>
        public static ITexture GetTextureEntity(ITexture texture)
        {
            if (texture == null)
            {
                return null;
            }

            IReferenceTexture referenceTexture = texture as IReferenceTexture;
            return referenceTexture != null ? referenceTexture.Target : texture;
        }

        /// <summary>
        /// テクスチャーサイズ情報の取得
        /// </summary>
        /// <param name="texture">ITextureオブジェクトです</param>
        /// <returns>サイズ情報の文字列</returns>
        public static string GetTextureDataSizeInfo(ITexture texture)
        {
            if (texture != null)
            {
                int size = Decoder.GetDataSize(texture);
                float detail = 0.0f;
                int modVal = 0;
                string unitStr = String.Empty;
                if (size >= 0x100000)
                {
                    detail = (float)size / (float)0x100000;
                    modVal = size % 0x100000;
                    unitStr = "MB";
                }
                else if (size > 0x400)
                {
                    detail = (float)size / (float)0x400;
                    modVal = size % 0x400;
                    unitStr = "KB";
                }
                else
                {
                    return String.Format("{0} Byte", size);
                }

                return String.Format(
                    "{0} {1}",
                    (modVal != 0 ? detail.ToString("F1") : detail.ToString("F0")),
                    unitStr);
            }

            return String.Empty;
        }
    }
}
