﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace App.Utility
{
    #region Enumerator for angle types

    /// <summary>
    /// 角度表現のタイプです
    /// </summary>
    public enum AngleTypes
    {
        /// <summary>Radian</summary>
        Radian,
        /// <summary>Degree</summary>
        Degree
    }

    #endregion

    /// <summary>
    /// 数学計算のユーティリティです。
    /// </summary>
    public static class MathUtility
    {
        /// <summary>
        /// 補間計算。
        /// </summary>
        /// <param name="min1">最小値１</param>
        /// <param name="max1">最大値１</param>
        /// <param name="val1">現在の値</param>
        /// <param name="min2">最小値２</param>
        /// <param name="max2">最大値２</param>
        /// <returns>補間計算値</returns>
        public static float Interpolate(float min1, float max1, float val1, float min2, float max2)
        {
            return (max2 - min2) * ((val1 - min1) / (max1 - min1)) + min2;
        }

        /// <summary>
        /// 入れ替え。
        /// </summary>
        /// <param name="val1">入れ替えるワーク１</param>
        /// <param name="val2">入れ替えるワーク２</param>
        public static void Swap(ref int val1, ref int val2)
        {
            int temp = val1;
            val1 = val2;
            val2 = temp;
        }

        /// <summary>
        /// 入れ替え。
        /// </summary>
        /// <param name="val1">入れ替えるワーク１</param>
        /// <param name="val2">入れ替えるワーク２</param>
        public static void Swap(ref float val1, ref float val2)
        {
            float temp = val1;
            val1 = val2;
            val2 = temp;
        }

        /// <summary>
        /// 切り詰め
        /// </summary>
        /// <param name="val">現在の値</param>
        /// <param name="min">最小値</param>
        /// <param name="max">最大値</param>
        public static long Clamp( long val, long min, long max )
        {
            if (val < min)
            {
                val = min;
            }
            else if (val > max)
            {
                val = max;
            }

            return val;
        }

        /// <summary>
        /// 切り詰め
        /// </summary>
        /// <param name="val">現在の値</param>
        /// <param name="min">最小値</param>
        /// <param name="max">最大値</param>
        public static ulong Clamp( ulong val, ulong min, ulong max )
        {
            if ( val < min )
            {
                val = min;
            }
            else if ( val > max )
            {
                val = max;
            }

            return val;
        }

        /// <summary>
        /// 切り詰め
        /// </summary>
        /// <param name="val">現在の値</param>
        /// <param name="min">最小値</param>
        /// <param name="max">最大値</param>
        public static int Clamp(int val, int min, int max)
        {
            if (val < min)
            {
                val = min;
            }
            else if (val > max)
            {
                val = max;
            }

            return val;
        }

        /// <summary>
        /// 切り詰め
        /// </summary>
        /// <param name="val">現在の値</param>
        /// <param name="min">最小値</param>
        /// <param name="max">最大値</param>
        public static uint Clamp( uint val, uint min, uint max )
        {
            if ( val < min )
            {
                val = min;
            }
            else if ( val > max )
            {
                val = max;
            }

            return val;
        }

        /// <summary>
        /// 切り詰め
        /// </summary>
        /// <param name="val">現在の値</param>
        /// <param name="min">最小値</param>
        /// <param name="max">最大値</param>
        public static float Clamp( float val, float min, float max )
        {
            if ( val < min )
            {
                val = min;
            }
            else if ( val > max )
            {
                val = max;
            }

            return val;
        }

        /// <summary>
        /// 切り詰め
        /// </summary>
        /// <param name="val">現在の値</param>
        /// <param name="min">最小値</param>
        /// <param name="max">最大値</param>
        public static double Clamp( double val, double min, double max )
        {
            if ( val < min )
            {
                val = min;
            }
            else if ( val > max )
            {
                val = max;
            }

            return val;
        }

        /// <summary>
        /// 2のべき乗か？
        /// </summary>
        /// <param name="number">判定する値</param>
        /// <returns>=true..YES</returns>
        public static bool IsPowerOf2(uint number)
        {
            return (number & (number - 1)) == 0;
        }

        /// <summary>
        /// Rotate Vector3 : Radian -> Degree
        /// </summary>
        /// <param name="radianVec">Radian Vector</param>
        /// <returns>Degree Vector</returns>
        public static NintendoWare.ToolDevelopmentKit.Vector3 ToDegreeVector(
            NintendoWare.ToolDevelopmentKit.Vector3 radianVec)
        {
            return new NintendoWare.ToolDevelopmentKit.Vector3(
                NintendoWare.ToolDevelopmentKit.MathUtility.ConvertRadianToDegrees(radianVec.X),
                NintendoWare.ToolDevelopmentKit.MathUtility.ConvertRadianToDegrees(radianVec.Y),
                NintendoWare.ToolDevelopmentKit.MathUtility.ConvertRadianToDegrees(radianVec.Z)
            );
        }

        /// <summary>
        /// Rotate Vector3 : Degree -> Radian
        /// </summary>
        /// <param name="degreeVec">Degree Vector</param>
        /// <returns>Radian Vector</returns>
        public static NintendoWare.ToolDevelopmentKit.Vector3 ToRadianVector(
            NintendoWare.ToolDevelopmentKit.Vector3 degreeVec)
        {
            return new NintendoWare.ToolDevelopmentKit.Vector3(
                NintendoWare.ToolDevelopmentKit.MathUtility.ConvertDegreeToRadians(degreeVec.X),
                NintendoWare.ToolDevelopmentKit.MathUtility.ConvertDegreeToRadians(degreeVec.Y),
                NintendoWare.ToolDevelopmentKit.MathUtility.ConvertDegreeToRadians(degreeVec.Z)
            );
        }


        /// <summary>
        /// Utility method for finding the most significant digit of the given value.
        /// </summary>
        /// <param name="fValue">The value.</param>
        /// <returns>Number of digits after decimal point.</returns>
        public static int FindMostSignificantDigit( float fValue )
        {
            int    iNumDecimalDigits = 0;
            double fOrigValue        = (double)Math.Abs( fValue );
            double fCurrValue        = fOrigValue;

            if ( fCurrValue<1.0 &&
                 fCurrValue>0.0 )
            {
                while ( Math.Round(fCurrValue, 6)<1.0 )
                {
                    fCurrValue = fOrigValue * (double)PowerOfTenF(iNumDecimalDigits+1);
                    ++iNumDecimalDigits;
                }
            }

            return iNumDecimalDigits;
        }


        /// <summary>
        /// Utility method for finding the least significant digit of the given value.
        /// </summary>
        /// <param name="fValue">The value.</param>
        /// <returns>Number of digits after decimal point.</returns>
        public static int FindLeastSignificantDigit( float fValue )
        {
            int    iNumDecimalDigits = 0;
            double fCurrValue        = (double)Math.Abs( fValue );
            double fPower            = 10.0;

            if ( fCurrValue==0.0f )
                return 0;

            // First find the most significant digit.
            int iDigit = 0;
            while ( iDigit==0 )
            {
                fPower = Math.Pow( 10.0, iNumDecimalDigits );
                iDigit = (int)Math.Truncate( fCurrValue * fPower );
                if ( iDigit>0 )
                    break;

                ++iNumDecimalDigits;
            }

            // Start looking for the least significant digit.
            int    iMaxNumDigits = 8 - (int)Math.Truncate(Math.Log10(fCurrValue));
            double fThreshold    = 0.001;
            while ( iNumDecimalDigits<iMaxNumDigits )
            {
                fPower = Math.Pow( 10.0, iNumDecimalDigits );

                double fCompare = fCurrValue * fPower;
                if ( Math.Abs(fCompare - Math.Round(fCompare))<fThreshold )
                    break;

                ++iNumDecimalDigits;
            }

            return iNumDecimalDigits;
        }


        /// <summary>
        /// Computer the given power of 10 as floating point number.
        /// The power can be ranged from -7 to 7.
        /// </summary>
        /// <param name="iPower">The power.</param>
        /// <returns>The result.</returns>
        public static float PowerOfTenF( int iPower )
        {
            System.Diagnostics.Debug.Assert( iPower>=s_iMinPowerOfTenF &&
                                             iPower<=s_iMaxPowerOfTenF );

            return s_powerOfTenF[ iPower - s_iMinPowerOfTenF ];
        }


        /// <summary>
        /// Computer the given power of 10 as integer number.
        /// The power can be ranged from 0 to 9.
        /// </summary>
        /// <param name="iPower">The power.</param>
        /// <returns>The result.</returns>
        public static int PowerOfTenI( int iPower )
        {
            System.Diagnostics.Debug.Assert( iPower>=s_iMinPowerOfTenI &&
                                             iPower<=s_iMaxPowerOfTenI );

            return s_powerOfTenI[ iPower - s_iMinPowerOfTenI ];
        }

        #region Table for power of ten

        private const int s_iMinPowerOfTenF = -7;
        private const int s_iMaxPowerOfTenF = 7;
        private static float[] s_powerOfTenF = new float[] { 0.0000001f,
                                                             0.000001f,
                                                             0.00001f,
                                                             0.0001f,
                                                             0.001f,
                                                             0.01f,
                                                             0.1f,
                                                             1.0f,
                                                             10.0f,
                                                             100.0f,
                                                             1000.0f,
                                                             10000.0f,
                                                             100000.0f,
                                                             1000000.0f,
                                                             10000000.0f };

        private const int s_iMinPowerOfTenI = 0;
        private const int s_iMaxPowerOfTenI = 9;
        private static int[] s_powerOfTenI = new int[] { 1,
                                                         10,
                                                         100,
                                                         1000,
                                                         10000,
                                                         100000,
                                                         1000000,
                                                         10000000,
                                                         100000000,
                                                         1000000000 };

        #endregion
    }
}
