﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using NintendoWare.ToolDevelopmentKit;

namespace App.Utility
{
    /// <summary>
    /// Contains graphics extension methods.
    /// </summary>
    public static class GraphicsExtensions
    {
        /// <summary>
        /// Draw lines, excludes duplicate points to prevent exceptions.
        /// </summary>
        /// <param name="g">The graphics object.</param>
        /// <param name="pen">The pen to draw the lines.</param>
        /// <param name="points">The points of the lines.</param>
        /// <param name="renderArea">The render area, anything off the area will be clipped.</param>
        public static void DrawLinesExcludeDuplicatePoints(
            this Graphics g,
            Pen pen,
            Point[] points,
            Rectangle renderArea)
        {
            if (g == null)
            {
                return;
            }

            if ((points == null) || (points.Length < 2))
            {
                return;
            }

            for (int i = 0; i < (points.Length - 1); ++i)
            {
                if (points[i].Equals(points[i + 1]) == true)
                {
                    continue;
                }

                var p1 = points[i];
                var p2 = points[i + 1];

                if ((p1.X < renderArea.Left && p2.X < renderArea.Left) ||
                    (p1.X > renderArea.Right && p2.X > renderArea.Right) ||
                    (p1.Y < renderArea.Top && p2.Y < renderArea.Top) ||
                    (p1.Y > renderArea.Bottom && p2.Y > renderArea.Bottom))
                {
                    // The line segment is outside of the rectangle and does not intersect with it.
                    continue;
                }
                else if ((p1.X >= renderArea.Left && p1.X <= renderArea.Right) &&
                         (p2.X >= renderArea.Left && p2.X <= renderArea.Right) &&
                         (p1.Y >= renderArea.Top && p1.Y <= renderArea.Bottom) &&
                         (p2.Y >= renderArea.Top && p2.Y <= renderArea.Bottom))
                {
                    // The line segment is inside of the rectangle, just render it.
                    g.DrawLine(pen, p1, p2);
                    continue;
                }

                if ((p1.X == p2.X) && (p1.X >= renderArea.Left) && (p1.X <= renderArea.Right))
                {
                    // It's a vertical line segment.
                    float minY = Math.Max(renderArea.Top, Math.Min(p1.Y, p2.Y));
                    float maxY = Math.Min(renderArea.Bottom, Math.Max(p1.Y, p2.Y));

                    g.DrawLine(pen, p1.X, minY, p1.X, maxY);
                    continue;
                }
                else if ((p1.Y == p2.Y) && (p1.Y >= renderArea.Top) && (p1.Y <= renderArea.Bottom))
                {
                    // It's a horizontal line segment.
                    float minX = Math.Max(renderArea.Left, Math.Min(p1.X, p2.X));
                    float maxX = Math.Min(renderArea.Right, Math.Max(p1.X, p2.X));

                    g.DrawLine(pen, minX, p1.Y, maxX, p1.Y);
                    continue;
                }

                // The line segment is either vertical nor horizontal, compute the slope.
                // Slope m = (y1 - y2) / (x1 - x2)
                float m = (float)(p1.Y - p2.Y) / (float)(p1.X - p2.X);

                // Line equation : y = mx + b, b = y - mx
                float b = (float)p1.Y - (m * (float)p1.X);

                // Find the intersection on each side of the clipping rectangle.
                float intersectTop    = ((float)renderArea.Top - b) / m;
                float intersectBottom = ((float)renderArea.Bottom - b) / m;
                float intersectLeft   = ((float)renderArea.Left * m) + b;
                float intersectRight  = ((float)renderArea.Right * m) + b;

                if ((intersectTop < renderArea.Left && intersectTop > renderArea.Right) &&
                    (intersectBottom < renderArea.Left && intersectBottom > renderArea.Right) &&
                    (intersectRight < renderArea.Top && intersectRight > renderArea.Bottom) &&
                    (intersectLeft < renderArea.Top && intersectLeft > renderArea.Bottom))
                {
                    // The line segment doesn't intersect with the view rectangle.
                    continue;
                }

                PointF fp1 = p1.X < p2.X ? new PointF(p1.X, p1.Y) : new PointF(p2.X, p2.Y);
                PointF fp2 = p1.X > p2.X ? new PointF(p1.X, p1.Y) : new PointF(p2.X, p2.Y);

                if (m > 0)
                {
                    fp1 = fp1.X > intersectTop ? fp1 : new PointF(intersectTop, (float)renderArea.Top);
                    fp1 = fp1.X > (float)renderArea.Left ? fp1 : new PointF((float)renderArea.Left, intersectLeft);

                    fp2 = fp2.X < intersectBottom ? fp2 : new PointF(intersectBottom, (float)renderArea.Bottom);
                    fp2 = fp2.X < (float)renderArea.Right ? fp2 : new PointF((float)renderArea.Right, intersectRight);
                }
                else
                {
                    fp1 = fp1.X > intersectBottom ? fp1 : new PointF(intersectBottom, (float)renderArea.Bottom);
                    fp1 = fp1.X > (float)renderArea.Left ? fp1 : new PointF((float)renderArea.Left, intersectLeft);

                    fp2 = fp2.X < intersectTop ? fp2 : new PointF(intersectTop, (float)renderArea.Top);
                    fp2 = fp2.X < (float)renderArea.Right ? fp2 : new PointF((float)renderArea.Right, intersectRight);
                }

                g.DrawLine(pen, fp1, fp2);
            }
        }


        /// <summary>
        /// Draw lines, excludes duplicate points to prevent exceptions.
        /// </summary>
        /// <param name="g">The graphics object.</param>
        /// <param name="pen">The pen to draw the lines.</param>
        /// <param name="points">The points of the lines.</param>
        public static void DrawLinesExcludeDuplicatePoints(
            this Graphics g,
            Pen pen,
            Point[] points)
        {
            if (g == null)
            {
                return;
            }

            if ((points == null) || (points.Length < 2))
            {
                return;
            }

            for (int i = 0; i < (points.Length - 1); ++i)
            {
                if (points[i].Equals(points[i + 1]) == false)
                {
                    g.DrawLine(pen, points[i], points[i + 1]);
                }
            }
        }
    }

    /// <summary>
    /// 描画ユーティリティクラス。
    /// </summary>
    public static class GraphicsUtility
    {
        /// <summary>
        /// Test if line segment A and B intersect with each other.
        /// </summary>
        /// <param name="a1">A vertex of line segment A.</param>
        /// <param name="a2">A vertex of line segment A.</param>
        /// <param name="b1">A vertex of line segment B.</param>
        /// <param name="b2">A vertex of line segment B.</param>
        /// <param name="intersection">The point of intersection if any.</param>
        /// <returns>True if the line segments intersect with each other.</returns>
        public static bool Intersects(
            PointF a1,
            PointF a2,
            PointF b1,
            PointF b2,
            out PointF intersection)
        {
            intersection = PointF.Empty;

            PointF b = new PointF(a2.X - a1.X, a2.Y - a1.Y);
            PointF d = new PointF(b2.X - b1.X, b2.Y - b1.Y);
            float bDotDPerp = b.X * d.Y - b.Y * d.X;

            // if b dot d == 0, it means the lines are parallel so have infinite intersection points
            if (bDotDPerp == 0)
                return false;

            PointF c = new PointF(b1.X - a1.X, b1.Y - a1.Y);
            float t = (c.X * d.Y - c.Y * d.X) / bDotDPerp;
            if (t < 0 || t > 1)
                return false;

            float u = (c.X * b.Y - c.Y * b.X) / bDotDPerp;
            if (u < 0 || u > 1)
                return false;

            intersection.X = a1.X + (t * b.X);
            intersection.Y = a1.Y + (t * b.Y);

            return true;
        }

        //---------------------------------------------------------------------
        // 四角形
        //---------------------------------------------------------------------
        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawRectangle(Graphics g, Pen pen, Rectangle rect)
        {
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawRectangle(Graphics g, Pen pen, RectangleF rect)
        {
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawCrossRectangle(Graphics g, Pen pen, Rectangle rect)
        {
            g.DrawLine(pen, rect.X, rect.Y, rect.Right - 1, rect.Bottom - 1);
            g.DrawLine(pen, rect.X, rect.Bottom - 1, rect.Right - 1, rect.Y);
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// Fill rounded rectangle.
        /// </summary>
        /// <param name="g">Graphics object.</param>
        /// <param name="brush">The brush to fill the color.</param>
        /// <param name="rect">The area of the rectangle.</param>
        /// <param name="fCornerRadius">The radius of the corners.</param>
        public static void FillRoundedRectangle( Graphics g,
                                                 Brush brush,
                                                 RectangleF rect,
                                                 float fCornerRadius )
        {
            GraphicsPath path = new GraphicsPath();

            // Create the path
            path.AddArc( rect.Left,
                         rect.Top,
                         fCornerRadius,
                         fCornerRadius,
                         180.0f,
                         90.0f );
            path.AddArc( rect.Right-fCornerRadius,
                         rect.Top,
                         fCornerRadius,
                         fCornerRadius,
                         270.0f,
                         90.0f );
            path.AddArc( rect.Right-fCornerRadius,
                         rect.Bottom-fCornerRadius,
                         fCornerRadius,
                         fCornerRadius,
                         0.0f,
                         90.0f );
            path.AddArc( rect.Left,
                         rect.Bottom-fCornerRadius,
                         fCornerRadius,
                         fCornerRadius,
                         90.0f,
                         90.0f );
            path.CloseAllFigures();

            // Render the background for texture path
            g.FillPath( brush, path );
        }

        /// <summary>
        /// Draw an outline of a rounded rectangle.
        /// </summary>
        /// <param name="g">Graphics object.</param>
        /// <param name="pen">The pen to draw with.</param>
        /// <param name="rect">The area of the rectangle.</param>
        /// <param name="fCornerRadius">The radius of the corners.</param>
        public static void DrawRoundedRectangle( Graphics g,
                                                 Pen pen,
                                                 RectangleF rect,
                                                 float fCornerRadius )
        {
            GraphicsPath path = new GraphicsPath();

            // Create the path
            path.AddArc( rect.Left,
                         rect.Top,
                         fCornerRadius,
                         fCornerRadius,
                         180.0f,
                         90.0f );
            path.AddArc( rect.Right-fCornerRadius,
                         rect.Top,
                         fCornerRadius,
                         fCornerRadius,
                         270.0f,
                         90.0f );
            path.AddArc( rect.Right-fCornerRadius,
                         rect.Bottom-fCornerRadius,
                         fCornerRadius,
                         fCornerRadius,
                         0.0f,
                         90.0f );
            path.AddArc( rect.Left,
                         rect.Bottom-fCornerRadius,
                         fCornerRadius,
                         fCornerRadius,
                         90.0f,
                         90.0f );
            path.CloseAllFigures();

            // Draw the outline of the path.
            g.DrawPath( pen, path );
        }

        /// <summary>
        /// 四角形を描画。
        /// </summary>
        public static void DrawCrossRectangle(Graphics g, Pen pen, RectangleF rect)
        {
            g.DrawLine(pen, rect.X, rect.Y, rect.Right - 1, rect.Bottom - 1);
            g.DrawLine(pen, rect.X, rect.Bottom - 1, rect.Right - 1, rect.Y);
            g.DrawRectangle(pen, rect.X, rect.Y, rect.Width - 1, rect.Height - 1);
        }

        /// <summary>
        /// 丸みのある四角形を描画。
        /// </summary>
        public static void DrawRoundishRectangle(Graphics g, Pen pen, RectangleF rect)
        {
            rect.Width -= 1;
            rect.Height -= 1;

            g.DrawLine(pen, rect.Left + 1, rect.Top, rect.Right - 1, rect.Top);
            g.DrawLine(pen, rect.Right, rect.Top + 1, rect.Right, rect.Bottom - 1);
            g.DrawLine(pen, rect.Left, rect.Top + 1, rect.Left, rect.Bottom - 1);
            g.DrawLine(pen, rect.Left + 1, rect.Bottom, rect.Right - 1, rect.Bottom);
        }

        /// <summary>
        /// ３Ｄ四角形を描画。
        /// </summary>
        public static void Draw3DRectangle(Graphics g, Pen penLT, Pen penRB, RectangleF rect)
        {
            rect.Width -= 1;
            rect.Height -= 1;

            g.DrawLine(penLT, rect.Left, rect.Top, rect.Right - 1, rect.Top);
            g.DrawLine(penLT, rect.Left, rect.Top, rect.Left, rect.Bottom - 1);
            g.DrawLine(penRB, rect.Right, rect.Top, rect.Right, rect.Bottom);
            g.DrawLine(penRB, rect.Left, rect.Bottom, rect.Right, rect.Bottom);
        }

        /// <summary>
        /// フォーカス四角形を描画。
        /// </summary>
        public static void DrawFocusRectangle(Graphics g, Rectangle rect)
        {
            IntPtr hdc = g.GetHdc();
            Win32.RECT rc = RectangleUtility.ToRECT(rect);
            Win32.User32.DrawFocusRect(hdc, ref rc);
            g.ReleaseHdc(hdc);
        }

        //---------------------------------------------------------------------
        // テキスト
        //---------------------------------------------------------------------
        /// <summary>
        /// 矩形領域にテキストを描画。
        /// </summary>
        public static void DrawText(Graphics g, string text, Font font, Color color, Rectangle rect, HorizontalAlignment alignment)
        {
            DrawText(g, text, font, color, rect, alignment, null);
        }

        /// <summary>
        /// 矩形領域にテキスト（＋イメージ）を描画。
        /// </summary>
        public static void DrawText(Graphics g, string text, Font font, Color color, Rectangle rect, HorizontalAlignment alignment, Image image)
        {
            // イメージ
            if (image != null)
            {
                Rectangle rcImage = rect;
                rcImage.X += 2;
                rcImage.Y += (rect.Height - image.Height) / 2;
                rcImage.Size = image.Size;
                g.DrawImageUnscaled(image, rcImage);

                // テキスト描画用にオフセット
                RectangleUtility.OffsetLeft(ref rect, rcImage.Right - rect.Left + 1);
            }

            // テキスト
            using (StringFormat sf = new StringFormat())
            {
                sf.LineAlignment = StringAlignment.Center;
                sf.Trimming = StringTrimming.None;
                sf.FormatFlags |= StringFormatFlags.NoWrap;

                switch (alignment)
                {
                    case HorizontalAlignment.Left: sf.Alignment = StringAlignment.Near; break;
                    case HorizontalAlignment.Right: sf.Alignment = StringAlignment.Far; break;
                    case HorizontalAlignment.Center: sf.Alignment = StringAlignment.Center; break;
                    default: break;
                }

                using (Brush brush = new SolidBrush(color))
                {
                    g.DrawString(text, font, brush, rect, sf);
                }
            }
        }

        //---------------------------------------------------------------------
        // カラー
        //---------------------------------------------------------------------
        /// <summary>
        /// カラーボックスを描画。
        /// </summary>
        public static void DrawColorBox(Graphics g, Rectangle bounds, Color color, bool enableAlpha)
        {
            // 枠線色
            Color lineColor = Color.Black;
            if (color.R == 0 && color.G == 0 && color.B == 0)
            {
                lineColor = Color.DimGray;
            }

            // カラー部
            using (Brush brush = new SolidBrush(Color.FromArgb(255, color)))
            {
                g.FillRectangle(brush, bounds);
            }

            // アルファ部
            if (enableAlpha)
            {
                Rectangle rcAlpha = bounds;
                RectangleUtility.OffsetLeft(ref rcAlpha, bounds.Width / 4 * 3);
                using (Brush brush = new SolidBrush(Color.FromArgb(color.A, color.A, color.A)))
                {
                    g.FillRectangle(brush, rcAlpha);
                }

                // 罫線
                if (color.A == 0)
                {
                    lineColor = Color.DimGray;
                }
                using (Pen pen = new Pen(lineColor))
                {
                    g.DrawLine(pen, rcAlpha.Left, rcAlpha.Top, rcAlpha.Left, rcAlpha.Bottom - 1);
                }
            }

            // 枠線
            using (Pen pen = new Pen(lineColor))
            {
                GraphicsUtility.DrawRectangle(g, pen, bounds);
            }
        }

        //---------------------------------------------------------------------
        // 特殊項目
        //---------------------------------------------------------------------
        /// <summary>
        /// サイズ変更グリップを描画。
        /// </summary>
        public static void DrawSizeGrip(Graphics g, Control control)
        {
            // Form.OnPaint()で実装されているグリップ描画と同じ方法
            Size szClient = control.ClientSize;
            ControlPaint.DrawSizeGrip(g, control.BackColor, szClient.Width - 0x10, szClient.Height - 0x10, 0x10, 0x10);
        }


        /// <summary>
        /// Copy the bitmap.
        /// </summary>
        /// <param name="src">The source bitmap.</param>
        /// <param name="dst">The destination bitmap.</param>
        /// <returns>True on success.</returns>
        public static bool CopyBitmap( Bitmap src,
                                       Bitmap dst )
        {
            if ( src==null || dst==null )
                return false;

            // Need to be in the same size.
            if ( src.Width!=dst.Width || src.Height!=dst.Height )
                return false;

            // Only these 2 formats are supported for now.
            if ( src.PixelFormat!=PixelFormat.Format24bppRgb &&
                 src.PixelFormat!=PixelFormat.Format32bppArgb )
            {
                return false;
            }

            // Only these 2 formats are supported for now.
            if ( dst.PixelFormat!=PixelFormat.Format24bppRgb &&
                 dst.PixelFormat!=PixelFormat.Format32bppArgb )
            {
                return false;
            }

            int iSrcBpp = Image.GetPixelFormatSize( src.PixelFormat ) / 8;
            int iDstBpp = Image.GetPixelFormatSize( dst.PixelFormat ) / 8;

            unsafe
            {
                BitmapData srcData = src.LockBits( new Rectangle( 0, 0, src.Width, src.Height ),
                                                   ImageLockMode.ReadOnly,
                                                   src.PixelFormat );

                BitmapData dstData = dst.LockBits( new Rectangle( 0, 0, dst.Width, dst.Height ),
                                                   ImageLockMode.ReadWrite,
                                                   dst.PixelFormat );

                byte* srcBuf = (byte*)(void*)srcData.Scan0;
                byte* dstBuf = (byte*)(void*)dstData.Scan0;

                int iSrcExtraStrideBytes = srcData.Stride - (srcData.Width * iSrcBpp);
                int iDstExtraStrideBytes = dstData.Stride - (dstData.Width * iDstBpp);

                int iSrcIndex = 0;
                int iDstIndex = 0;

                byte r, g, b, a;

                int i, j;
                for ( i=0;i<srcData.Height;++i )
                {
                    for ( j=0;j<srcData.Width;++j )
                    {
                        if ( iSrcBpp==4 )
                            a = srcBuf[iSrcIndex++];
                        else
                            a = 255;

                        r = srcBuf[iSrcIndex++];
                        g = srcBuf[iSrcIndex++];
                        b = srcBuf[iSrcIndex++];

                        if ( iDstBpp==4 )
                            dstBuf[iDstIndex++] = a;

                        dstBuf[iDstIndex++] = r;
                        dstBuf[iDstIndex++] = g;
                        dstBuf[iDstIndex++] = b;
                    }

                    iSrcIndex += iSrcExtraStrideBytes;
                    iDstIndex += iDstExtraStrideBytes;
                }

                src.UnlockBits( srcData );
                dst.UnlockBits( dstData );
            }

            return true;
        }


        /// <summary>
        /// Combine RGB and alpha bitmap pixels into single RGBA byte array.
        /// </summary>
        /// <param name="srcRGB">The source RGB bitmap.</param>
        /// <param name="srcAlpha">The source Alpha bitmap.</param>
        /// <param name="dst">The destination byte array.</param>
        /// <returns>True on success.</returns>
        public static bool CombineBitmap( Bitmap srcRGB,
                                          Bitmap srcAlpha,
                                          out byte[] dst )
        {
            dst = null;
            if ( srcRGB==null || srcAlpha==null )
                return false;

            // Need to be in the same size.
            if ( srcRGB.Width!=srcAlpha.Width || srcRGB.Height!=srcAlpha.Height )
                return false;

            // Create the destination bitmap.
            dst = new byte[ srcRGB.Width * srcRGB.Height * 4]; // RGBA

            unsafe
            {
                BitmapData srcColorData = srcRGB.LockBits( new Rectangle(0, 0, srcRGB.Width, srcRGB.Height),
                                                           ImageLockMode.ReadOnly,
                                                           PixelFormat.Format24bppRgb );
                BitmapData srcAlphaData = srcAlpha.LockBits( new Rectangle(0, 0, srcAlpha.Width, srcAlpha.Height),
                                                             ImageLockMode.ReadOnly,
                                                             PixelFormat.Format24bppRgb );

                byte* srcColorBuf = (byte*)(void*)srcColorData.Scan0;
                byte* srcAlphaBuf = (byte*)(void*)srcAlphaData.Scan0;

                int iExtraStrideBytes = srcColorData.Stride - (srcColorData.Width * 3);

                int iSrcIndex = 0;
                int iDstIndex = 0;

                int i, j;
                for ( i=0;i<srcColorData.Height;++i )
                {
                    for ( j=0;j<srcColorData.Width;++j )
                    {
                        dst[iDstIndex++] = srcColorBuf[iSrcIndex+2];
                        dst[iDstIndex++] = srcColorBuf[iSrcIndex+1];
                        dst[iDstIndex++] = srcColorBuf[iSrcIndex];
                        dst[iDstIndex++] = srcAlphaBuf[iSrcIndex];

                        iSrcIndex += 3;
                    }

                    iSrcIndex += iExtraStrideBytes;
                }

                srcRGB.UnlockBits( srcColorData );
                srcAlpha.UnlockBits( srcAlphaData );
            }

            return true;
        }


        /// <summary>
        /// Create a bitmap image and render checker board on it.
        /// </summary>
        /// <param name="iWidth">The width of the image to create.</param>
        /// <param name="iHeight">The height of the image to create.</param>
        /// <param name="iCheckerSize">The checker size.</param>
        /// <returns>The created checker board bitmap image.</returns>
        public static Bitmap CreateCheckerBoardImage(int iWidth,
                                                     int iHeight,
                                                     int iCheckerSize)
        {
            return CreateCheckerBoardImage(iWidth,
                                           iHeight,
                                           iCheckerSize,
                                           Color.White,
                                           Color.FromArgb(255, 230, 230, 230));
        }


        /// <summary>
        /// Create a bitmap image and render checker board on it.
        /// </summary>
        /// <param name="iWidth">The width of the image to create.</param>
        /// <param name="iHeight">The height of the image to create.</param>
        /// <param name="iCheckerSize">The checker size.</param>
        /// <param name="lightColor">The light color of the checker pattern.</param>
        /// <param name="darkColor">The dark color of the checker pattern.</param>
        /// <returns>The created checker board bitmap image.</returns>
        public static Bitmap CreateCheckerBoardImage(int iWidth,
                                                     int iHeight,
                                                     int iCheckerSize,
                                                     Color lightColor,
                                                     Color darkColor )
        {
            Bitmap image = new Bitmap(iWidth,
                                      iHeight,
                                      PixelFormat.Format8bppIndexed);

            // Setup the palette for the bitmap
            ColorPalette palette = image.Palette;
            {
                palette.Entries[0] = darkColor;
                palette.Entries[1] = lightColor;
            }

            image.Palette = palette;

            // Setup the rectangle for the whole image to lock
            Rectangle rect = new Rectangle(0, 0, iWidth, iHeight);

            // Lock the bitmap for write
            BitmapData bmpData = image.LockBits(rect,
                                                ImageLockMode.ReadWrite,
                                                PixelFormat.Format8bppIndexed);

            unsafe
            {
                byte* buffer = (byte*)(void*)bmpData.Scan0;

                int x, y;
                int iIndex = 0;

                // Render the checker board
                for (y = 0; y < iHeight; ++y)
                {
                    int iModY = (y / iCheckerSize) % 2;
                    for (x = 0; x < iWidth; ++x)
                    {
                        int iModX = (x / iCheckerSize) % 2;
                        if (iModX == iModY)
                            buffer[iIndex + x] = 0;
                        else
                            buffer[iIndex + x] = 1;
                    }

                    iIndex += bmpData.Stride;
                }
            }

            // Unlock the bitmap
            image.UnlockBits(bmpData);

            return image;
        }
    }
}
