﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Security.AccessControl;

namespace App.IO
{
    /// <summary>
    /// Postfixを纏めました
    /// </summary>
    public static class Postfix
    {
        /// <summary>Emitter</summary>
        public const string EmitterPostfix = "Emitter";
        /// <summary>ChildParticle</summary>
        public const string ChildParticlePostfix = "Child";
        /// <summary>Model</summary>
        public const string ModelPostfix = "Model";
        /// <summary>Folder</summary>
        public const string Folder = "Folder";
    }

    /// <summary>
    /// ドキュメント定数。
    /// </summary>
    public sealed class DocumentConstants
    {
        #region Constants

        /// <summary>Required documents file type key.</summary>
        public const string RequiredDocFileType = "RegDoc";

        /// <summary>Template file type key.</summary>
        public const string TemplateTypeKey = "Template";

        /// <summary>Model file type key.</summary>
        public const string ModelFileTypeKey = "Model";

        /// <summary>Shader generator project file type key.</summary>
        public const string ShaderGenProjFileTypeKey = "ShaderGenProj";

        /// <summary>Model animation file type key.</summary>
        public const string ModelAnimFileTypeKey = "ModelAnim";

        /// <summary>Material animation file type key.</summary>
        public const string MatAnimFileTypeKey = "MatAnim";

        /// <summary>Visibility animation file type key.</summary>
        public const string VisAnimFileTypeKey = "VisAnim";

        /// <summary>テクスチャ</summary>
        public const string Textures = "Textures";

        /// <summary>テクスチャフォルダ</summary>
        public const string TextureDirectory = "Textures\\";

        /// <summary>プリミティブ</summary>
        public const string Primitives = "Primitives";

        /// <summary>プリミティブフォルダ</summary>
        public const string PrimitiveDirectory = "Primitives\\";

        /// <summary>エフェクト</summary>
        public const string Effects = "Effects";

        /// <summary>エフェクトフォルダ</summary>
        public const string EffectDirectory = "Effects\\";

        /// <summary>連携</summary>
        public const string PreviewDirectory = "PreviewSettings\\";

        /// <summary>NW4Fルート環境変数</summary>
        public const string NW4FRootEnvironment = "NW4F_ROOT";

        /// <summary>NW4Fルート</summary>
        public static readonly string NW4FRoot;

        #endregion

        #region Static constructor

        /// <summary>
        /// Static constructor.
        /// </summary>
        static DocumentConstants()
        {
            NW4FRoot = Environment.GetEnvironmentVariable(NW4FRootEnvironment);
            if (NW4FRoot != null)
            {
                NW4FRoot = Path.GetFullPath(NW4FRoot);
            }
        }

        #endregion

        #region File / path utility

        #region General

        /// <summary>
        /// Get the directory name of the given path without the ending "\" and "/".
        /// The given path must be an absolute path and must not be empty.
        ///
        /// - For a file path, System.IO.Path.GetDirectoryName() returns correct path.
        /// - For a dir path with ending "/" or "\", System.IO.Path.GetDirectoryName()
        ///   returns correct path.
        /// - For a dir path WITHOUT ending "/" or "\", System.IO.Path.GetDirectoryName()
        ///   returns the parent directory path.
        ///
        /// So we need a method which always returns the current directory without "/" and "\".
        /// </summary>
        /// <param name="path">The file or directory path.</param>
        /// <returns>The directory path without ending "\" and "/".</returns>
        public static string GetDirectoryName( string path )
        {
            // Validate the given path first ( must not be empty, and must be absolute path ).
            if ( string.IsNullOrEmpty( path )==true )
                return string.Empty;

            if ( Path.IsPathRooted( path )==false )
                return string.Empty;

            string dirPath;

            // Is the given path a directory?
            // We want the directory path without the ending "\" and "/"
            if ( string.IsNullOrEmpty(Path.GetExtension(path))==true )
            {
                // It is a directory, remove the ending "\" and "/"
                int iIndex = -1;
                for ( int i=(path.Length-1);i>=0;--i )
                {
                    if ( path.EndsWith( "\\" )==false &&
                         path.EndsWith( "/" )==false )
                    {
                        iIndex = i;
                        break;
                    }
                }

                // Invalid file path.
                if ( iIndex<0 )
                    return string.Empty;

                // Get the directory path without the ending "\" and "/"
                dirPath = path.Substring( 0, iIndex+1 );
            }
            else
            {
                // It is a file path, just get the directory name.
                dirPath = Path.GetDirectoryName( path );
            }

            return dirPath;
        }


        /// <summary>
        /// True if the path is both set and absolute
        /// </summary>
        /// <param name="path">Path</param>
        /// <returns>True if both set and absolute</returns>
        public static bool IsPathSetAndRooted( string path )
        {
            try
            {
                if ( string.IsNullOrEmpty(path)==false &&
                     Path.IsPathRooted(path)==true )
                {
                    return true;
                }
            }
            catch
            {
            }

            return false;
        }


        /// <summary>
        /// Returns true if the path is set but cannot be found.
        /// </summary>
        /// <param name="assetPath">Path of the asset</param>
        /// <returns>True if unreachable</returns>
        public static bool NoSuchAssetPath( string assetPath )
        {
            try
            {
                if ( string.IsNullOrEmpty(assetPath)==false )
                {
                    if ( Path.IsPathRooted(assetPath)==false ||
                         File.Exists(assetPath)==false )
                    {
                        return true;
                    }
                }
            }
            catch
            {
            }

            return false;
        }


        /// <summary>
        /// Enumerate through all the sub folders.
        /// </summary>
        /// <param name="currDirInfo">The directory info of the folder to start seaching.</param>
        /// <returns>The directory info of all the sub folders.</returns>
        public static IEnumerable<DirectoryInfo> EnumerateSubFolders( DirectoryInfo currDirInfo,
                                                                      bool bIncludeSelf = true )
        {
            // Validate the given folder path.
            if ( currDirInfo==null ||
                 currDirInfo.Exists==false )
            {
                yield break;
            }

            if ( (currDirInfo.Attributes & FileAttributes.System)!=0 )
                yield break;

            try
            {
                DirectorySecurity dirSecurity = currDirInfo.GetAccessControl();
            }
            catch ( Exception )
            {
                // The current user is not authorized to access this folder.
                yield break;
            }

            if ( bIncludeSelf==true )
                yield return currDirInfo;

            // Enumerate through sub folders.
            foreach ( DirectoryInfo subDirInfo in currDirInfo.EnumerateDirectories() )
            {
                IEnumerable<DirectoryInfo> enumerator = null;
                try
                {
                    // Get the enumerator for the children of the sub folder.
                    enumerator = EnumerateSubFolders( subDirInfo, true );
                }
                catch
                {
                    enumerator = null;
                }

                if ( enumerator!=null )
                {
                    // Enumerate through the children of the sub folder.
                    foreach ( DirectoryInfo dirInfo in enumerator )
                    {
                        if ( (dirInfo.Attributes & FileAttributes.System)==0 )
                            yield return dirInfo;
                    }
                }
            }
        }


        /// <summary>
        /// Enumerate all the valid and existing asset directory info
        /// for the given emitter set file path.
        /// </summary>
        /// <param name="assetFolderName">The folder name of the assets.</param>
        /// <param name="esetFilePath">The emitter set file path ( or the directory ).</param>
        /// <returns>A list of valid and existing asset directory infos.</returns>
        public static IEnumerable<DirectoryInfo>
            EnumerateValidAssetPaths( string assetFolderName,
                                      string esetFilePath )
        {
            // Get the eset folder.
            string currDir = DocumentConstants.GetDirectoryName( esetFilePath );

            // If the eset file path is empty or not an absolute path,
            // DocumentConstants.GetDirectoryName() returns an empty string,
            // in that case just fail out.
            if ( string.IsNullOrEmpty(currDir)==true )
                yield break;

            DirectoryInfo dirInfo = new DirectoryInfo(currDir);
            if ( dirInfo==null || dirInfo.Exists==false )
                yield break;

            // First yield the folder where the eset is. ( ./ )
            yield return dirInfo;

            // Loop through all the (parent folder)/(asset folder)
            while ( Path.IsPathRooted(currDir)==true )
            {
                // The asset folder of the current folder. ( eq. ./Textures )
                dirInfo = new DirectoryInfo( Path.Combine(currDir, assetFolderName) );
                if ( dirInfo!=null && dirInfo.Exists==true )
                    yield return dirInfo;

                // Advance to the parent directory.
                currDir = Path.GetDirectoryName( currDir );
            }
        }


        /// <summary>
        /// List all the valid asset file paths for the given emitter set file path.
        ///
        /// The first returned path would be the directory that the emitter set is
        /// at, no matter if it exists or not.
        /// </summary>
        /// <param name="assetFolderName">The folder name of the assets.</param>
        /// <param name="esetFilePath">The emitter set file path ( or the directory ).</param>
        /// <param name="bCheckDirExistence">True to filter out the paths that do not exist.</param>
        /// <returns>A list of valid and existing asset directory paths.</returns>
        public static List<string> ListValidAssetPaths( string assetFolderName,
                                                        string esetFilePath,
                                                        bool bCheckDirExistence )
        {
            List<string> pathList = new List<string>();

            string dirPath = GetDirectoryName( esetFilePath );

            // Validate the eset file path first.
            if ( IsPathSetAndRooted( dirPath )==false )
                return pathList;

            // Add the directory of the eset file to our list.
            pathList.Add( dirPath );

            // If the folder exists, add it to our list.
            string texPath = Path.Combine( dirPath, assetFolderName );
            if ( bCheckDirExistence==false ||
                 Directory.Exists( texPath )==true )
            {
                pathList.Add( texPath );
            }

            // Traverse to the root directory
            while ( Path.IsPathRooted( dirPath )==true )
            {
                // Get to the parent directory.
                dirPath = Path.GetDirectoryName( dirPath );
                if ( string.IsNullOrEmpty(dirPath)==true )
                    break;

                // If the folder exists, add it to our list.
                texPath = Path.Combine( dirPath, assetFolderName );
                if ( bCheckDirExistence==false ||
                     Directory.Exists( texPath )==true )
                {
                    pathList.Add( texPath );
                }
            }

            return pathList;
        }


        /// <summary>
        /// Check if the given asset path ( either file or directory path ) is valid
        /// for the emitter set.
        /// </summary>
        /// <param name="assetFolderName">The folder name of the assets.</param>
        /// <param name="esetFilePath">The file or directory path of the emitter set.</param>
        /// <param name="assetFilePath">The file or directory path of the asset.</param>
        /// <returns>True if the asset path is valid.</returns>
        public static bool ValidateAssetPath( string assetFolderName,
                                              string esetFilePath,
                                              string assetFilePath )
        {
            string assetDir = GetDirectoryName( assetFilePath );

            // Validate the asset path first.
            if ( IsPathSetAndRooted( assetDir )==false )
                return false;

            // Convert the path to lower case.
            assetDir = assetDir.ToLower();

            #region Loop through the valid asset paths

            string esetDir = GetDirectoryName( esetFilePath );

            // Validate the eset file path first.
            if ( IsPathSetAndRooted( esetDir )==false )
                return false;

            string currDir = null;

            // Same directory?
            currDir = esetDir.ToLower();
            if ( currDir.Length==assetDir.Length && currDir==assetDir )
                return true;

            // The ".\#asset folder name#" folder?
            currDir = Path.Combine( esetDir, assetFolderName ).ToLower();
            if ( currDir.Length==assetDir.Length && currDir==assetDir )
                return true;

            // Traverse to the root directory
            while ( Path.IsPathRooted( esetDir )==true )
            {
                // Get to the parent directory.
                esetDir = Path.GetDirectoryName( esetDir );
                if ( string.IsNullOrEmpty(esetDir)==true )
                    break;

                currDir = Path.Combine( esetDir, assetFolderName ).ToLower();
                if ( currDir.Length==assetDir.Length && currDir==assetDir )
                    return true;
            }

            #endregion

            return false;
        }


        /// <summary>
        /// Locate the asset file with the given asset file name and return
        /// the full path to the file.
        /// </summary>
        /// <param name="assetFolderName">The folder name of the assets.</param>
        /// <param name="esetFilePath">The file or directory path of the emitter set.</param>
        /// <param name="assetFileName">The FILE NAME of the asset file ( FileName.extension ).</param>
        /// <param name="fullPath">The full path to the asset file.</param>
        /// <returns>True if the file is located.</returns>
        public static bool LocateAssetByFileName( string assetFolderName,
                                                  string esetFilePath,
                                                  string assetFileName,
                                                  out string fullPath )
        {
            // Initialize the output variable first.
            fullPath = string.Empty;

            string fileName = Path.GetFileName( assetFileName );

            #region Loop through the valid asset paths

            string esetDir   = GetDirectoryName( esetFilePath );
            string assetPath = Path.Combine( assetFolderName,
                                             fileName );

            // Validate the eset file path first.
            if ( IsPathSetAndRooted( esetDir )==false )
                return false;

            string filePath = null;

            // Same directory?
            filePath = Path.Combine( esetDir, fileName );
            if ( File.Exists(filePath)==true )
            {
                fullPath = filePath;
                return true;
            }

            // Is the asset in the ".\#asset folder name#" folder?
            filePath = Path.Combine( esetDir, assetPath );
            if ( File.Exists(filePath)==true )
            {
                fullPath = filePath;
                return true;
            }

            // Traverse to the root directory
            while ( Path.IsPathRooted(esetDir)==true )
            {
                // Get to the parent directory.
                esetDir = Path.GetDirectoryName( esetDir );
                if ( string.IsNullOrEmpty(esetDir)==true )
                    break;

                filePath = Path.Combine( esetDir, assetPath );
                if ( File.Exists(filePath)==true )
                {
                    fullPath = filePath;
                    return true;
                }
            }

            #endregion

            return false;
        }


        /// <summary>
        /// Enumerate through all the possible folders that contain emitter
        /// sets those can reference to the assets in the given asset folder
        /// or asset file.
        /// </summary>
        /// <param name="assetPath">The path to the file or folder of the asset.</param>
        /// <param name="assetFolderName">The default folder name of the kind of asset.</param>
        /// <returns>The enumerator.</returns>
        public static IEnumerable<DirectoryInfo>
            EnumerateEsetFoldersForAsset( string assetPath,
                                          string assetFolderName )
        {
            // Get the parent folder path of the asset.
            string currDir = DocumentConstants.GetDirectoryName( assetPath );

            // If the asset file path is empty or not an absolute path,
            // DocumentConstants.GetDirectoryName() returns an empty string,
            // in that case just fail out.
            if ( string.IsNullOrEmpty(currDir)==true )
                yield break;

            DirectoryInfo dirInfo = new DirectoryInfo(currDir);
            if ( dirInfo==null ||
                 dirInfo.Exists==false )
            {
                yield break;
            }

            // First yield the folder where the asset is. ( ./ )
            yield return dirInfo;

            string tmpAssetFolderName = assetFolderName;
            if ( tmpAssetFolderName.EndsWith("\\")==true )
            {
                // Remove the trailing "\".
                tmpAssetFolderName =
                    tmpAssetFolderName.Substring( 0, tmpAssetFolderName.Length-1 );
            }

            // If the folder name doesn't match the default asset folder name,
            // this folder is the only possible folder, finish the search.
            if ( Path.GetFileName(currDir)!=tmpAssetFolderName )
                yield break;

            // Go the the parent folder.
            // We use Path.GetDirectory here, because
            // DocumentConstants.GetDirectoryName() works differently. Refer
            // to DocumentConstants.GetDirectoryName() above for detail.
            dirInfo = dirInfo.Parent;

            // The folder might already be the root of the drive if
            // Path.GetDirectoryName() returns null, so just finish
            // the search here.
            if ( dirInfo==null ||
                 dirInfo.Exists==false )
                yield break;

            // yield this folder too. ( ../ )
            yield return dirInfo;

            // Recursively list all the sub folder.
            foreach ( DirectoryInfo subDirInfo in EnumerateSubFolders(dirInfo, false) )
            {
                yield return subDirInfo;
            }
        }

        #endregion

        #region Texture path for emitter sets

        /// <summary>
        /// List all the valid texture paths for the given emitter set file path.
        ///
        /// The first returned path would be the directory that the emitter set is
        /// at, no matter if it exists or not.
        /// </summary>
        /// <param name="esetFilePath">The emitter set file path ( or the directory ).</param>
        /// <param name="bCheckDirExistence">True to filter out the paths that do not exist.</param>
        /// <returns>A list of valid and existing texture directory paths.</returns>
        public static List<string> ListValidTexturePaths( string esetFilePath,
                                                          bool bCheckDirExistence )
        {
            return ListValidAssetPaths( DocumentConstants.Textures,
                                        esetFilePath,
                                        bCheckDirExistence );
        }


        /// <summary>
        /// Check if the given texture path ( either file or directory path ) is valid
        /// for the emitter set.
        /// </summary>
        /// <param name="esetFilePath">The file or directory path of the emitter set.</param>
        /// <param name="texFilePath">The file or directory path of the texture.</param>
        /// <returns>True if the texture path is valid.</returns>
        public static bool ValidateTexturePath( string esetFilePath,
                                                string texFilePath )
        {
            return ValidateAssetPath( DocumentConstants.Textures,
                                      esetFilePath,
                                      texFilePath );
        }


        /// <summary>
        /// Locate the texture file with the given texture file name and return
        /// the full path to the file.
        /// </summary>
        /// <param name="esetFilePath">The file or directory path of the emitter set.</param>
        /// <param name="texFileName">The FILE NAME of the texture file ( FileName.extension ).</param>
        /// <param name="texFullPath">The full path to the texture file.</param>
        /// <returns>True if the file is located.</returns>
        public static bool LocateTextureByFileName( string esetFilePath,
                                                    string texFileName,
                                                    out string texFullPath )
        {
            return LocateAssetByFileName( DocumentConstants.Textures,
                                          esetFilePath,
                                          texFileName,
                                          out texFullPath );
        }

        #endregion

        #region Primitive path for emitter sets

        /// <summary>
        /// List all the valid primitive paths for the given emitter set file path.
        ///
        /// The first returned path would be the directory that the emitter set is
        /// at, no matter if it exists or not.
        /// </summary>
        /// <param name="esetFilePath">The emitter set file path ( or the directory ).</param>
        /// <param name="bCheckDirExistence">True to filter out the paths that do not exist.</param>
        /// <returns>A list of valid and existing primitive directory paths.</returns>
        public static List<string> ListValidPrimitivePaths( string esetFilePath,
                                                            bool bCheckDirExistence )
        {
            return ListValidAssetPaths( DocumentConstants.Primitives,
                                        esetFilePath,
                                        bCheckDirExistence );
        }


        /// <summary>
        /// Check if the given primitive path ( either file or directory path ) is valid
        /// for the emitter set.
        /// </summary>
        /// <param name="esetFilePath">The file or directory path of the emitter set.</param>
        /// <param name="filePath">The file or directory path of the primitive.</param>
        /// <returns>True if the primitive path is valid.</returns>
        public static bool ValidatePrimitivePath( string esetFilePath,
                                                  string filePath )
        {
            return ValidateAssetPath( DocumentConstants.Primitives,
                                      esetFilePath,
                                      filePath );
        }


        /// <summary>
        /// Locate the primitive file with the given primitive file name and return
        /// the full path to the file.
        /// </summary>
        /// <param name="esetFilePath">The file or directory path of the emitter set.</param>
        /// <param name="fileName">The FILE NAME of the primitive file ( FileName.extension ).</param>
        /// <param name="fullPath">The full path to the primitive file.</param>
        /// <returns>True if the file is located.</returns>
        public static bool LocatePrimitiveByFileName( string esetFilePath,
                                                      string fileName,
                                                      out string fullPath )
        {
            return LocateAssetByFileName( DocumentConstants.Primitives,
                                          esetFilePath,
                                          fileName,
                                          out fullPath );
        }

        #endregion

        #endregion

        #region File extensions

        //---------------------------------------------------------------------
        // 拡張子リスト
        //---------------------------------------------------------------------
        /// <summary>拡張子</summary>
        public const string Eset = "eset";
        /// <summary>.拡張子</summary>
        public const string DotEset = ".eset";

        /// <summary>拡張子</summary>
        public const string Prev = "prev";
        /// <summary>.拡張子</summary>
        public const string DotPrev = ".prev";

        /// <summary>拡張子</summary>
        public const string FEnv = "work";
        /// <summary>.拡張子</summary>
        public const string DotFEnv = ".work";

        //---------------------------------------------------------------------
        // テクスチャ拡張子リスト
        //---------------------------------------------------------------------
        /// <summary>拡張子</summary>
        public const string Ftx = "ftx";
        /// <summary>.拡張子</summary>
        public const string DotFtx = ".ftx";

        /// <summary>拡張子</summary>
        public const string Ftxa = "ftxa";
        /// <summary>.拡張子</summary>
        public const string DotFtxa = ".ftxa";

        /// <summary>拡張子</summary>
        public const string Ftxb = "ftxb";
        /// <summary>.拡張子</summary>
        public const string DotFtxb = ".ftxb";

        /// <summary>拡張子</summary>
        public const string Tga = "tga";
        /// <summary>.拡張子</summary>
        public const string DotTga = ".tga";

        public const string TextureFileSearchFilter = "*.ftxa,*.ftxb,*.tga";
        public const string LegalTextureFileExtensions = ".ftxa;.ftxb;.tga";

        //---------------------------------------------------------------------
        // モデル拡張子リスト
        //---------------------------------------------------------------------
        /// <summary>Model file extension.</summary>
        public const string Cmdl = "cmdl";
        /// <summary>Model file extension with the leading dot.</summary>
        public const string DotCmdl = ".cmdl";

        /// <summary>拡張子</summary>
        public const string Fmd = "fmd";
        /// <summary>.拡張子</summary>
        public const string DotFmd = ".fmd";

        /// <summary>拡張子</summary>
        public const string Fmda = "fmda";
        /// <summary>.拡張子</summary>
        public const string DotFmda = ".fmda";

        /// <summary>拡張子</summary>
        public const string Fmdb = "fmdb";
        /// <summary>.拡張子</summary>
        public const string DotFmdb = ".fmdb";

        #if BUILD_FOR_CTR
        public const string ModelFileSearchFilter = "*.cmdl";
        #else
        public const string ModelFileSearchFilter = "*.fmda,*.fmdb";
        #endif


        //---------------------------------------------------------------------
        // アニメーション拡張子リスト
        //---------------------------------------------------------------------
        /// <summary>Animation file extension.</summary>
        public const string Cskla = "cskla";
        /// <summary>Animation file extension with the leading dot.</summary>
        public const string DotCskla = ".cskla";

        /// <summary>Animation file extension.</summary>
        public const string Fskb = "fskb";
        /// <summary>Animation file extension with the leading dot.</summary>
        public const string DotFskb = ".fskb";


        //---------------------------------------------------------------------
        // バイナリ拡張子リスト
        //---------------------------------------------------------------------
        /// <summary>Eset のバイナリー拡張子</summary>
        public const string Beset = "ptcl";
        /// <summary>Eset のバイナリー拡張子（dot付）</summary>
        public const string DotBeset = ".ptcl";

        /// <summary>
        /// Get the native texture file extension filter.
        /// </summary>
        /// <returns>The filter string.</returns>
        public static string GetNativeTextureFileDialogFilter()
        {
            #if BUILD_FOR_CTR
            return res.Strings.UI_CTR_TEXTURE_PROPERTY_FILE_DIALOG_FILTER;
            #else
            return res.Strings.UI_NW4F_TEXTURE_PROPERTY_FILE_DIALOG_FILTER;
            #endif
        }

        /// <summary>
        /// Get the native texture file extension ( with the leading period(".") ).
        /// </summary>
        /// <returns>The file extension.</returns>
        public static string GetNativeTextureFileExt()
        {
            #if BUILD_FOR_CTR
            return DotTga;
            #else
            return DotFtxb;
            #endif
        }

        /// <summary>
        /// Get the native model file extension filter.
        /// </summary>
        /// <returns>The filter string.</returns>
        public static string GetNativeModelFileDialogFilter()
        {
            #if BUILD_FOR_CTR
            return res.Strings.FILE_FILTER_MODEL_FILE_NW4C;
            #else
            return res.Strings.FILE_FILTER_MODEL_FILE_NW4F;
            #endif
        }

        /// <summary>
        /// Get the native model file extension ( with the leading period(".") ).
        /// </summary>
        /// <param name="bWithLeadingPeriod">True to include the leading period to the extension.</param>
        /// <returns>The file extension.</returns>
        public static string GetNativeModelFileExt( bool bWithLeadingPeriod = true )
        {
            if ( bWithLeadingPeriod==true )
            {
                #if BUILD_FOR_CTR
                return DotCmdl;
                #else
                return DotFmdb;
                #endif
            }
            else
            {
                #if BUILD_FOR_CTR
                return Cmdl;
                #else
                return Fmdb;
                #endif
            }
        }

        /// <summary>
        /// Get the native animation file extension filter.
        /// </summary>
        /// <returns>The filter string.</returns>
        public static string GetNativeAnimationFileDialogFilter()
        {
            #if BUILD_FOR_CTR
            return res.Strings.FILE_FILTER_ANIM_FILE_NW4C;
            #else
            return res.Strings.FILE_FILTER_ANIM_FILE_NW4F;
            #endif
        }

        /// <summary>
        /// Get the native animation file extension ( with the leading period(".") ).
        /// </summary>
        /// <returns>The file extension.</returns>
        public static string GetNativeAnimationFileExt()
        {
            #if BUILD_FOR_CTR
            return DotCskla;
            #else
            return DotFskb;
            #endif
        }

        #endregion

        /// <summary>
        /// エフェクトパス変換
        /// </summary>
        public static string ConvertEffectPath(string path)
        {
            string extension = Path.GetExtension(path).ToLower();
            if (extension == DocumentConstants.DotEset)
            {
                string folder =
                    Path.Combine(Path.GetDirectoryName(path), DocumentConstants.EffectDirectory);
                return Path.Combine(folder, Path.GetFileName(path));
            }
            return path;
        }

        //---------------------------------------------------------------------
        // パスの拡張子
        //---------------------------------------------------------------------
        /// <summary>
        /// 不正なパスか
        /// </summary>
        public static bool IsInvalidPath(string filePath)
        {
            bool valid = false;
            valid |= IsEsetPath(filePath);
//              valid |= IsCmdlPath(filePath);
            return (!valid);
        }

        /// <summary>
        /// エミッタセットパスか
        /// </summary>
        public static bool IsEsetPath(string filePath)
        {
            return (Path.GetExtension(filePath) == DotEset);
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private DocumentConstants() {}
    }
}
