﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;

using App.IO;

using PrevVerDataModel  = NWCore.DataModel.Major_0.Minor_0.Build_0.Revision_0;
using CurrVerDataModel  = NWCore.DataModel.Major_0.Minor_9.Build_0.Revision_0;

using PrevVerSerializer = NWCore.Serializer.Major_0.Minor_0.Build_0.Revision_0;
using CurrVerSerializer = NWCore.Serializer.Major_0.Minor_9.Build_0.Revision_0;

namespace App.Data
{
    /// <summary>
    /// Emitter set data model updater class.
    /// Version [ 0.0.0.0 ] => [ 0.9.0.0 ]
    /// </summary>
    public class EsetUpdater_0_9_0_0 : IDataModelUpdater
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public EsetUpdater_0_9_0_0()
        {
        }

        #endregion

        #region Properties

        /// <summary>Get the data model version this updater converts from.</summary>
        public Version PreviousVersion { get { return m_prevVersion; } }


        /// <summary>Get the data model version this updater converts to.</summary>
        public Version CurrentVersion { get { return m_currVersion; } }

        #endregion

        #region Derived from IDataModelUpdater

        /// <summary>
        /// Check if the updater can deserialize the data model with
        /// the given binary version.
        /// </summary>
        /// <param name="binaryVersion">The binary version.</param>
        /// <returns>True if the updater can deserialize the file.</returns>
        public bool CanDeserialize( Version binaryVersion )
        {
            // This updater is the first updater, it should cover
            // both the first and the second versions.
            if ( binaryVersion.Equals( m_prevVersion )==true ||
                 binaryVersion.Equals( m_currVersion )==true )
            {
                return true;
            }

            return false;
        }


        /// <summary>
        /// Check if the data model can be updated with this updater.
        /// </summary>
        /// <param name="dataModel">The data model to be updated.</param>
        /// <returns>True if the data model can be updated by this updater.</returns>
        public bool CanUpdate( object dataModel )
        {
            if ( dataModel is PrevVerSerializer.EmitterSetDataXml )
                return true;

            return false;
        }


        /// <summary>
        /// Deserialize the data in the given memory stream and return the
        /// deserialized data model.
        /// </summary>
        /// <param name="binaryVersion">The binary version of the data.</param>
        /// <param name="stream">The memory stream contains the data to be deserialized.</param>
        /// <returns>The deserialized data model.</returns>
        public object Deserialize( Version binaryVersion,
                                   Stream stream )
        {
            // This is the earliest updater, both the deserializers of the
            // old and new versions has to be covered.
            CurrVerSerializer.EmitterSetDataXml data = null;
            if ( binaryVersion.Equals( m_prevVersion )==true )
            {
                PrevVerSerializer.EmitterSetDataXml content =
                    DocumentIO.DeserializeFromFile<PrevVerSerializer.EmitterSetDataXml>( stream );
                if ( content==null )
                    return null;

                // Update the old data model to the newer version
                data = Update( content ) as CurrVerSerializer.EmitterSetDataXml;
            }
            else if ( binaryVersion.Equals( m_currVersion )==true )
            {
                CurrVerSerializer.EmitterSetDataXml content =
                    DocumentIO.DeserializeFromFile<CurrVerSerializer.EmitterSetDataXml>( stream );
                if ( content==null )
                    return null;

                data = content;
            }

            return data;
        }


        /// <summary>
        /// Update the given data model to a later version.
        /// </summary>
        /// <param name="src">The source data model.</param>
        /// <returns>The updated data model.</returns>
        public object Update( object src )
        {
            PrevVerSerializer.EmitterSetDataXml srcData =
                src as PrevVerSerializer.EmitterSetDataXml;
            if ( srcData==null )
                return null;

            CurrVerSerializer.EmitterSetDataXml dstData =
                new CurrVerSerializer.EmitterSetDataXml();

            #region Convert emitter set data

//<name>タグを削除
//            dstData.name           = string.Copy( srcData.name );
            dstData.userData       = srcData.userData;
            dstData.lastUpdateDate = srcData.lastUpdateDate;

            dstData.EmitterSetData.EditData.ColorLabelIndex = srcData.EmitterSetData.EditData.ColorLabelIndex;
            dstData.EmitterSetData.EditData.Comment = string.Copy( srcData.EmitterSetData.EditData.Comment );

            #endregion

            // TODO : The data to convert is inside the #region below
            #region Convert emitter data

            foreach ( PrevVerSerializer.ComplexEmitterDataXml srcEmitter in srcData.EmitterList )
            {
                CurrVerSerializer.ComplexEmitterDataXml dstEmitter =
                    new CurrVerSerializer.ComplexEmitterDataXml();

                dstEmitter.EditData.Initialize();

                dstEmitter.name = srcEmitter.name;
                dstEmitter.texture = srcEmitter.texture;
                dstEmitter.subTexture = srcEmitter.subTexture;
                dstEmitter.UI_randomSeed = srcEmitter.UI_randomSeed;
                dstEmitter.volumeRadius = srcEmitter.volumeRadius;
                dstEmitter.UI_volumeSweep = srcEmitter.UI_volumeSweep;
                dstEmitter.UI_volumeSweepStart = srcEmitter.UI_volumeSweepStart;
                dstEmitter.volumeTblIndex = srcEmitter.volumeTblIndex;
                dstEmitter.UI_Fog = srcEmitter.UI_Fog;
                dstEmitter.figureVel = srcEmitter.figureVel;
                dstEmitter.emitterVel = srcEmitter.emitterVel;
                dstEmitter.UI_initVelRnd = srcEmitter.UI_initVelRnd;
                dstEmitter.spreadVec = srcEmitter.spreadVec;
                dstEmitter.emitterVelDir = srcEmitter.emitterVelDir;
                dstEmitter.startFrame = srcEmitter.startFrame;
                dstEmitter.UI_emitTime = srcEmitter.UI_emitTime;
                dstEmitter.UI_isOneTime = srcEmitter.UI_isOneTime;
                dstEmitter.lifeStep = srcEmitter.lifeStep;
                dstEmitter.emitRate = srcEmitter.emitRate;
                dstEmitter.UI_emitRate = srcEmitter.UI_emitRate;
                dstEmitter.UI_life = srcEmitter.UI_life;
                dstEmitter.UI_lifeRnd = srcEmitter.UI_lifeRnd;
                dstEmitter.UI_lifeInfinit = srcEmitter.UI_lifeInfinit;
                dstEmitter.UI_lifeStepRnd = srcEmitter.UI_lifeStepRnd;
                dstEmitter.airRegist = srcEmitter.airRegist;
                dstEmitter.UI_gravityDir = srcEmitter.UI_gravityDir;
                dstEmitter.UI_gravity = srcEmitter.UI_gravity;
                dstEmitter.UI_isFieldCalcType = srcEmitter.UI_isFieldCalcType;
                dstEmitter.UI_alpha1 = srcEmitter.UI_alpha1;
                dstEmitter.UI_alpha2 = srcEmitter.UI_alpha2;
                dstEmitter.UI_alpha3 = srcEmitter.UI_alpha3;
                dstEmitter.UI_alphaTime1 = srcEmitter.UI_alphaTime1;
                dstEmitter.UI_alphaTime2 = srcEmitter.UI_alphaTime2;
                dstEmitter.UI_scale2 = srcEmitter.UI_scale2;
                dstEmitter.UI_scaleTime1 = srcEmitter.UI_scaleTime1;
                dstEmitter.UI_scaleTime2 = srcEmitter.UI_scaleTime2;
                dstEmitter.UI_scaleRand = srcEmitter.UI_scaleRand;
                dstEmitter.UI_rotBasis = srcEmitter.UI_rotBasis;
                dstEmitter.UI_transformScale = srcEmitter.UI_transformScale;
                dstEmitter.UI_fadeAlphaFrameInFade = srcEmitter.UI_fadeAlphaFrameInFade;
                dstEmitter.UI_texPatDivX = srcEmitter.UI_texPatDivX;
                dstEmitter.UI_texPatDivY = srcEmitter.UI_texPatDivY;
                dstEmitter.UI_texPatNumPattern = srcEmitter.UI_texPatNumPattern;
                dstEmitter.UI_texPatFreqFrame = srcEmitter.UI_texPatFreqFrame;
                dstEmitter.UI_numUseTexPatTbl = srcEmitter.UI_numUseTexPatTbl;
                dstEmitter.UI_texPatAnimMode = srcEmitter.UI_texPatAnimMode;
                dstEmitter.UI_texPatAnim = srcEmitter.UI_texPatAnim;
                dstEmitter.UI_texWrapU = srcEmitter.UI_texWrapU;
                dstEmitter.UI_texWrapV = srcEmitter.UI_texWrapV;
                dstEmitter.UI_texPatTbl_0 = srcEmitter.UI_texPatTbl_0;
                dstEmitter.UI_texPatTbl_1 = srcEmitter.UI_texPatTbl_1;
                dstEmitter.UI_texPatTbl_2 = srcEmitter.UI_texPatTbl_2;
                dstEmitter.UI_texPatTbl_3 = srcEmitter.UI_texPatTbl_3;
                dstEmitter.UI_texPatTbl_4 = srcEmitter.UI_texPatTbl_4;
                dstEmitter.UI_texPatTbl_5 = srcEmitter.UI_texPatTbl_5;
                dstEmitter.UI_texPatTbl_6 = srcEmitter.UI_texPatTbl_6;
                dstEmitter.UI_texPatTbl_7 = srcEmitter.UI_texPatTbl_7;
                dstEmitter.UI_texPatTbl_8 = srcEmitter.UI_texPatTbl_8;
                dstEmitter.UI_texPatTbl_9 = srcEmitter.UI_texPatTbl_9;
                dstEmitter.UI_texPatTbl_10 = srcEmitter.UI_texPatTbl_10;
                dstEmitter.UI_texPatTbl_11 = srcEmitter.UI_texPatTbl_11;
                dstEmitter.UI_texPatTbl_12 = srcEmitter.UI_texPatTbl_12;
                dstEmitter.UI_texPatTbl_13 = srcEmitter.UI_texPatTbl_13;
                dstEmitter.UI_texPatTbl_14 = srcEmitter.UI_texPatTbl_14;
                dstEmitter.UI_texPatTbl_15 = srcEmitter.UI_texPatTbl_15;
                dstEmitter.UI_texPatTbl_16 = srcEmitter.UI_texPatTbl_16;
                dstEmitter.UI_texPatTbl_17 = srcEmitter.UI_texPatTbl_17;
                dstEmitter.UI_texPatTbl_18 = srcEmitter.UI_texPatTbl_18;
                dstEmitter.UI_texPatTbl_19 = srcEmitter.UI_texPatTbl_19;
                dstEmitter.UI_texPatTbl_20 = srcEmitter.UI_texPatTbl_20;
                dstEmitter.UI_texPatTbl_21 = srcEmitter.UI_texPatTbl_21;
                dstEmitter.UI_texPatTbl_22 = srcEmitter.UI_texPatTbl_22;
                dstEmitter.UI_texPatTbl_23 = srcEmitter.UI_texPatTbl_23;
                dstEmitter.UI_texPatTbl_24 = srcEmitter.UI_texPatTbl_24;
                dstEmitter.UI_texPatTbl_25 = srcEmitter.UI_texPatTbl_25;
                dstEmitter.UI_texPatTbl_26 = srcEmitter.UI_texPatTbl_26;
                dstEmitter.UI_texPatTbl_27 = srcEmitter.UI_texPatTbl_27;
                dstEmitter.UI_texPatTbl_28 = srcEmitter.UI_texPatTbl_28;
                dstEmitter.UI_texPatTbl_29 = srcEmitter.UI_texPatTbl_29;
                dstEmitter.UI_texPatTbl_30 = srcEmitter.UI_texPatTbl_30;
                dstEmitter.UI_texPatTbl_31 = srcEmitter.UI_texPatTbl_31;
                dstEmitter.childTexture = srcEmitter.childTexture;
                dstEmitter.childFlg = srcEmitter.childFlg;
                dstEmitter.UI_childLife = srcEmitter.UI_childLife;
                dstEmitter.UI_childRotBasis = srcEmitter.UI_childRotBasis;
                dstEmitter.childEmitRate = srcEmitter.childEmitRate;
                dstEmitter.childEmitTiming = srcEmitter.childEmitTiming;
                dstEmitter.childEmitStep = srcEmitter.childEmitStep;
                dstEmitter.childInitPosRand = srcEmitter.childInitPosRand;
                dstEmitter.childVelInheritRate = srcEmitter.childVelInheritRate;
                dstEmitter.childFigureVel = srcEmitter.childFigureVel;
                dstEmitter.childRandVel = srcEmitter.childRandVel;
                dstEmitter.childAirRegist = srcEmitter.childAirRegist;
                dstEmitter.UI_childTexPatDivX = srcEmitter.UI_childTexPatDivX;
                dstEmitter.UI_childTexPatDivY = srcEmitter.UI_childTexPatDivY;
                dstEmitter.UI_childTexPatNumPattern = srcEmitter.UI_childTexPatNumPattern;
                dstEmitter.UI_childTexPatFreqFrame = srcEmitter.UI_childTexPatFreqFrame;
                dstEmitter.UI_childNumUseTexPatTbl = srcEmitter.UI_childNumUseTexPatTbl;
                dstEmitter.UI_childTexPatAnimMode = srcEmitter.UI_childTexPatAnimMode;
                dstEmitter.UI_childTexPatAnim = srcEmitter.UI_childTexPatAnim;
                dstEmitter.UI_childTexWrapU = srcEmitter.UI_childTexWrapU;
                dstEmitter.UI_childTexWrapV = srcEmitter.UI_childTexWrapV;
                dstEmitter.UI_childConstColor0 = srcEmitter.UI_childConstColor;
                dstEmitter.UI_childColor = srcEmitter.UI_childColor;
                dstEmitter.UI_childInheritanceColor = srcEmitter.UI_childInheritanceColor;
                dstEmitter.childAlpha = srcEmitter.childAlpha;
                dstEmitter.childAlphaTarget = srcEmitter.childAlphaTarget;
                dstEmitter.childAlphaInit = srcEmitter.childAlphaInit;
                dstEmitter.UI_childAlphaSec = srcEmitter.UI_childAlphaSec;
                dstEmitter.UI_childAlphaBaseSec = srcEmitter.UI_childAlphaBaseSec;
                dstEmitter.childScaleInheritRate = srcEmitter.childScaleInheritRate;
                dstEmitter.childScale = srcEmitter.childScale;
                dstEmitter.UI_childScaleSec = srcEmitter.UI_childScaleSec;
                dstEmitter.UI_childGravity = srcEmitter.UI_childGravity;
                dstEmitter.UI_childGravityDir = srcEmitter.UI_childGravityDir;
                dstEmitter.childShaderType = CurrVerDataModel.ShaderType.Particle;
                dstEmitter.childShaderParam0 = 0.0f;
                dstEmitter.childShaderParam1 = 0.0f;
                dstEmitter.fieldFlg = srcEmitter.fieldFlg;
                dstEmitter.fieldRandomBlank = srcEmitter.fieldRandomBlank;
                dstEmitter.fieldRandomVelAdd = srcEmitter.fieldRandomVelAdd;
                dstEmitter.fieldMagnetPower = srcEmitter.fieldMagnetPower;
                dstEmitter.fieldMagnetPos = srcEmitter.fieldMagnetPos;
                dstEmitter.fieldMagnetFlg = srcEmitter.fieldMagnetFlg;
                dstEmitter.fieldSpinRotate = srcEmitter.fieldSpinRotate;
                dstEmitter.fieldSpinAxis = srcEmitter.fieldSpinAxis;
                dstEmitter.fieldCollisionType = srcEmitter.fieldCollisionType;
                dstEmitter.fieldCollisionIsWorld = srcEmitter.fieldCollisionIsWorld;
                dstEmitter.fieldCollisionCoord = srcEmitter.fieldCollisionCoord;
                dstEmitter.fieldCollisionCoef = srcEmitter.fieldCollisionCoef;
                dstEmitter.fieldConvergencePos = srcEmitter.fieldConvergencePos;
                dstEmitter.fieldConvergenceRatio = srcEmitter.fieldConvergenceRatio;
                dstEmitter.fieldPosAdd = srcEmitter.fieldPosAdd;
                dstEmitter.fluctuationFlg = srcEmitter.fluctuationFlg;
                dstEmitter.fluctuationScale = srcEmitter.fluctuationScale;
                dstEmitter.fluctuationPhaseRnd = srcEmitter.fluctuationPhaseRnd;
                dstEmitter.UI_fluctuationFreq = srcEmitter.UI_fluctuationFreq;
                dstEmitter.stripeFlg = srcEmitter.stripeFlg;
                dstEmitter.stripeStartAlpha = srcEmitter.stripeStartAlpha;
                dstEmitter.stripeEndAlpha = srcEmitter.stripeEndAlpha;
                dstEmitter.stripeUVScroll = srcEmitter.stripeUVScroll;
                dstEmitter.stripeNumHistory = srcEmitter.stripeNumHistory;
                dstEmitter.UI_stripeHistoryStep = srcEmitter.UI_stripeHistoryStep;
                dstEmitter.stripeDirInterpolate = srcEmitter.stripeDirInterpolate;
                dstEmitter.useSortParticle = srcEmitter.useSortParticle;
                dstEmitter.shaderParam0 = srcEmitter.shaderParam0;
                dstEmitter.shaderParam1 = srcEmitter.shaderParam1;
                dstEmitter.uvScroll = srcEmitter.uvScroll;
                dstEmitter.UI_transformTrans = srcEmitter.UI_transformTrans;
                dstEmitter.UI_childScaleTarget.X = srcEmitter.UI_childScaleTarget.X;
                dstEmitter.UI_childScaleTarget.Y = srcEmitter.UI_childScaleTarget.Y;
                dstEmitter.order = srcEmitter.order;

                //-------------------------------------------------------------
                // Color
                //-------------------------------------------------------------

                switch (srcEmitter.UI_colorType)
                {
                    case /*0*/"ConstColorNoTex":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ConstColor];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;
                            dstEmitter.UI_color0_color1 = srcEmitter.UI_constColor;

                            /*
                                case 0:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_0;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_constColor" );
                            */
                        }
                        break;
                    case /*1*/"ColorAnimNoTex":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ConstColor];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Animation;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color0_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color0_color3 = srcEmitter.UI_color3;

                            dstEmitter.UI_color0_time1 = srcEmitter.UI_colorTime1;
                            dstEmitter.UI_color0_time2 = srcEmitter.UI_colorTime2;
                            dstEmitter.UI_color0_time3 = srcEmitter.UI_colorTime3;
                            dstEmitter.UI_color0_numColorRepeat = srcEmitter.UI_numColorRepeat;

                            /*
                                case 1:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_0;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM_LINEAR3COLOR;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color3" );

                                data.UI_ColorTime1[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime1" );
                                data.UI_ColorTime2[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime2" );
                                data.UI_ColorTime3[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime3" );
                                data.UI_ColorRepeatNum[nw::eft::EFT_COLOR_KIND_0]       = emitter->GetAsInt( "UI_numColorRepeat" );
                            */
                        }
                        break;
                    case /*2*/"RandomColorNoTex":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ConstColor];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Random;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color0_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color0_color3 = srcEmitter.UI_color3;

                            /*
                                case 2:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_0;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color3" );
                            */
                        }
                        break;
                    case /*3*/"Modulate(no anim)":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ModulateTex];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_constColor;

                            /*
                                case 3:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_1;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_constColor" );
                             */
                        }
                        break;
                    case /*4*/"Modulate(3 colors anim)":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ConstColor];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Animation;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color0_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color0_color3 = srcEmitter.UI_color3;

                            dstEmitter.UI_color0_time1 = srcEmitter.UI_colorTime1;
                            dstEmitter.UI_color0_time2 = srcEmitter.UI_colorTime2;
                            dstEmitter.UI_color0_time3 = srcEmitter.UI_colorTime3;
                            dstEmitter.UI_color0_numColorRepeat = srcEmitter.UI_numColorRepeat;

                            /*
                                case 4:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_1;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM_LINEAR3COLOR;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color3" );

                                data.UI_ColorTime1[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime1" );
                                data.UI_ColorTime2[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime2" );
                                data.UI_ColorTime3[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime3" );
                                data.UI_ColorRepeatNum[nw::eft::EFT_COLOR_KIND_0]       = emitter->GetAsInt( "UI_numColorRepeat" );
                             */
                        }
                        break;
                    case /*5*/"Modulate(random color)":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ModulateTex];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Random;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color0_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color0_color3 = srcEmitter.UI_color3;

                            /*
                                case 5:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_1;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color3" );
                             */
                        }
                        break;
                    case /*6*/"2ColorInterpolate(no anim)":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ColorInterpolate];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_constColor;
                            dstEmitter.UI_color1_color1 = srcEmitter.UI_color1;

                            /*
                                case 6:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_2;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_constColor" );
                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color1" );
                             */
                        }
                        break;
                    case /*7*/"Inv2ColorInterpolate(3 colors anim)":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ColorInterpolate];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Animation;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color0_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color0_color3 = srcEmitter.UI_color3;

                            dstEmitter.UI_color0_time1 = srcEmitter.UI_colorTime1;
                            dstEmitter.UI_color0_time2 = srcEmitter.UI_colorTime2;
                            dstEmitter.UI_color0_time3 = srcEmitter.UI_colorTime3;
                            dstEmitter.UI_color0_numColorRepeat = srcEmitter.UI_numColorRepeat;

                            dstEmitter.UI_color1_color1 = srcEmitter.UI_constColor;

                            /*
                                case 7:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_2;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM_LINEAR3COLOR;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color3" );

                                data.UI_ColorTime1[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime1" );
                                data.UI_ColorTime2[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime2" );
                                data.UI_ColorTime3[nw::eft::EFT_COLOR_KIND_0]           = emitter->GetAsInt( "UI_colorTime3" );
                                data.UI_ColorRepeatNum[nw::eft::EFT_COLOR_KIND_0]       = emitter->GetAsInt( "UI_numColorRepeat" );

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_constColor" );
                             */
                        }
                        break;
                    case /*8*/"2ColorInterpolate(3 colors anim)":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ColorInterpolate];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Animation;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_constColor;

                            dstEmitter.UI_color1_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color1_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color1_color3 = srcEmitter.UI_color3;

                            dstEmitter.UI_color1_time1 = srcEmitter.UI_colorTime1;
                            dstEmitter.UI_color1_time2 = srcEmitter.UI_colorTime2;
                            dstEmitter.UI_color1_time3 = srcEmitter.UI_colorTime3;
                            dstEmitter.UI_color1_numColorRepeat = srcEmitter.UI_numColorRepeat;

                            /*
                                case 8:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_2;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM_LINEAR3COLOR;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_constColor" );

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color3" );

                                data.UI_ColorTime1[nw::eft::EFT_COLOR_KIND_1]           = emitter->GetAsInt( "UI_colorTime1" );
                                data.UI_ColorTime2[nw::eft::EFT_COLOR_KIND_1]           = emitter->GetAsInt( "UI_colorTime2" );
                                data.UI_ColorTime3[nw::eft::EFT_COLOR_KIND_1]           = emitter->GetAsInt( "UI_colorTime3" );
                                data.UI_ColorRepeatNum[nw::eft::EFT_COLOR_KIND_1]       = emitter->GetAsInt( "UI_numColorRepeat" );
                             */
                        }
                        break;
                    case /*9*/"Inv2ColorInterpolate(random color)":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ColorInterpolate];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Random;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color0_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color0_color3 = srcEmitter.UI_color3;

                            dstEmitter.UI_color1_color1 = srcEmitter.UI_constColor;

                            /*
                                case 9:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_2;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_color3" );

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_constColor" );
                             */
                        }
                        break;
                    case /*10*/"2ColorInterpolate(random color)":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.ConstColor];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Animation;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_constColor;

                            dstEmitter.UI_color1_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color1_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color1_color3 = srcEmitter.UI_color3;

                            /*
                                case 10:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_2;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_constColor" );

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color3" );
                             */
                        }
                        break;
                    case /*11*/"AddConst":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.AddColor];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_constColor;
                            dstEmitter.UI_color1_color1 = srcEmitter.UI_color1;

                            /*
                                case 11:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_3;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_constColor" );
                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color1" );
                             */
                        }
                        break;
                    case /*12*/"AddRandom":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.AddColor];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Random;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_constColor;

                            dstEmitter.UI_color1_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color1_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color1_color3 = srcEmitter.UI_color3;

                            /*
                                case 12:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_3;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_constColor" );

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color3" );
                             */
                        }
                        break;
                    case /*13*/"AddAnim":
                        {
                            dstEmitter.UI_colorType = ColorTypeString[(int)EsetUpdater_0_9_0_0.ColorType.AddColor];
                            dstEmitter.UI_color0_type = (int)NWCore.DataModel.Constances.ColorSettingType.Constant;
                            dstEmitter.UI_color1_type = (int)NWCore.DataModel.Constances.ColorSettingType.Animation;

                            dstEmitter.UI_color0_color1 = srcEmitter.UI_constColor;

                            dstEmitter.UI_color1_color1 = srcEmitter.UI_color1;
                            dstEmitter.UI_color1_color2 = srcEmitter.UI_color2;
                            dstEmitter.UI_color1_color3 = srcEmitter.UI_color3;

                            dstEmitter.UI_color1_time1 = srcEmitter.UI_colorTime1;
                            dstEmitter.UI_color1_time2 = srcEmitter.UI_colorTime2;
                            dstEmitter.UI_color1_time3 = srcEmitter.UI_colorTime3;
                            dstEmitter.UI_color1_numColorRepeat = srcEmitter.UI_numColorRepeat;

                            /*
                                case 13:
                                data.UI_ColorCombinerType = nw::eft::EFT_COMBINER_TYPE_3;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_0] = nw::eft::EFT_COLOR_CALC_TYPE_NONE;
                                data.UI_ColorCalcType[nw::eft::EFT_COLOR_KIND_1] = nw::eft::EFT_COLOR_CALC_TYPE_RANDOM_LINEAR3COLOR;

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_0], emitter, "UI_constColor" );

                                _copyColor( &data.UI_Color1[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color1" );
                                _copyColor( &data.UI_Color2[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color2" );
                                _copyColor( &data.UI_Color3[nw::eft::EFT_COLOR_KIND_1], emitter, "UI_color3" );

                                data.UI_ColorTime1[nw::eft::EFT_COLOR_KIND_1]           = emitter->GetAsInt( "UI_colorTime1" );
                                data.UI_ColorTime2[nw::eft::EFT_COLOR_KIND_1]           = emitter->GetAsInt( "UI_colorTime2" );
                                data.UI_ColorTime3[nw::eft::EFT_COLOR_KIND_1]           = emitter->GetAsInt( "UI_colorTime3" );
                                data.UI_ColorRepeatNum[nw::eft::EFT_COLOR_KIND_1]       = emitter->GetAsInt( "UI_numColorRepeat" );
                             */
                        }
                        break;

                    default:
                        break;
                }

                //-------------------------------------------------------------
                // TODO : Convert old data to new data model
                //-------------------------------------------------------------

                dstEmitter.EditData.EditData.Comment = string.Copy( srcEmitter.EditData.EditData.Comment );
                dstEmitter.type = (CurrVerDataModel.EmitterType)srcEmitter.type;
                dstEmitter.volumeType = (CurrVerDataModel.VolumeType)srcEmitter.volumeType;
                dstEmitter.UI_blendType = (CurrVerDataModel.BlendType)srcEmitter.UI_blendType;
                dstEmitter.UI_rotType = (CurrVerDataModel.PtclRotType)srcEmitter.UI_rotType;
                dstEmitter.billboardType = (CurrVerDataModel.BillboardType)srcEmitter.billboardType;
                dstEmitter.UI_followType = (CurrVerDataModel.FollowType)srcEmitter.UI_followType;
                dstEmitter.UI_textureAddressing = (CurrVerDataModel.TextureAddressing)srcEmitter.UI_textureAddressing;
                dstEmitter.UI_alphaDstType = (CurrVerDataModel.AlphaDst)srcEmitter.UI_alphaDstType;
                dstEmitter.zBufATestType = (CurrVerDataModel.ZBufATestType)srcEmitter.zBufATestType;
                dstEmitter.UI_texFilterMode = (CurrVerDataModel.FilterMode)srcEmitter.UI_texFilterMode;
                dstEmitter.UI_fadeOutType = (CurrVerDataModel.FadeOutType)srcEmitter.UI_fadeOutType;
                dstEmitter.UI_scale1 = ConvertVector(srcEmitter.UI_scale1);
                dstEmitter.UI_scale3 = ConvertVector(srcEmitter.UI_scale3);
                dstEmitter.UI_initRot = ConvertVector(srcEmitter.UI_initRot);
                dstEmitter.UI_initRotRand = ConvertVector(srcEmitter.UI_initRotRand);
                dstEmitter.UI_rotVel = ConvertVector(srcEmitter.UI_rotVel);
                dstEmitter.UI_rotVelRand = ConvertVector(srcEmitter.UI_rotVelRand);
                dstEmitter.UI_childBlendType = (CurrVerDataModel.BlendType)srcEmitter.UI_childBlendType;
                dstEmitter.childBillboardType = (CurrVerDataModel.BillboardType)srcEmitter.childBillboardType;
                dstEmitter.UI_childAlphaDstType = (CurrVerDataModel.AlphaDst)srcEmitter.UI_childAlphaDstType;
                dstEmitter.UI_childRotType = (CurrVerDataModel.PtclRotType)srcEmitter.UI_childRotType;
                dstEmitter.UI_childTextureAddressing = (CurrVerDataModel.TextureAddressing)srcEmitter.UI_childTextureAddressing;
                dstEmitter.stripeType = (CurrVerDataModel.StripeType)srcEmitter.stripeType;
                dstEmitter.childZBufATestType = (CurrVerDataModel.ZBufATestType)srcEmitter.childZBufATestType;
                dstEmitter.UI_childTexFilterMode = (CurrVerDataModel.FilterMode)srcEmitter.UI_childTexFilterMode;
                dstEmitter.UI_childInitRot = ConvertVector(srcEmitter.UI_childInitRot);
                dstEmitter.UI_childInitRotRand = ConvertVector(srcEmitter.UI_childInitRotRand);
                dstEmitter.UI_childRotVel = ConvertVector(srcEmitter.UI_childRotVel);
                dstEmitter.UI_childRotVelRand = ConvertVector(srcEmitter.UI_childRotVelRand);
                dstEmitter.UI_transformRot = ConvertVector(srcEmitter.UI_transformRot);

                dstData.EmitterList.Add( dstEmitter );
            }

            #endregion

            return dstData;
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Convert vector from the previous to the current version.
        /// </summary>
        /// <param name="src">The source vector of the previous version.</param>
        /// <returns>The converted current version vector.</returns>
        private CurrVerDataModel.Vector1i ConvertVector( PrevVerDataModel.Vector1i src )
        {
            return new CurrVerDataModel.Vector1i( src.X );
        }


        /// <summary>
        /// Convert vector from the previous to the current version.
        /// </summary>
        /// <param name="src">The source vector of the previous version.</param>
        /// <returns>The converted current version vector.</returns>
        private CurrVerDataModel.Vector2i ConvertVector( PrevVerDataModel.Vector2i src )
        {
            return new CurrVerDataModel.Vector2i( src.X,
                                                  src.Y );
        }


        /// <summary>
        /// Convert vector from the previous to the current version.
        /// </summary>
        /// <param name="src">The source vector of the previous version.</param>
        /// <returns>The converted current version vector.</returns>
        private CurrVerDataModel.Vector3i ConvertVector( PrevVerDataModel.Vector3i src )
        {
            return new CurrVerDataModel.Vector3i( src.X,
                                                  src.Y,
                                                  src.Z );
        }


        /// <summary>
        /// Convert vector from the previous to the current version.
        /// </summary>
        /// <param name="src">The source vector of the previous version.</param>
        /// <returns>The converted current version vector.</returns>
        private CurrVerDataModel.Vector4i ConvertVector( PrevVerDataModel.Vector4i src )
        {
            return new CurrVerDataModel.Vector4i( src.X,
                                                  src.Y,
                                                  src.Z,
                                                  src.W );
        }

        private enum ColorType
        {
            ConstColor = 0,
            ModulateTex,
            ColorInterpolate,
            AddColor,
        }

        private static string[] ColorTypeString = new string[]
        {
            /*00*/"ConstColor" ,
            /*01*/"ModulateTex" ,
            /*02*/"2ColorInterpolate" ,
            /*03*/"AddColor" ,
        };

        #endregion

        #region Member variables

        private Version m_prevVersion = new Version( 0, 0, 0, 0 );
        private Version m_currVersion = new Version( 0, 9, 0, 0 );

        #endregion
    }
}
