﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using NW4F.Common;
using NW4F.DataModel;

namespace App.Data
{
    /// <summary>
    /// ＵＩテキストクラス。
    /// </summary>
    public static class UIText
    {
        #region フラグ
        /// <summary>
        /// フラグ（On/Off）。
        /// </summary>
        public static string FlagOnOff(bool flag)
        {
            if (flag) { return StringResource.Get("FLAG_ON");  }
            else      { return StringResource.Get("FLAG_OFF"); }
        }

        /// <summary>
        /// フラグ（はい/いいえ）。
        /// </summary>
        public static string FlagYesNo(bool flag)
        {
            if (flag) { return StringResource.Get("FLAG_YES"); }
            else      { return StringResource.Get("FLAG_NO");  }
        }

        /// <summary>
        /// フラグ（有効/無効）。
        /// </summary>
        public static string FlagEnableDisable(bool flag)
        {
            if (flag) { return StringResource.Get("FLAG_ENABLE");  }
            else      { return StringResource.Get("FLAG_DISABLE"); }
        }

        /// <summary>
        /// フラグ（あり/なし）。
        /// </summary>
        public static string FlagExistOrNot(bool flag)
        {
            if (flag) { return StringResource.Get("FLAG_EXIST");    }
            else      { return StringResource.Get("FLAG_NOTEXIST"); }
        }

        /// <summary>
        /// フラグ（使用する/使用しない）。
        /// </summary>
        public static string FlagUseOrNot(bool flag)
        {
            if (flag) { return StringResource.Get("FLAG_USE");    }
            else      { return StringResource.Get("FLAG_NOTUSE"); }
        }
        #endregion

        #region 日時
        /// <summary>
        /// 日時。
        /// </summary>
        public static string Date(string date)
        {
            string result = "????????";
            try
            {
                DateTime dt = DateTime.Parse(date);
                result = dt.ToString();
            }
            catch (FormatException)
            {
            }
            return result;
        }
        #endregion

        #region 番号付きラベル
        /// <summary>
        /// 番号付きラベル。
        /// </summary>
        public static string NumberedLabel(string label, int number)
        {
            // 英語版（"label *"）
            if (TheApp.IsEnglishUI)
            {
                return string.Format("{0} {1}", label, number);
            }
            // 日本語版（"label*"）
            else
            {
                return string.Format("{0}{1}", label, number);
            }
        }
        #endregion

        #region GuiObjectID
        private static readonly string[] _strIDGuiObjectID = {
#if false
            "OBJECT_MODEL",
            "OBJECT_NODE",
            "OBJECT_POLYGON",
            "OBJECT_MATERIAL",
            "OBJECT_TEXTURE",
            "OBJECT_PALETTE",
            "OBJECT_CHARACTER_ANIMATION",
            "OBJECT_VISIBILITY_ANIMATION",
            "OBJECT_COLOR_ANIMATION",
            "OBJECT_TEXPATTERN_ANIMATION",
            "OBJECT_TEXSRT_ANIMATION",
            "OBJECT_SHAPE_ANIMATION",
            "OBJECT_SCENE",
            "OBJECT_CAMERA",
            "OBJECT_AMBIENTLIGHT",
            "OBJECT_LIGHT",
            "OBJECT_LIGHTSET",
            "OBJECT_FOG",
#endif
            "OBJECT_MODEL",
            "OBJECT_MATERIAL",
            "OBJECT_TEXTURE",
            "OBJECT_EMITTERSET",
            "OBJECT_EMITTER",
            "OBJECT_CHILD",
            "OBJECT_EFFECTPROJECT",
            "OBJECT_FOLDER",
            "OBJECT_BONELINKAGE",
            "OBJECT_FIELD",
            "OBJECT_FLUCTUATION",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(GuiObjectID value)
        {
            DebugAssertCheckArrayToEnum(_strIDGuiObjectID, value.GetType());
            return StringResource.Get(_strIDGuiObjectID[(int)value]);
        }
        #endregion

        #region AttrCoordQuantize
        private static string[] _strIDAttrCoordQuantize = {
            "ATTR_COORDQUANTIZE_FLOAT",
            "ATTR_COORDQUANTIZE_FIXED",
            "ATTR_COORDQUANTIZE_AUTO",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrCoordQuantize value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrCoordQuantize, value.GetType());
            return StringResource.Get(_strIDAttrCoordQuantize[(int)value]);
        }
        #endregion

        #region AttrNormalQuantize
        private static string[] _strIDAttrNormalQuantize = {
            "ATTR_NORMALQUANTIZE_FLOAT",
            "ATTR_NORMALQUANTIZE_FIXED6",
            "ATTR_NORMALQUANTIZE_FIXED14",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrNormalQuantize value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrNormalQuantize, value.GetType());
            return StringResource.Get(_strIDAttrNormalQuantize[(int)value]);
        }
        #endregion

        #region AttrColorType
        private static string[] _strIDAttrColorType = {
            "ATTR_COLORTYPE_RGB565",
            "ATTR_COLORTYPE_RGB8",
            "ATTR_COLORTYPE_RGBX8",
            "ATTR_COLORTYPE_RGBA4",
            "ATTR_COLORTYPE_RGBA6",
            "ATTR_COLORTYPE_RGBA8",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrColorType value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrColorType, value.GetType());
            return StringResource.Get(_strIDAttrColorType[(int)value]);
        }
        #endregion

        #region AttrVertexUse
        private static string[] _strIDAttrVertexUse = {
            "ATTR_VERTEXUSE_STANDARD",
            "ATTR_VERTEXUSE_BASESHAPE",
            "ATTR_VERTEXUSE_KEYSHAPE",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrVertexUse value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrVertexUse, value.GetType());
            return StringResource.Get(_strIDAttrVertexUse[(int)value]);
        }
        #endregion

        #region AttrScalingRule
        private static string[] _strIDAttrScalingRule = {
            "ATTR_SCALINGRULE_STD",
            "ATTR_SCALINGRULE_SI",
            "ATTR_SCALINGRULE_MAYA",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrScalingRule value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrScalingRule, value.GetType());
            return StringResource.Get(_strIDAttrScalingRule[(int)value]);
        }
        #endregion

        #region AttrTexMatrixMode
        private static string[] _strIDAttrTexMatrixMode = {
            "ATTR_TEXMATRIXMODE_MAYA",
            "ATTR_TEXMATRIXMODE_XSI",
            "ATTR_TEXMATRIXMODE_MAX",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrTexMatrixMode value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrTexMatrixMode, value.GetType());
            return StringResource.Get(_strIDAttrTexMatrixMode[(int)value]);
        }
        #endregion

        #region AttrBillboard
        private static string[] _strIDAttrBillboard = {
            "ATTR_BILLBOARD_OFF",
            "ATTR_BILLBOARD_STD",
            "ATTR_BILLBOARD_PERSPSTD",
            "ATTR_BILLBOARD_ROT",
            "ATTR_BILLBOARD_PERSPROT",
            "ATTR_BILLBOARD_Y",
            "ATTR_BILLBOARD_PERSPY",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrBillboard value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrBillboard, value.GetType());
            return StringResource.Get(_strIDAttrBillboard[(int)value]);
        }
        #endregion

        #region AttrPrimitiveType
        private static string[] _strIDAttrPrimitiveType = {
            "ATTR_PRIMITIVETYPE_POINTS",
            "ATTR_PRIMITIVETYPE_LINES",
            "ATTR_PRIMITIVETYPE_LINESTRIP",
            "ATTR_PRIMITIVETYPE_TRIANGLES",
            "ATTR_PRIMITIVETYPE_TRIANGLEFAN",
            "ATTR_PRIMITIVETYPE_TRIANGLESTRIP",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrPrimitiveType value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrPrimitiveType, value.GetType());
            return StringResource.Get(_strIDAttrPrimitiveType[(int)value]);
        }
        #endregion

        #region AttrDisplayFace
        private static string[] _strIDAttrDisplayFace = {
            "ATTR_DISPLAYFACE_FRONT",
            "ATTR_DISPLAYFACE_BACK",
            "ATTR_DISPLAYFACE_BOTH",
            "ATTR_DISPLAYFACE_NONE",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrDisplayFace value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrDisplayFace, value.GetType());
            return StringResource.Get(_strIDAttrDisplayFace[(int)value]);
        }
        #endregion

        #region AttrTransparencyMode
        private static readonly string[] _strIDAttrTransparencyMode = {
            "ATTR_TRANSPARENCYMODE_OPA",
            "ATTR_TRANSPARENCYMODE_XLU",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrTransparencyMode value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrTransparencyMode, value.GetType());
            return StringResource.Get(_strIDAttrTransparencyMode[(int)value]);
        }
        #endregion

        #region AttrColorAssignTarget
        private static readonly string[] _strIDAttrColorAssignTarget = {
            "ATTR_COLORASSIGNTARGET_MAT0_RGB",
            "ATTR_COLORASSIGNTARGET_MAT0_A",
            "ATTR_COLORASSIGNTARGET_MAT1_RGB",
            "ATTR_COLORASSIGNTARGET_MAT1_A",
            "ATTR_COLORASSIGNTARGET_AMB0_RGB",
            "ATTR_COLORASSIGNTARGET_AMB0_A",
            "ATTR_COLORASSIGNTARGET_AMB1_RGB",
            "ATTR_COLORASSIGNTARGET_AMB1_A",
            "ATTR_COLORASSIGNTARGET_TEVC0_RGB",
            "ATTR_COLORASSIGNTARGET_TEVC0_A",
            "ATTR_COLORASSIGNTARGET_TEVC1_RGB",
            "ATTR_COLORASSIGNTARGET_TEVC1_A",
            "ATTR_COLORASSIGNTARGET_TEVC2_RGB",
            "ATTR_COLORASSIGNTARGET_TEVC2_A",
            "ATTR_COLORASSIGNTARGET_TEVK0_RGB",
            "ATTR_COLORASSIGNTARGET_TEVK0_A",
            "ATTR_COLORASSIGNTARGET_TEVK1_RGB",
            "ATTR_COLORASSIGNTARGET_TEVK1_A",
            "ATTR_COLORASSIGNTARGET_TEVK2_RGB",
            "ATTR_COLORASSIGNTARGET_TEVK2_A",
            "ATTR_COLORASSIGNTARGET_TEVK3_RGB",
            "ATTR_COLORASSIGNTARGET_TEVK3_A",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrColorAssignTarget value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrColorAssignTarget, value.GetType());
            return StringResource.Get(_strIDAttrColorAssignTarget[(int)value]);
        }
        #endregion

        #region AttrImageFormat
        private static string[] _strIDAttrImageFormat = {
            "ATTR_IMAGEFORMAT_I4",
            "ATTR_IMAGEFORMAT_IA4",
            "ATTR_IMAGEFORMAT_I8",
            "ATTR_IMAGEFORMAT_IA8",
            "ATTR_IMAGEFORMAT_RGB565",
            "ATTR_IMAGEFORMAT_RGB5A3",
            "ATTR_IMAGEFORMAT_RGBA8",
            "ATTR_IMAGEFORMAT_CMPR",
            "ATTR_IMAGEFORMAT_C4",
            "ATTR_IMAGEFORMAT_C8",
            "ATTR_IMAGEFORMAT_C14",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrImageFormat value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrImageFormat, value.GetType());
            return StringResource.Get(_strIDAttrImageFormat[(int)value]);
        }
        #endregion

        #region AttrPaletteFormat
        private static string[] _strIDAttrPaletteFormat = {
            "ATTR_PALETTEFORMAT_IA8",
            "ATTR_PALETTEFORMAT_RGB565",
            "ATTR_PALETTEFORMAT_RGB5A3",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrPaletteFormat value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrPaletteFormat, value.GetType());
            return StringResource.Get(_strIDAttrPaletteFormat[(int)value]);
        }
        #endregion

        #region AttrCameraType
        private static string[] _strIDAttrCameraType = {
            "ATTR_CAMERATYPE_ROTATE",
            "ATTR_CAMERATYPE_AIM",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrCameraType value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrCameraType, value.GetType());
            return StringResource.Get(_strIDAttrCameraType[(int)value]);
        }
        #endregion

        #region AttrProjectionType
        private static readonly string[] _strIDAttrProjectionType = {
            "ATTR_PROJECTIONTYPE_ORTHOGRAPHIC",
            "ATTR_PROJECTIONTYPE_PERSPECTIVE",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrProjectionType value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrProjectionType, value.GetType());
            return StringResource.Get(_strIDAttrProjectionType[(int)value]);
        }
        #endregion

        #region AttrLightType
        private static readonly string[] _strIDAttrLightType = {
            "ATTR_LIGHTTYPE_POINT",
            "ATTR_LIGHTTYPE_DIRECTIONAL",
            "ATTR_LIGHTTYPE_SPOT",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrLightType value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrLightType, value.GetType());
            return StringResource.Get(_strIDAttrLightType[(int)value]);
        }
        #endregion

        #region AttrSpotFunc
        private static readonly string[] _strIDAttrSpotFunc = {
            "ATTR_SPOTFUNC_OFF",
            "ATTR_SPOTFUNC_FLAT",
            "ATTR_SPOTFUNC_COS",
            "ATTR_SPOTFUNC_COS2",
            "ATTR_SPOTFUNC_SHARP",
            "ATTR_SPOTFUNC_RING1",
            "ATTR_SPOTFUNC_RING2",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrSpotFunc value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrSpotFunc, value.GetType());
            return StringResource.Get(_strIDAttrSpotFunc[(int)value]);
        }
        #endregion

        #region AttrDistFunc
        private static readonly string[] _strIDAttrDistFunc = {
            "ATTR_DISTFUNC_OFF",
            "ATTR_DISTFUNC_GENTLE",
            "ATTR_DISTFUNC_MEDIUM",
            "ATTR_DISTFUNC_STEEP",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrDistFunc value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrDistFunc, value.GetType());
            return StringResource.Get(_strIDAttrDistFunc[(int)value]);
        }
        #endregion

        #region AttrFogType
        private static readonly string[] _strIDAttrFogType = {
            "ATTR_FOGTYPE_NONE",
            "ATTR_FOGTYPE_LIN",
            "ATTR_FOGTYPE_EXP",
            "ATTR_FOGTYPE_EXP2",
            "ATTR_FOGTYPE_REVEXP",
            "ATTR_FOGTYPE_REVEXP2",
        };

        /// <summary>
        /// 列挙値。
        /// </summary>
        public static string EnumValue(AttrFogType value)
        {
            DebugAssertCheckArrayToEnum(_strIDAttrFogType, value.GetType());
            return StringResource.Get(_strIDAttrFogType[(int)value]);
        }
        #endregion

        /// <summary>
        /// 配列要素数と列挙定義数の対応確認。
        /// </summary>
        [Conditional("DEBUG")]
        private static void DebugAssertCheckArrayToEnum(Array array, Type enumType)
        {
            Debug.Assert(
                array.Length == Enum.GetValues(enumType).Length,
                string.Format("{0} 型と対応する配列要素数が合いません。", enumType.FullName)
            );
        }
    }
}
