﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//#define TM_LOGGING

using System;
using System.Linq;
using System.IO;
using System.Windows.Forms;

using NWCore.DataModel;

namespace App.Data
{
    #region Enumerator for the results of the texture loading

    /// <summary>
    /// Enumerator for the results of the texture loading.
    /// </summary>
    public enum LoadTextureResults
    {
        Success,

        ErrCannotAccessFile,
        ErrFileNotFound,
        ErrInvalidFilePath,
        ErrNW4FRootNotFound,
        ErrFailLoadingFTX,
        ErrUnknowPixelFormat,
        ErrFailLoadingTGA,
        ErrUnknowTexType,

        ErrUnknown
    }

    #endregion

    /// <summary>
    /// Class for managing the textures used in the application.
    /// </summary>
    public class TextureManager
    {
        #region Logger

        private class Logger
        {
            private static readonly string logFile = TheApp.ApplicationPath + "\\TextureManager.log";
            private static readonly object threadLock = new object();
            private static DateTime start;
            private static int NextId = 0;
            private int id;

            public Logger()
            {
                lock (threadLock)
                {
                    if (NextId == 0)
                    {
                        start = DateTime.Now;
                        if (File.Exists(logFile))
                            File.Delete(logFile);
                    }

                    id = NextId++;
                    using (var sw = new StreamWriter(logFile, true))
                    {
                        sw.Write(string.Format("{0:000}  Logging start date: {1:yyyy-MM-dd HH:mm:ss.fff}" + Environment.NewLine, id, DateTime.Now));
                    }
                }
            }

            private void Write(string info)
            {
                lock (threadLock)
                {
                    if (this == null)
                        return;

                    using (var sw = new StreamWriter(logFile, true))
                    {
                        sw.Write(string.Format("{0:000}  {1:dd HH:mm:ss.fff}: {2}" + Environment.NewLine, id, DateTime.Now, info));
                    }
                }
            }

            public static void Write(Logger logger, string info)
            {
                if (logger != null) logger.Write(info);
            }

            public static string DumpException(Exception ex)
            {
                if (ex == null)
                    throw new ArgumentNullException("ex");
                return DumpExceptionRecursive(ex, 0);
            }

            private static string DumpExceptionRecursive(Exception ex, int level)
            {
                var sb = new System.Text.StringBuilder();

                var indent = new string(' ', level * 4);

                sb.AppendLine(string.Format("{0}{1}: {2}", indent, "Type", ex.GetType().FullName));
                sb.AppendLine(string.Format("{0}{1}: {2}", indent, "Message", ex.Message));

                if (ex.Data.Keys.Count > 0)
                {
                    sb.AppendLine(string.Format("{0}{1}:", indent, "Data"));
                    foreach (var key in ex.Data.Keys)
                    {
                        sb.AppendLine(string.Format("{0}  {1}: {2}, {3}: {4}", indent,
                            "Key", key,
                            "Value", ex.Data[key]));
                    }
                }

                if (ex is System.Reflection.ReflectionTypeLoadException)
                {
                    foreach (var loaderException in ((System.Reflection.ReflectionTypeLoadException)ex).LoaderExceptions)
                        sb.AppendLine(DumpExceptionRecursive(loaderException, level + 1));
                }
                else if (ex is AggregateException)
                {
                    foreach (var innerException in ((AggregateException)ex).InnerExceptions)
                        sb.AppendLine(DumpExceptionRecursive(innerException, level + 1));
                }

                if (ex.StackTrace != null)
                {
                    sb.AppendLine(string.Format("{0}{1}:", indent, "Call stack"));
                    DumpStackTrace(level, ex.StackTrace, sb);
                }

                if (ex.InnerException != null)
                {
                    sb.AppendLine();
                    sb.AppendLine(DumpExceptionRecursive(ex.InnerException, level + 1));
                }

                return sb.ToString();
            }

            private static void DumpStackTrace(int level, string stackTrace, System.Text.StringBuilder sb)
            {
                var indent = new string(' ', 2 + level * 4);

                foreach (var line in stackTrace.Split('\n').Select(l => l.Trim()))
                    sb.AppendLine(string.Format("{0}{1}", indent, line));
            }
        }

        #endregion

        #region Class for texture loading message box block

        /// <summary>
        /// Class for showing warning message boxes when texture loading.
        /// </summary>
        public sealed class TextureLoadingMsgBoxBlock : IDisposable
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            public TextureLoadingMsgBoxBlock()
            {
                m_bOrigFlag = TheApp.TextureManager.m_bShowLoadTexWarningMsgBox;

                TheApp.TextureManager.m_bShowLoadTexWarningMsgBox = true;
            }


            /// <summary>
            /// Dispose.
            /// </summary>
            public void Dispose()
            {
                TheApp.TextureManager.m_bShowLoadTexWarningMsgBox = m_bOrigFlag;
            }


            private bool m_bOrigFlag = false;
        }

        #endregion


        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public TextureManager()
        {
#if TM_LOGGING
                logger = new Logger();
#endif
        }

        #endregion

        #region Properties

        /// <summary>
        /// Get or set the flag indicating whether to reload
        /// the cached texture if it was modified.
        /// </summary>
        public bool IsCheckModTimeEnabled
        {
            get { return m_bCheckModTime; }
            set { m_bCheckModTime = value; }
        }

        /// <summary>
        /// Get or set the maximum size of all the texture cache data.
        /// </summary>
        public long MaxTextureCacheSize
        {
            get { return m_iMaxCacheSize; }
            set
            {
                m_iMaxCacheSize = value;
                CheckCacheSizeLimit();
            }
        }

        #endregion

        #region Texture cache data


        /// <summary>
        /// Check if the cached texture data already reached the size limits.
        /// Remove old cache data if it does.
        /// </summary>
        private void CheckCacheSizeLimit()
        {
        }


        /// <summary>
        /// Update the total texture cache size.
        /// </summary>
        private void UpdateTotalCacheSize()
        {
        }


        /// <summary>
        /// Output texture cache contents for debugging.
        /// </summary>
        public void DebugOutputTextureCache()
        {
            #if DEBUG

            DebugConsole.WriteLine( "========== TextureManager Cache Output ==========" );


            #endif
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Determine if the image of the given image format is already gamma corrected.
        /// </summary>
        /// <param name="imageFormat">The image format.</param>
        /// <returns>True if gamma corrected.</returns>
        public bool IsGammaCorrectedImage( string imageFormat )
        {
            if ( m_invGammaCorrImgFormatCRCs==null )
                return false;

            uint iFormatCRC = TheApp.CRC32Helper.ComputeCRC32Str( imageFormat );

            foreach ( uint iNeededFormat in m_invGammaCorrImgFormatCRCs )
            {
                if ( iFormatCRC==iNeededFormat )
                    return true;
            }

            return false;
        }

        /// <summary>
        /// 編集モードとテクスチャタイプが一致しているかの結果です
        /// </summary>
        public enum TextureTypeCheckResult
        {
            /// <summary>
            /// 編集モードとテクスチャタイプが一致しています
            /// </summary>
            OK,

            /// <summary>
            /// 編集モードはリニアではありませんがテクスチャがリニアです
            /// </summary>
            LinearTexture,

            /// <summary>
            /// 編集モードはリニアですがテクスチャがリニアではありません
            /// </summary>
            NonlinearTexture,

            /// <summary>
            /// The texture to be checked has not been loaded.
            /// </summary>
            TextureNotLoaded
        };

        /// <summary>
        /// Check if the texture linear flags matches the application linear edit mode settings.
        /// </summary>
        /// <param name="texInfo">The texture information.</param>
        /// <param name="bShowDialog">
        /// True to show the warning dialog if the linear mode is different from
        /// the application settings.
        /// </param>
        /// <returns>The check result.</returns>
        public TextureTypeCheckResult CheckTextureLinearEditMode( TextureInfo texInfo,
                                                                  bool bShowDialog )
        {
            if ( texInfo==null )
                return TextureTypeCheckResult.OK;

            // Is any of the linear flag of the texture true?
            bool bLinearTexture = texInfo.Linear != null && texInfo.Linear.Any(x => x);

            // Is the texture any of the srgb pixel formats?
            if ( TheApp.TextureManager.IsGammaCorrectedImage(texInfo.NativeDataFormat)==true )
                bLinearTexture = true;

            TextureTypeCheckResult result = TextureTypeCheckResult.OK;

            // Is the linear edit mode settings different from the texture?
            if ( bLinearTexture!=TheApp.IsGammaCorrectionEnabled )
            {
                string msg;
                if ( bLinearTexture==true )
                {
                    msg = res.Strings.WARNING_TEXTURE_TYPE_MISMATCH + "\n" + res.Strings.WARNING_TEXTURE_LINEAR;
                    result = TextureTypeCheckResult.LinearTexture;
                }
                else
                {
                    msg = res.Strings.WARNING_TEXTURE_TYPE_MISMATCH + "\n" + res.Strings.WARNING_TEXTURE_NONLINEAR;
                    result = TextureTypeCheckResult.NonlinearTexture;
                }

                if ( TheApp.ConsoleMode==false ||
                     TheApp.EnableConsoleModeWarningDialogs==true )
                {
                }
                else
                {
                    TheApp.OutputLogMsg( NWCore.LogLevels.Warning, msg );
                }
            }

            return result;
        }

        #endregion

        #region Events

        /// <summary>
        /// Delegation for document event handler.
        /// </summary>
        /// <param name="document">The document for the event.</param>
        public delegate void TextureEventHandler(TextureInfo textureInfo);

        /// <summary>
        /// Event triggered when a texture is loaded being created.
        /// </summary>
        public event TextureEventHandler TextureLoaded = null;

        private void NotifyTextureLoaded(TextureInfo texInfo)
        {
            var handler = TextureLoaded;
            if (handler != null) handler(texInfo);
        }

        #endregion

        #region Member variables

        private bool               m_bCheckModTime  = false;

        private bool               m_bShowLoadTexWarningMsgBox = false;

        private long               m_iMaxCacheSize   = -1; // No limitations by default.

        static private object s_SyncObject = new object();

        private uint[] m_invGammaCorrImgFormatCRCs = null;

        #endregion
    }
}
