﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;
using System.Xml.Serialization;

using NWCore.Serializer;

using App.IO;
using System.Runtime.Serialization.Formatters.Binary;

namespace App.Data
{
    /// <summary>
    /// Enum of preset template types.
    /// </summary>
    public enum PresetTypes
    {
        /// <summary>
        /// All types of presets.
        /// </summary>
        All = -1,

        /// <summary>
        /// Emitter set preset.
        /// Sort key as 0.
        /// </summary>
        EmitterSet = 0,

        /// <summary>
        /// Emitter preset.
        /// Sort key as 1.
        /// </summary>
        Emitter = 1,
    }

    /// <summary>
    /// Class to hold preset template data.
    /// </summary>
    public class PresetTemplateData
    {
        /// <summary>Child template data list.</summary>
        private List<PresetTemplateData> children = new List<PresetTemplateData>();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="type">The preset template type.</param>
        /// <param name="displayName">The display name of the preset template.</param>
        /// <param name="name">The document name of the preset template.</param>
        /// <param name="file">The file info of the preset template.</param>
        public PresetTemplateData(
            PresetTypes type,
            string name,
            FileInfo file)
        {
            this.ID = (file.FullName + "_" + type.ToString() + "_" + name).ToLower().GetHashCode();
            this.Type = type;
            this.Name = name;
            this.FilePath = file.FullName.ToLower();
            this.FileModifyTime = file.LastWriteTimeUtc;
        }

        /// <summary>Get the ID of the preset.</summary>
        public int ID { get; private set; }

        /// <summary>Get the type of the preset template.</summary>
        public PresetTypes Type { get; private set; }

        /// <summary>Get the document name of the preset template.</summary>
        public string Name { get; private set; }

        /// <summary>Get or set the file path of the template.</summary>
        public string FilePath { get; private set; }

        /// <summary>Get or set the last modify time of the template file.</summary>
        public DateTime FileModifyTime { get; private set; }

        /// <summary>Enumerate child preset data.</summary>
        public IEnumerable<PresetTemplateData> Children
        {
            get { return this.children; }
        }

        /// <summary>Get the number of children.</summary>
        public int ChildCount
        {
            get { return this.children.Count; }
        }

        /// <summary>
        /// Add the child preset data to the list.
        /// </summary>
        /// <param name="child">The child preset data.</param>
        public void AddChild(PresetTemplateData child)
        {
            this.children.Add(child);
        }

        /// <summary>
        /// Sort the child list.
        /// </summary>
        public void SortChildren()
        {
            // Sort the cache list.
            this.children.Sort((x, y) =>
            {
                if (x.Type > y.Type)
                {
                    return 1;
                }
                else if (x.Type < y.Type)
                {
                    return -1;
                }
                else
                {
                    return string.Compare(x.Name, y.Name);
                }
            });
        }
    }

    /// <summary>
    /// Static class that manages preset template files.
    /// </summary>
    public static class PresetTemplateManager
    {
        /// <summary>The preset data list.</summary>
        private static List<PresetTemplateData> presetList =
            new List<PresetTemplateData>();

        /// <summary>
        /// Class holds preset file data used for updating the cached preset list.
        /// </summary>
        private class PresetFileData
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="file">The file info of the preset file.</param>
            public PresetFileData(FileInfo info)
            {
                this.Info = info;
                this.NeedUpdate = true;
            }

            /// <summary>Get the file info of the preset file.</summary>
            public FileInfo Info { get; private set; }

            /// <summary>Get the flag indicating whether should load the preset.</summary>
            public bool NeedUpdate { get; set; }
        }

        /// <summary>
        /// Get a list of all the preset templates.
        /// </summary>
        /// <returns>The preset template list.</returns>
        public static IEnumerable<PresetTemplateData> GetPresetList()
        {
            string dirPath = TheApp.PresetFolderPath;
            string fileFilter = "*" + DocumentConstants.DotEset;

            if (string.IsNullOrEmpty(dirPath) == true ||
                Path.IsPathRooted(dirPath) == false ||
                Directory.Exists(dirPath) == false)
            {
                return Enumerable.Empty<PresetTemplateData>();
            }

            // First get all the file infos in the preset folder.
            Dictionary<string, PresetFileData> files = null;
            {
                IEnumerable<string> filesInFolder = Directory.GetFiles(dirPath, fileFilter);
                if (filesInFolder == null)
                {
                    return Enumerable.Empty<PresetTemplateData>();
                }

                // First convert the file paths to FileInfos.
                IEnumerable<FileInfo> tmp =
                    filesInFolder.Select(path => new FileInfo(path));

                // Make a dictionary of the file path and modify time to speed up the process.
                files = tmp.ToDictionary(
                    item => item.FullName.ToLower(),
                    item => new PresetFileData(item));
            }

            // First remove the presets that is out-dated or removed from the preset folder.
            for (int i = PresetTemplateManager.presetList.Count - 1; i >= 0; --i)
            {
                PresetTemplateData data = PresetTemplateManager.presetList[i];

                // Remove the cached preset data if these conditions are met:
                // 1. The cached preset file path is not found in the file list.
                // 2. The modify time is different.
                PresetFileData file;
                if ((files.TryGetValue(data.FilePath, out file) == false) ||
                    (file.Info.LastWriteTimeUtc.Equals(data.FileModifyTime) == false))
                {
                    PresetTemplateManager.presetList.RemoveAt(i);
                }
                else
                {
                    // The file is cached correctly, set the flag so it won't be loaded again.
                    file.NeedUpdate = false;
                }
            }

            // Load the files that has not been cached or out-dated.
            foreach (PresetFileData file in files.Values)
            {
                if (file.NeedUpdate == false)
                {
                    // The preset file doesn't need to be reloaded.
                    continue;
                }

                // Load the emitter set file as an XML document.
                var doc = new XmlDocument();
                try
                {
                    doc.Load(file.Info.FullName);
                }
                catch
                {
                    // Failed loading the emitter set, skip it.
                    continue;
                }

                // Get the root document element.
                XmlElement root = doc.DocumentElement;
                if (root == null)
                {
                    // Failed loading the emitter set, skip it.
                    continue;
                }

                string esetName = Path.GetFileNameWithoutExtension(file.Info.Name);

                // Create the preset data.
                var data = new PresetTemplateData(
                    PresetTypes.EmitterSet,
                    esetName,
                    file.Info);

                PresetTemplateManager.presetList.Add(data);

                // Get the emitter list node.
                var emitterListNode = root.SelectSingleNode("EmitterList") as XmlElement;
                if (emitterListNode == null)
                {
                    continue;
                }

                // Parse the emitters in the emitter set.
                foreach (XmlNode emitterNode in root.GetElementsByTagName("Emitter"))
                {
                    var emitterElement = emitterNode as XmlElement;
                    if (emitterElement == null)
                    {
                        continue;
                    }

                    // Get the name node of the emitter element.
                    XmlNode emitterNameNode = emitterElement.SelectSingleNode("name");
                    if (emitterNameNode == null)
                    {
                        continue;
                    }

                    var emitterPresetData = new PresetTemplateData(
                        PresetTypes.Emitter,
                        emitterNameNode.InnerText,
                        file.Info);

                    // Add as a child.
                    data.AddChild(emitterPresetData);
                }

                // Sort the child list of the emitter set preset data.
                data.SortChildren();
            }

            // Sort the cache list.
            PresetTemplateManager.presetList.Sort((x, y) =>
                {
                    if (x.Type > y.Type)
                    {
                        return 1;
                    }
                    else if (x.Type < y.Type)
                    {
                        return -1;
                    }
                    else
                    {
                        return string.Compare(x.Name, y.Name);
                    }
                });

            // Return the information of the presets.
            return PresetTemplateManager.presetList;
        }
    }
}
