﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Reflection;
using System.ComponentModel;
using System.Threading;

using NintendoWare.ToolDevelopmentKit;
using NWCore.DataModel;
using NWCore.Utility;

namespace App.Data
{
    #region Document event handler

    /// <summary>
    /// Delegation for document event handler.
    /// </summary>
    /// <param name="document">The document for the event.</param>
    public delegate void DocumentEventHandler( IDocument document );

    #endregion

    /// <summary>
    /// ドキュメントマネージャクラス。
    /// </summary>
    public static class DocumentManager
    {
        #region Class for storing the modification info of the data sources

        /// <summary>
        /// Class for storing the modification info of the data sources.
        /// </summary>
        private class DataModifyInfo
        {
            #region Constructor

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="originalData">The data before any modification.</param>
            public DataModifyInfo( object originalData )
            {
                this.OriginalData = originalData;
                this.Ignore       = 0;
            }

            #endregion

            #region Properties

            /// <summary>The original data.</summary>
            public object OriginalData
            {
                get { return m_originalData; }
                set
                {
                    if ( value is ICloneable )
                        this.m_originalData = (value as ICloneable).Clone();
                    else
                        this.m_originalData = value;
                }
            }


            /// <summary>The last modified data.</summary>
            public object LastModifiedData
            {
                get { return m_lastModifiedData; }
                set
                {
                    if ( value is ICloneable )
                        this.m_lastModifiedData = (value as ICloneable).Clone();
                    else
                        this.m_lastModifiedData = value;
                }
            }


            /// <summary>The flag specifying if the data source is modified.</summary>
            public bool IsModified
            {
                get { return m_bModified; }
                set
                {
                    m_bModified = value;
                }
            }


            /// <summary>
            /// Flag to determine if ignore this node or not when clearing the
            /// modification flags.
            /// </summary>
            public int Ignore { get; set; }

            #endregion

            #region Member variables

            private object m_originalData     = null;
            private object m_lastModifiedData = null;
            private bool   m_bModified        = false;

            #endregion
        }

        #endregion

        #region Properties

        /// <summary>
        /// ドキュメントリスト。
        /// </summary>
        public static ReadOnlyList<IDocument> Documents
        {
            get { return new ReadOnlyList<IDocument>( s_documents ); }
        }

        /// <summary>
        /// The file path of the active document.
        /// </summary>
        public static string ActiveDocumentPath
        {
            get
            {
                if ( s_activeFileDoc==null )
                    return string.Empty;

                return s_activeFileDoc.FileLocation;
            }
        }

        /// <summary>
        /// Flag indicating whether suppressing document events.
        /// </summary>
        public static bool IsSuppressingDocumentEvents
        {
            get { return s_bSuppressDocEvents; }
            set { s_bSuppressDocEvents = value; }
        }

#if false //-> ProjectManager.ActiveDocumentへ移行しました。
        /// <summary>
        /// 選択ドキュメント。
        /// </summary>
        public static IDocument ActiveDocument
        {
            get { return s_activeDocument; }
            set { }
        }
#endif

        #endregion

        #region Initialization

        /// <summary>
        /// 初期化処理。
        /// </summary>
        public static void Initialize()
        {
            s_documents.Clear();
            s_activeFileDoc  = null;
        }

        #endregion

        #region Methods for data modification flags

        /// <summary>
        /// Check if the data pointed by the given data source path is modified.
        /// </summary>
        /// <param name="dataSrcPath">The data source path.</param>
        /// <returns>True if the data is modified.</returns>
        public static bool IsDataModified( string dataSrcPath )
        {
            return IsDataModified( new Path( dataSrcPath ) );
        }


        /// <summary>
        /// Check if the data pointed by the given data source path is modified.
        /// </summary>
        /// <param name="dataSrcPath">The data source path.</param>
        /// <returns>True if the data is modified.</returns>
        public static bool IsDataModified( Path dataSrcPath )
        {
            // Find the tree node, see if it's modified.
            PathTreeNode node = s_modificationDataSet.FindPathData( dataSrcPath, false );
            if ( node==null )
                return false;

            // Is this data source modified?
            DataModifyInfo info = node.Data as DataModifyInfo;
            if ( info!=null &&
                 info.IsModified==true )
            {
                return true;
            }

            // Traverse through all the children and see if they are modified.
            PathTreeNode currChild = s_modificationDataSet.Traverse( node, null );
            while ( currChild!=null &&
                    currChild!=node )
            {
                // Is this child data modified?
                info = currChild.Data as DataModifyInfo;
                if ( info!=null &&
                     info.IsModified==true )
                {
                    return true;
                }

                // Get to the next child node.
                currChild = s_modificationDataSet.Traverse( node, currChild );
            }

            return false;
        }


        /// <summary>
        /// Notify the document manager that the given data source has been
        /// touched. Document manager needs to check if the data source is
        /// modified, and set the flag accordingly.
        /// </summary>
        /// <param name="dataSrcPath">The path to the data source.</param>
        /// <param name="oldValue">The value originally set to the data source.</param>
        /// <param name="newValue">The value about to set to the data source.</param>
        public static void NotifyDataSrcTouched( string dataSrcPath,
                                                 object oldValue,
                                                 object newValue )
        {
            NotifyDataSrcTouched( new Path( dataSrcPath ),
                                  oldValue,
                                  newValue );
        }


        /// <summary>
        /// Notify the document manager that the given data source has been
        /// touched. Document manager needs to check if the data source is
        /// modified, and set the flag accordingly.
        /// </summary>
        /// <param name="dataSrcPath">The path to the data source.</param>
        /// <param name="oldValue">The value originally set to the data source.</param>
        /// <param name="newValue">The value about to set to the data source.</param>
        public static void NotifyDataSrcTouched( Path dataSrcPath,
                                                 object oldValue,
                                                 object newValue )
        {
            // Get the data source first.
            object data = oldValue;

            // Find the tree node
            PathTreeNode node = s_modificationDataSet.FindPathData( dataSrcPath, false );
            if ( node==null )
            {
                DataModifyInfo newInfo = new DataModifyInfo( data );

                node = s_modificationDataSet.AddPath( dataSrcPath,
                                                      newInfo,
                                                      false );
            }

            // Get the data modification info from the tree node.
            DataModifyInfo info = node.Data as DataModifyInfo;
            if ( info==null )
            {
                // The tree node data has not been created, create it.
                info = new DataModifyInfo( data );
            }
            else
            {
                // Get the original data.
                data = info.OriginalData;
            }

            // Save the new value.
            info.LastModifiedData = newValue;

            // Check if the values are modified or not.
            bool bEqual = DocumentManager.IsEqual( data, newValue );

            // Set the flag.
            info.IsModified = !bEqual;
        }


        /// <summary>
        /// Clear the data modification flag.
        /// This method also resets the cached original value of the data source
        /// for comparing if the data source is modified.
        /// </summary>
        /// <param name="dataSrcPath">The data source path to clear the flag.</param>
        public static void ClearDataModifyFlag( string dataSrcPath )
        {
            ClearDataModifyFlag( new Path( dataSrcPath ) );
        }


        /// <summary>
        /// Clear the data modification flag.
        /// This method also resets the cached original value of the data source
        /// for comparing if the data source is modified.
        /// </summary>
        /// <param name="dataSrcPath">The data source path to clear the flag.</param>
        public static void ClearDataModifyFlag( Path dataSrcPath )
        {
            // Find the tree node, see if it's modified.
            PathTreeNode node = s_modificationDataSet.FindPathData( dataSrcPath, false );
            if ( node==null )
                return;

            // Traverse through all the children and see if they are modified.
            DataModifyInfo info;
            PathTreeNode   currNode = s_modificationDataSet.Traverse( node, null );
            while ( currNode!=null )
            {
                // Reset the flag and its original value.
                info = currNode.Data as DataModifyInfo;
                if ( info!=null &&
                     info.IsModified==true )
                {
                    // Reset the data and flag
                    info.IsModified   = false;
                    info.OriginalData = info.LastModifiedData;
                }

                // Get to the next child node.
                currNode = s_modificationDataSet.Traverse( node, currNode );
            }
        }


        /// <summary>
        /// Clear the data modification flag.
        /// This method also resets the cached original value of the data source
        /// for comparing if the data source is modified.
        /// </summary>
        /// <param name="pathList">The data source paths to clear the flag.</param>
        public static void ClearDataModifyFlag( List<Path> pathList )
        {
            PathTreeNode   rootNode = s_modificationDataSet.Root;
            PathTreeNode   currNode;
            PathTreeNode   parentNode;
            DataModifyInfo parentInfo;
            DataModifyInfo info;
            object         data;

            // First ignore all documents.
            DocumentManager.IgnoreAllDocuments();

            #region Mark the tree nodes to clear with -1

            foreach ( Path path in pathList )
            {
                currNode = s_modificationDataSet.FindPathData( path, true );
                if ( currNode==null )
                    continue;

                info = currNode.Data as DataModifyInfo;
                if ( info==null )
                {
                    currNode.Data = new DataModifyInfo( GetDataByPath(path) );
                }

                info.Ignore = -1;
            }

            #endregion

            #region Traverse the rest of the tree nodes and setup the ignore flags

            currNode = s_modificationDataSet.Traverse( rootNode, null );
            while ( currNode!=null )
            {
                // Reset the flag and its original value.
                info = currNode.Data as DataModifyInfo;
                if ( info==null )
                {
                    // Get to the next child node.
                    currNode = s_modificationDataSet.Traverse( rootNode, currNode );
                    continue;
                }

                int iIgnore = 0;

                // Find the first parent with the ignore flag
                parentNode = currNode.Parent;
                while ( parentNode!=null )
                {
                    // Get the node data.
                    parentInfo = parentNode.Data as DataModifyInfo;
                    if ( parentInfo!=null &&
                         parentInfo.Ignore!=0 )
                    {
                        iIgnore = parentInfo.Ignore;
                        break;
                    }

                    // Get the next parent.
                    parentNode = parentNode.Parent;
                }

                // Set the ignore flag inherited from the parent
                info.Ignore = iIgnore;

                // Get to the next child node.
                currNode = s_modificationDataSet.Traverse( rootNode, currNode );
            }

            #endregion

            #region Traverse the all the tree nodes and clear the flag

            currNode = s_modificationDataSet.Traverse( rootNode, null );
            while ( currNode!=null )
            {
                // Reset the flag and its original value.
                info = currNode.Data as DataModifyInfo;
                if ( info!=null )
                {
                    if ( info.Ignore==-1 &&
                         info.IsModified==true )
                    {
                        // Get the data
                        data = DocumentManager.GetDataByPath( currNode.FullPath );

                        // Reset the data and flag
                        info.IsModified   = false;
                        info.OriginalData = data;
                    }

                    // Also reset the ignore flag for next time
                    info.Ignore = 0;
                }

                // Get to the next child node.
                currNode = s_modificationDataSet.Traverse( rootNode, currNode );
            }

            #endregion
        }


        /// <summary>
        /// Private helper method to mark all documents as ignore.
        /// This method must be called before you clear the
        /// modification flags.
        /// </summary>
        private static void IgnoreAllDocuments()
        {
            PathTreeNode   rootNode = s_modificationDataSet.Root;
            PathTreeNode   currNode;
            DataModifyInfo info;
            IDocument      doc;

            currNode = s_modificationDataSet.Traverse( rootNode, null );
            while ( currNode!=null )
            {
                // Get the data
                doc = DocumentManager.GetDataByPath( currNode.FullPath ) as IDocument;

                // Get modification info
                info = currNode.Data as DataModifyInfo;
                if ( info==null )
                {
                    info = new DataModifyInfo( doc );
                    currNode.Data = info;
                }

                // Set ignore flag
                if ( doc!=null &&
                     String.IsNullOrEmpty(doc.FileExt)==false )
                {
                    info.Ignore = 1;
                }

                // Get to the next child node.
                currNode = s_modificationDataSet.Traverse( rootNode, currNode );
            }
        }

        #endregion

        #region Methods work with data source paths

        /// <summary>
        /// Register document path.
        /// </summary>
        /// <param name="doc">The document to register.</param>
        /// <param name="docPath">Data path of the document.</param>
        public static void RegisterDocumentPath( IDocument doc,
                                                 string docPath )
        {
            PathTreeNode newNode = s_documentTree.AddPath( docPath, doc, false );

            if ( newNode==null )
            {
                newNode = s_documentTree.FindPathData( docPath, false );
                if ( newNode!=null )
                {
                    newNode.Data = doc;
                }
            }
        }


        /// <summary>
        /// Unregister document path.
        /// </summary>
        /// <param name="doc">The document to unregister.</param>
        /// <param name="docPath">Data path of the document.</param>
        public static void UnregisterDocumentPath( IDocument doc,
                                                   string docPath )
        {
            s_documentTree.RemovePath( docPath, false );
        }


        /// <summary>
        /// Find the sub path to the data of the specified type.
        /// </summary>
        /// <param name="path">The path to find into.</param>
        /// <param name="dataType">The data type of the object to find.</param>
        /// <returns>The sub path to the data of the specified type, or null if not found.</returns>
        public static Path FindSubPathByDataType( Path path,
                                                  Type dataType )
        {
            if ( path.Length<=0 )
                return null;

            Path         dstPath = new Path();
            IDocument    currDoc = null;
            PathTreeNode node    = s_documentTree.Root;

            // Find in registered documents
            int i;
            for ( i=0;i<path.Length;++i )
            {
                // Get child tree node which contains a document.
                PathTreeNode child = node.FindFirstChild( path[i].NameCRC, path[i].ID );
                if ( child==null )
                    break;

                // Is this document of the right data type?
                if ( child.Data!=null &&
                     dataType.Equals(child.Data.GetType())==true )
                {
                    dstPath.Copy( path, i+1 );
                    return dstPath;
                }

                // Remember the document for later.
                currDoc = child.Data as IDocument;

                // Step in to the child node.
                node = child;
            }

            if ( i>=path.Length )
                return null;

            if ( currDoc==null )
                return null;

            System.Reflection.PropertyInfo propInfo = null;

            // Use the data to find the object with the rest of the path.
            object currObj = currDoc;
            for ( ;i<path.Length;++i )
            {
                if ( dataType.Equals( currObj.GetType() )==true )
                {
                    dstPath.Copy( path, i+1 );
                    return dstPath;
                }

                // Get the property from the object type
                propInfo = currObj.GetType().GetProperty( path[i].Name );
                if ( propInfo==null )
                    return null;

                currObj = propInfo.GetValue( currObj, null );
                if ( currObj==null )
                    return null;
            }

            if ( dataType.Equals( currObj.GetType() )==true )
                return path.Clone();

            return null;
        }


        /// <summary>
        /// Utility method for finding the last document in the given path.
        /// </summary>
        /// <param name="path">The data source path.</param>
        /// <returns>The last document found in the path.</returns>
        public static Document FindLastDocumentInPath( Path path )
        {
            if ( path.Length<=0 )
                return null;

            Document     currDoc = null;
            PathTreeNode node    = s_documentTree.Root;

            // Find in registered documents
            int i;
            for ( i=0;i<path.Length;++i )
            {
                // Get child tree node which contains a document.
                PathTreeNode child = node.FindFirstChild( path[i].NameCRC, path[i].ID );
                if ( child==null )
                    break;

                // Remember the document for later.
                currDoc = child.Data as Document;

                // Step in to the child node.
                node = child;
            }

            if ( i>=path.Length )
                return currDoc;

            if ( currDoc==null )
                return null;

            TypeDescriptionProvider      provider   = null;
            ICustomTypeDescriptor        descriptor = null;
            PropertyDescriptorCollection properties = null;
            PropertyDescriptor           propDesc   = null;

            // Use the data to find the object with the rest of the path.
            object currObj = currDoc;
            for ( ;i<path.Length;++i )
            {
                #region Get the property

                provider = TypeDescriptor.GetProvider( currObj.GetType() );
                if ( provider==null )
                    return currDoc;

                descriptor = provider.GetTypeDescriptor( currObj );
                if ( descriptor==null )
                    return currDoc;

                properties = descriptor.GetProperties();
                if ( properties==null )
                    return currDoc;

                propDesc = null;
                foreach ( PropertyDescriptor desc in properties )
                {
                    if ( desc.Name==path[i].Name )
                    {
                        propDesc = desc;
                        break;
                    }
                }

                if ( propDesc==null )
                    return currDoc;

                currObj = propDesc.GetValue( currObj );
                if ( currObj==null )
                    return currDoc;

                // Get object from an array
                if ( currObj is Array &&
                     path[i].ID!=0 )
                {
                    Array arr    = currObj as Array;
                    int   iIndex = (int)path[i].ID - 1;
                    if ( iIndex<0 || iIndex>=arr.Length )
                        return currDoc;

                    currObj = arr.GetValue( iIndex );
                }

                if ( currObj is Document )
                    currDoc = currObj as Document;
                else
                    return currDoc;

                #endregion
            }

            return currDoc;
        }

        #endregion

        #region Get / set data by the data source path

        /// <summary>
        /// Get the data with the given path.
        ///
        /// NOTE that if you use the ID of the path nodes to access array items,
        /// make sure you add 1 to the index.
        /// For example, if you want to access the second item in the array
        /// (the index should be 1), assign 2 to the ID.
        /// </summary>
        /// <param name="path">The path to the data.</param>
        /// <returns>The data with the given path.</returns>
        public static bool SafeGetDataByPath( Path path,
                                              out object data )
        {
            data = null;
            if ( path.Length<=0 )
                return false;

            IDocument    currDoc = null;
            PathTreeNode node    = s_documentTree.Root;

            // Find in registered documents
            int i;
            for ( i=0;i<path.Length;++i )
            {
                // Get child tree node which contains a document.
                PathTreeNode child = node.FindFirstChild( path[i].NameCRC, path[i].ID );
                if ( child==null )
                    break;

                // Remember the document for later.
                currDoc = child.Data as IDocument;

                // Step in to the child node.
                node = child;
            }

            if ( i>=path.Length )
            {
                data = currDoc;
                return true;
            }

            if ( currDoc==null )
                return false;

            TypeDescriptionProvider      provider   = null;
            ICustomTypeDescriptor        descriptor = null;
            PropertyDescriptorCollection properties = null;
            PropertyDescriptor           propDesc   = null;

            // Use the data to find the object with the rest of the path.
            object currObj = currDoc;
            for ( ;i<path.Length;++i )
            {
                #region Get the property

                provider = TypeDescriptor.GetProvider( currObj.GetType() );
                if ( provider==null )
                    return false;

                descriptor = provider.GetTypeDescriptor( currObj );
                if ( descriptor==null )
                    return false;

                properties = descriptor.GetProperties();
                if ( properties==null )
                    return false;

                propDesc = null;
                foreach ( PropertyDescriptor desc in properties )
                {
                    if ( desc.Name==path[i].Name )
                    {
                        propDesc = desc;
                        break;
                    }
                }

                if ( propDesc==null )
                    return false;

                currObj = propDesc.GetValue( currObj );
                if ( currObj==null )
                    return false;

                // Get object from an array
                if ( currObj is Array &&
                     path[i].ID!=0 )
                {
                    Array arr    = currObj as Array;
                    int   iIndex = (int)path[i].ID - 1;
                    if ( iIndex<0 || iIndex>=arr.Length )
                        return false;

                    currObj = arr.GetValue( iIndex );
                }

                #endregion
            }

            if ( currObj is DataModelFieldInfo )
                data = ( currObj as DataModelFieldInfo ).Value;
            else
                data = currObj;

            return true;
        }

        /// <summary>
        /// Get the data with the given path.
        ///
        /// NOTE that if you use the ID of the path nodes to access array items,
        /// make sure you add 1 to the index.
        /// For example, if you want to access the second item in the array
        /// (the index should be 1), assign 2 to the ID.
        /// </summary>
        /// <param name="path">The path to the data.</param>
        /// <returns>The data with the given path.</returns>
        public static object GetDataByPath( Path path )
        {
            if ( path.Length<=0 )
                return false;

            IDocument    currDoc = null;
            PathTreeNode node    = s_documentTree.Root;

            // Find in registered documents
            int i;
            for ( i=0;i<path.Length;++i )
            {
                // Get child tree node which contains a document.
                PathTreeNode child = node.FindFirstChild( path[i].NameCRC, path[i].ID );
                if ( child==null )
                    break;

                // Remember the document for later.
                currDoc = child.Data as IDocument;

                // Step in to the child node.
                node = child;
            }

            if ( i>=path.Length )
                return currDoc;

            if ( currDoc==null )
                return null;

            TypeDescriptionProvider      provider   = null;
            ICustomTypeDescriptor        descriptor = null;
            PropertyDescriptorCollection properties = null;
            PropertyDescriptor           propDesc   = null;

            // Use the data to find the object with the rest of the path.
            object currObj = currDoc;
            for ( ;i<path.Length;++i )
            {
                #region Get the property

                provider = TypeDescriptor.GetProvider( currObj.GetType() );
                if ( provider==null )
                    return null;

                descriptor = provider.GetTypeDescriptor( currObj );
                if ( descriptor==null )
                    return null;

                properties = descriptor.GetProperties();
                if ( properties==null )
                    return null;

                propDesc = null;
                foreach ( PropertyDescriptor desc in properties )
                {
                    if ( desc.Name==path[i].Name )
                    {
                        propDesc = desc;
                        break;
                    }
                }

                if ( propDesc==null )
                    return null;

                currObj = propDesc.GetValue( currObj );
                if ( currObj==null )
                    return null;

                // Get object from an array
                if ( currObj is Array &&
                     path[i].ID!=0 )
                {
                    Array arr    = currObj as Array;
                    int   iIndex = (int)path[i].ID - 1;
                    if ( iIndex<0 || iIndex>=arr.Length )
                        return null;

                    currObj = arr.GetValue( iIndex );
                }

                #endregion
            }

            if ( currObj is DataModelFieldInfo )
                return ( currObj as DataModelFieldInfo ).Value;
            else
                return currObj;
        }


        /// <summary>
        /// Set the given data to the data pointed by the specified path.
        ///
        /// NOTE that if you use the ID of the path nodes to access array items,
        /// make sure you add 1 to the index.
        /// For example, if you want to access the second item in the array
        /// (the index should be 1), assign 2 to the ID.
        /// </summary>
        /// <param name="path">The path to the data.</param>
        /// <param name="value">The value to set.</param>
        /// <returns>True on success.</returns>
        public static bool SetDataByPath( Path path,
                                          object value )
        {
            if ( path.Length<=0 )
                return false;

            // We need to ask animation table manager to set the value for us
            // is the value is a animation key frame list.
            if ( value is BaseKeyFrameList )
            {
                return TheApp.AnimationTableManager.SetAnimationTable( path,
                                                                       value as BaseKeyFrameList );
            }

            IDocument    currDoc = null;
            PathTreeNode node    = s_documentTree.Root;

            // Find in registered documents
            int i;
            for ( i=0;i<path.Length;++i )
            {
                // Get child tree node which contains a document.
                PathTreeNode child = node.FindFirstChild( path[i].NameCRC, path[i].ID );
                if ( child==null )
                    break;

                // Remember the document for later.
                currDoc = child.Data as IDocument;

                // Step in to the child node.
                node = child;
            }

            if ( i>=path.Length )
                return false;

            if ( currDoc==null )
                return false;

            TypeDescriptionProvider      provider   = null;
            ICustomTypeDescriptor        descriptor = null;
            PropertyDescriptorCollection properties = null;
            PropertyDescriptor           propDesc   = null;

            // Use the data to find the object with the rest of the path.
            object currObj = currDoc;
            for ( ;i<path.Length-1;++i )
            {
                #region Find the property

                provider = TypeDescriptor.GetProvider( currObj.GetType() );
                if ( provider==null )
                    return false;

                descriptor = provider.GetTypeDescriptor( currObj );
                if ( descriptor==null )
                    return false;

                properties = descriptor.GetProperties();
                if ( properties==null )
                    return false;

                propDesc = null;
                foreach ( PropertyDescriptor desc in properties )
                {
                    if ( desc.Name==path[i].Name )
                    {
                        propDesc = desc;
                        break;
                    }
                }

                if ( propDesc==null )
                    return false;

                currObj = propDesc.GetValue( currObj );
                if ( currObj==null )
                    return false;

                // Get object from an array
                if ( currObj is Array &&
                     path[i].ID!=0 )
                {
                    Array arr    = currObj as Array;
                    int   iIndex = (int)path[i].ID - 1;
                    if ( iIndex<0 || iIndex>=arr.Length )
                        return false;

                    currObj = arr.GetValue( iIndex );
                }

                #endregion
            }

            #region Find the last property

            provider = TypeDescriptor.GetProvider( currObj.GetType() );
            if ( provider==null )
                return false;

            descriptor = provider.GetTypeDescriptor( currObj );
            if ( descriptor==null )
                return false;

            properties = descriptor.GetProperties();
            if ( properties==null )
                return false;

            propDesc = null;
            foreach ( PropertyDescriptor desc in properties )
            {
                if ( desc.Name==path[path.Length - 1].Name )
                {
                    propDesc = desc;
                    break;
                }
            }

            if ( propDesc==null )
            {
                string msg = String.Format("DocumentManager.SetDataByPath : The data path \"{0}\" is not found.", path.PathName);
                DebugConsole.WriteLine(msg);
                Debug.WriteLine(msg);
                return false;
            }

            bool bIsArray = false;
            Type propType = propDesc.PropertyType;
            if ( propType.IsSubclassOf( typeof( Array ) )==true &&
                 path[i].ID!=0 )
            {
                Array arr    = propDesc.GetValue( currObj ) as Array;
                int   iIndex = (int)path[i].ID - 1;
                if ( iIndex<0 || iIndex>=arr.Length )
                {
                    string msg = String.Format( "DocumentManager.SetDataByPath : The array index \"{0}\" is out of range.", iIndex );
                    DebugConsole.WriteLine( msg );
                    Debug.WriteLine( msg );
                    return false;
                }

                bIsArray = true;
                propType = arr.GetValue( iIndex ).GetType();
            }

            #endregion

            #region Set value to the property

            if ( bIsArray==false &&
                 propDesc.IsReadOnly==true )
            {
                string msg = String.Format("DocumentManager.SetDataByPath : The data source \"{0}\" does not have a setter.", path.PathName);
                DebugConsole.WriteLine(msg);
                Debug.WriteLine(msg);
                return false;
            }

            if ( propType.IsPrimitive==true &&
                 value==null )
            {
                string msg = String.Format("DocumentManager.SetDataByPath : Unable to set null to primitive typed data source \"{0}\".", path.PathName);
                DebugConsole.WriteLine(msg);
                Debug.WriteLine(msg);
                return false;
            }

            if ( propType.Equals(value.GetType())==false )
            {
                string msg = String.Format("DocumentManager.SetDataByPath : The data source \"{0}\" has different data type.", path.PathName);
                DebugConsole.WriteLine(msg);
                Debug.WriteLine(msg);
                return false;
            }

            // Set the given value to the property of the current object
            if ( bIsArray==true )
            {
                Array arr    = propDesc.GetValue( currObj ) as Array;
                int   iIndex = (int)path[i].ID - 1;

                arr.SetValue( value,
                              iIndex );
            }
            else if ( currObj is DataModelInfo )
            {
                DataModelFieldInfo field =
                    propDesc.GetValue( currObj ) as DataModelFieldInfo;
                if ( field!=null )
                    field.Value = value;
            }
            else
            {
                propDesc.SetValue( currObj,
                                   value );
            }

            #endregion

            return true;
        }

        #endregion

        #region Data model proxies

        /// <summary>
        /// Register a newly created data model proxy.
        /// </summary>
        /// <param name="instance">The data model proxy instance.</param>
        public static void RegisterDataModelProxy( DataModelProxy instance )
        {
            // Has the data model proxy already been registered?
            if ( s_dataModelProxies.IndexOf( instance )>=0 )
                return;

            s_dataModelProxies.Add( instance );

            foreach ( DataModelInfo dataModel in s_dataModels.Values )
            {
                instance.AddDataModel( dataModel );
            }
        }


        /// <summary>
        /// Unregister a data model proxy.
        /// </summary>
        /// <param name="instance">The data model proxy instance.</param>
        public static void UnregisterDataModelProxy( DataModelProxy instance )
        {
            s_dataModelProxies.Remove( instance );
        }


        /// <summary>
        /// Begin creating a new data model.
        /// </summary>
        /// <param name="iID">ID for the data model.</param>
        /// <param name="name">Name for the data model.</param>
        /// <param name="label">Label for the property page.</param>
        /// <returns>The created data model.</returns>
        public static bool BeginCreatingDataModel( uint iID,
                                                   string name,
                                                   string label )
        {
            // Do not allow nested calls.
            if ( s_currDataModel!=null )
                return false;

            // Does the data model already exist?
            DataModelInfo dataModel = GetDataModelTemplate( name );
            if ( dataModel!=null )
                return false;

            // Create a new data model.
            dataModel = new DataModelInfo( label, name, iID );
            s_dataModels.Add( dataModel.NameCRC, dataModel );

            s_currDataModel = dataModel;

            return true;
        }


        /// <summary>
        /// Add a new field to the creating data model.
        /// </summary>
        /// <param name="fieldName">Name of the value field.</param>
        /// <param name="valueType">Type of the value.</param>
        /// <param name="defaultValue">Default value to set the the field.</param>
        /// <returns>True on success.</returns>
        public static bool AddDataModelField( string fieldName,
                                              Type valueType,
                                              object defaultValue )
        {
            if ( s_currDataModel==null )
                return false;

            if ( s_currDataModel.AddField( fieldName, valueType, defaultValue )==null )
                return false;

            return true;
        }


        /// <summary>
        /// End creating the data model.
        /// </summary>
        public static void EndCreatingDataModel()
        {
            if ( s_currDataModel==null )
                return;

            foreach ( DataModelProxy item in s_dataModelProxies )
            {
                item.AddDataModel( s_currDataModel );
            }

            s_currDataModel = null;
        }


        /// <summary>
        /// Find the data model with the given name.
        /// </summary>
        /// <param name="name">Name of the data model.</param>
        /// <returns>The found data model or null if not found.</returns>
        public static DataModelInfo GetDataModelTemplate( string name )
        {
            uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( name );

            // Does the data model already exist?
            DataModelInfo dataModel = null;
            if ( s_dataModels.TryGetValue( iNameCRC, out dataModel )==true )
            {
                return dataModel;
            }

            return null;
        }


        /// <summary>
        /// Determine if the given path points to an user data model.
        /// </summary>
        /// <param name="path">The data source path.</param>
        /// <returns>True if the path points to an user data model.</returns>
        public static bool IsUserDataModelPath( Path path )
        {
            if ( path.Length<=0 )
                return false;

            IDocument    currDoc = null;
            PathTreeNode node    = s_documentTree.Root;

            // Find in registered documents
            int i;
            for ( i=0;i<path.Length;++i )
            {
                // Get child tree node which contains a document.
                PathTreeNode child = node.FindFirstChild( path[i].NameCRC, path[i].ID );
                if ( child==null )
                    break;

                // Remember the document for later.
                currDoc = child.Data as IDocument;

                // Step in to the child node.
                node = child;
            }

            if ( i>=path.Length )
                return false;

            if ( currDoc==null )
                return false;

            TypeDescriptionProvider      provider   = null;
            ICustomTypeDescriptor        descriptor = null;
            PropertyDescriptorCollection properties = null;
            PropertyDescriptor           propDesc   = null;

            // Use the data to find the object with the rest of the path.
            object currObj = currDoc;
            for ( ;i<path.Length;++i )
            {
                // It does not count as an user data model
                // if the last path node of the path points
                // to the data model proxy, so we put the
                // if statement in the beginning.
                if ( currObj is DataModelProxy )
                    return true;

                #region Get the property

                provider = TypeDescriptor.GetProvider( currObj.GetType() );
                if ( provider==null )
                    return false;

                descriptor = provider.GetTypeDescriptor( currObj );
                if ( descriptor==null )
                    return false;

                properties = descriptor.GetProperties();
                if ( properties==null )
                    return false;

                propDesc = null;
                foreach ( PropertyDescriptor desc in properties )
                {
                    if ( desc.Name==path[i].Name )
                    {
                        propDesc = desc;
                        break;
                    }
                }

                if ( propDesc==null )
                    return false;

                currObj = propDesc.GetValue( currObj );
                if ( currObj==null )
                    return false;

                // Get object from an array
                if ( currObj is Array &&
                     path[i].ID!=0 )
                {
                    Array arr    = currObj as Array;
                    int   iIndex = (int)path[i].ID - 1;
                    if ( iIndex<0 || iIndex>=arr.Length )
                        return false;

                    currObj = arr.GetValue( iIndex );
                }

                #endregion
            }

            return false;
        }


        /// <summary>
        /// Get data model for serialization.
        /// </summary>
        /// <param name="path">The data source path to the data model.</param>
        /// <returns>The data model serialize list object.</returns>
        public static DataModelSerializeList GetDataModelForSerialize( Path path )
        {
            if ( path.Length<=0 )
                return null;

            IDocument    currDoc = null;
            PathTreeNode node    = s_documentTree.Root;

            // Find in registered documents
            int i;
            for ( i=0;i<path.Length;++i )
            {
                // Get child tree node which contains a document.
                PathTreeNode child = node.FindFirstChild( path[i].NameCRC, path[i].ID );
                if ( child==null )
                    break;

                // Remember the document for later.
                currDoc = child.Data as IDocument;

                // Step in to the child node.
                node = child;
            }

            if ( i>=path.Length )
                return null;

            if ( currDoc==null )
                return null;

            TypeDescriptionProvider      provider   = null;
            ICustomTypeDescriptor        descriptor = null;
            PropertyDescriptorCollection properties = null;
            PropertyDescriptor           propDesc   = null;

            uint iDataModelNameCRC = 0;

            // Use the data to find the object with the rest of the path.
            object currObj = currDoc;
            for ( ;i<path.Length;++i )
            {
                if ( currObj is DataModelProxy )
                {
                    iDataModelNameCRC = path[i].NameCRC;
                    break;
                }

                #region Get the property

                provider = TypeDescriptor.GetProvider( currObj.GetType() );
                if ( provider==null )
                    return null;

                descriptor = provider.GetTypeDescriptor( currObj );
                if ( descriptor==null )
                    return null;

                properties = descriptor.GetProperties();
                if ( properties==null )
                    return null;

                propDesc = null;
                foreach ( PropertyDescriptor desc in properties )
                {
                    if ( desc.Name==path[i].Name )
                    {
                        propDesc = desc;
                        break;
                    }
                }

                if ( propDesc==null )
                    return null;

                currObj = propDesc.GetValue( currObj );
                if ( currObj==null )
                    return null;

                // Get object from an array
                if ( currObj is Array &&
                     path[i].ID!=0 )
                {
                    Array arr    = currObj as Array;
                    int   iIndex = (int)path[i].ID - 1;
                    if ( iIndex<0 || iIndex>=arr.Length )
                        return null;

                    currObj = arr.GetValue( iIndex );
                }

                #endregion
            }

            if ( (currObj is DataModelProxy)==false )
                return null;

            return (currObj as DataModelProxy).GetDataModelForSerialize( iDataModelNameCRC );
        }

        #endregion

        #region Add / remove children

        /// <summary>
        /// ドキュメントを追加。
        /// </summary>
        public static void Add(IDocument document)
        {
            Debug.Assert(document != null);
            Debug.Assert(!s_documents.Contains(document));

            s_documents.Add(document);
            document.NotifyAddedToParent();
        }

        /// <summary>
        /// ドキュメントを追加（子供一緒に追加します）
        /// </summary>
        public static void AddWithChild(IDocument document)
        {
            Debug.Assert(document != null);

            // 挿入します。
            if (s_documents.Contains(document) == false)
            {
                s_documents.Add( document );
                document.NotifyAddedToParent();
            }
        }

        /// <summary>
        /// ドキュメントを削除。
        /// </summary>
        public static void RemoveWithChild(IDocument document)
        {
            Debug.Assert(document != null);

            // 削除します。
            if ( s_documents.Contains(document)==true )
            {
                s_documents.Remove( document );

                // Notify the document that it's been removed
                document.NotifyRemovedFromParent();
            }
        }

        /// <summary>
        /// ドキュメントを削除。
        /// </summary>
        public static void Remove(IDocument document)
        {
            Debug.Assert(document != null);
            // 削除
            if ( s_documents.Contains(document)==true )
            {
                s_documents.Remove( document );

                // Notify the document that it's been removed
                document.NotifyRemovedFromParent();
            }
        }

        #endregion

        #region Find / name checking

        /// <summary>
        /// ドキュメントが含まれている？
        /// </summary>
        /// <param name="document">検査するドキュメントです。</param>
        /// <returns>=true..既に含まれています。</returns>
        public static bool Contains(IDocument document)
        {
            Debug.Assert(document != null);
            return s_documents.Contains(document);
        }

        /// <summary>
        /// ドキュメントに指定の名前が含まれているか？
        /// </summary>
        /// <param name="name">検索する名前</param>
        /// <returns>=true..既にある</returns>
        public static bool ContainsName(string name)
        {
            return s_documents.Find(
                (IDocument item) => (item.Name == name)
            ) != null;
        }

        /// <summary>
        /// ドキュメントを検索します。
        /// </summary>
        /// <param name="objId">GuiObjectIDです。</param>
        /// <param name="name">オブジェクト名です。</param>
        /// <param name="path">パスです。</param>
        /// <returns></returns>
        public static IDocument Find(GuiObjectID objId, string name, string path = "")
        {
            // only compare the name
            if (path == "")
            {
                return s_documents.Find(
                    (IDocument item) => (
                            item.ObjectID == objId &&
                            item.Name == name)
                );
            }

            // compare both name and path
            string lowPath = path.ToLower();
            return s_documents.Find(
                (IDocument item) => (
                        item.ObjectID == objId &&
                        item.Name == name &&
                        item.FilePath.ToLower().Equals(lowPath))
            );
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Get an unique document ID.
        /// </summary>
        /// <returns>An unique document ID.</returns>
        public static int GetUniqueDocumentID()
        {
            return s_iDocSerialID++;
        }


        /// <summary>
        /// Determine if the value of two objects are equal.
        /// </summary>
        /// <param name="data1">Data 1</param>
        /// <param name="data2">Data 2</param>
        /// <returns>True if equal.</returns>
        public static bool IsEqual( object data1,
                                    object data2 )
        {
            // Two null objects
            if ( data1==null &&
                 data2==null )
            {
                return true;
            }

            // Not equal if any of the values are null.
            if ( data1==null ||
                 data2==null )
            {
                return false;
            }

            // Not equal if the types are different.
            Type type1 = data1.GetType();
            Type type2 = data2.GetType();
            if ( type1.Equals(type2)==false )
                return false;

            // Value / enum types
            if ( type1.IsPrimitive==true ||
                 type1.IsEnum==true )
            {
                #region Compare the values

                return data1.Equals(data2);

                #endregion
            }

            // Vectors
            else if ( data1 is IVector )
            {
                #region Compare the values

                IVector vec1 = (IVector)data1;
                IVector vec2 = (IVector)data2;

                // Compare the elements of the vector.
                for ( int i=0;i<vec1.Dimension;++i )
                {
                    if ( vec1[i]!=vec2[i] )
                    {
                        return false;
                    }
                }

                return true;

                #endregion
            }

            // Integer vectors
            else if ( data1 is IVectori )
            {
                #region Compare the values

                IVectori vec1 = (IVectori)data1;
                IVectori vec2 = (IVectori)data2;

                // Compare the elements of the vector.
                for ( int i=0;i<vec1.Dimension;++i )
                {
                    if ( vec1[i]!=vec2[i] )
                    {
                        return false;
                    }
                }

                return true;

                #endregion
            }

            // Arrays
            else if ( data1 is Array )
            {
                #region Compare the values

                Array array1 = data1 as Array;
                Array array2 = data2 as Array;
                if ( array1.Length!=array2.Length )
                    return false;

                int i;
                for ( i=0;i<array1.Length;++i )
                {
                    if ( IsEqual( array1.GetValue(i),
                                  array2.GetValue(i) )==false )
                    {
                        return false;
                    }
                }

                return true;

                #endregion
            }

            // Animation table key frame list
            else if ( data1 is BaseKeyFrameList )
            {
                #region Compare the values

                BaseKeyFrameList list1 = data1 as BaseKeyFrameList;
                BaseKeyFrameList list2 = data2 as BaseKeyFrameList;

                return list1.IsEqual( list2 );

                #endregion
            }

            // Unknown types
            else
            {
                #region Compare the values

                return data1.Equals(data2);

                #endregion
            }
        }

        #endregion

        #region Events

        /// <summary>
        /// Event triggered when documents start loading.
        /// </summary>
        public static event EventHandler BeginDocLoading = null;

        /// <summary>
        /// Event triggered when documents end loading.
        /// </summary>
        public static event EventHandler EndDocLoading = null;

        /// <summary>
        /// Event triggered when documents being created.
        /// </summary>
        public static event DocumentEventHandler DocumentCreated = null;


        /// <summary>
        /// Event triggered when documents being removed.
        /// </summary>
        public static event DocumentEventHandler DocumentRemoved = null;


        /// <summary>
        /// Event triggered when documents being modified.
        /// </summary>
        public static event DocumentEventHandler DocumentModified = null;

        /// <summary>
        /// Notify document manager that some documents are about to be loaded.
        /// </summary>
        public static void NotifyBeginDocLoading()
        {
            if ( BeginDocLoading!=null )
                BeginDocLoading( null, EventArgs.Empty );
        }

        /// <summary>
        /// Notify document manager that document loading is finished.
        /// </summary>
        public static void NotifyEndDocLoading()
        {
            if ( EndDocLoading!=null )
                EndDocLoading( null, EventArgs.Empty );
        }

        /// <summary>
        /// Notify document manager that the a document is created.
        /// </summary>
        /// <param name="document">The created document.</param>
        public static void NotifyDocumentCreated( IDocument document )
        {
            if ( DocumentManager.IsSuppressingDocumentEvents==false &&
                 DocumentCreated!=null )
            {
                DocumentCreated( document );
            }
        }


        /// <summary>
        /// Notify document manager that the a document is about to be removed.
        /// </summary>
        /// <param name="document">The document about to be removed.</param>
        public static void NotifyDocumentRemoved( IDocument document )
        {
            if ( DocumentManager.IsSuppressingDocumentEvents==false &&
                 DocumentRemoved!=null )
            {
                DocumentRemoved( document );
            }
        }


        /// <summary>
        /// Notify document manager that the a document is modified.
        /// </summary>
        /// <param name="document">The modified document.</param>
        public static void NotifyDocumentModified( IDocument document )
        {
            if ( DocumentManager.IsSuppressingDocumentEvents==false &&
                 DocumentModified!=null )
            {
                DocumentModified( document );
            }
        }

        /// <summary>
        /// Recieve texture chaned events from FileSystemWatcher in TextureManager.
        /// </summary>
        /// <param name="source"></param>
        /// <param name="e"></param>
        public static void OnTextureChanged(object source, System.IO.FileSystemEventArgs e)
        {
            if (Config.Data.Option.AutoTextureReload)
            {
                if ( e.ChangeType==System.IO.WatcherChangeTypes.Deleted ||
                     e.ChangeType==System.IO.WatcherChangeTypes.Renamed )
                {
                    NotifyTextureChanged( null );
                }
                else
                {
                    NotifyTextureChanged(e.FullPath);
                }
            }
        }

        /// <summary>
        /// Notify that texture is Removed/Renamed/Deleted/Created.
        /// </summary>
        /// <param name="path"></param>
        public static void NotifyTextureChanged(string path)
        {
            #region Check if the texture can be opened

            if ( string.IsNullOrEmpty(path)==false &&
                 System.IO.File.Exists(path)==true )
            {
                System.IO.FileStream stream = null;

                // Try to access the file for 10 times.
                const int iNumTries = 10;

                bool bSuccess = false;
                for ( int i=0;i<iNumTries;++i )
                {
                    try
                    {
                        stream = System.IO.File.Open( path,
                                                      System.IO.FileMode.Open,
                                                      System.IO.FileAccess.Read,
                                                      System.IO.FileShare.Read );
                        if ( stream!=null )
                        {
                            bSuccess = true;
                            break;
                        }
                    }
                    catch ( System.IO.IOException ex )
                    {
                        // Sleep for a moment if failed to open the texture.
                        if ( ex!=null ) // Shut up the compiler warnings
                            System.Threading.Thread.Sleep( 100 );
                    }
                    finally
                    {
                        if ( stream!=null )
                            stream.Close();
                    }
                }

                // Failed opening the file.
                if ( bSuccess==false )
                    return;
            }

            #endregion

            // Notify each document that the texture is reloaded.
            List<EmitterSetDocument> esetList = new List<EmitterSetDocument>();
            foreach ( IDocument doc in s_documents )
            {
                EffectProjectDocument project = (doc as EffectProjectDocument);
                if ( project!=null )
                {
                    foreach ( EmitterSetDocument emitterSet in project.EmitterSetDocuments )
                    {
                        bool bReloaded = false;
                        foreach ( EmitterDocument emitter in emitterSet.EmitterDocuments )
                        {
                            bReloaded |= emitter.ReloadTexture0(path);
                            bReloaded |= emitter.ReloadTexture1(path);
                            bReloaded |= emitter.ReloadTexture2(path);

                            if ( emitter.ChildDocument!=null )
                            {
                                bReloaded |= emitter.ChildDocument.ReloadTexture(path);
                            }
                        }

                        // Add the EmitterSet to list if any emitter/child reloaded the texture.
                        if ( bReloaded==true )
                        {
                            esetList.Add(emitterSet);
                        }
                    }
                }
            }

            // Notify the viewer that the texture is reloaded
            if (ProjectManager.ActiveProject != null &&
                ProjectManager.ActiveProject.PreviewDocument as PreviewDocument != null)
            {
                var previewDocument = (PreviewDocument)ProjectManager.ActiveProject.PreviewDocument;
                previewDocument.ReloadTexture(path);
            }
        }

        #endregion

        #region Member variables

        // Document list and active document.
        private static readonly List<IDocument> s_documents      = new List<IDocument>();
//        private static IDocument                s_activeDocument = null;
        private static IDocument                s_activeFileDoc  = null;

        private static int                      s_iDocSerialID   = 0;

        private static bool                     s_bSuppressDocEvents = false;

        // The path tree for storing the data modification flags.
        private static PathTree s_modificationDataSet = new PathTree( false );

        // This tree keeps track on all the documents with their source data path.
        private static PathTree s_documentTree = new PathTree( null );

        private static List<DataModelProxy> s_dataModelProxies = new List<DataModelProxy>();

        private static DataModelInfo s_currDataModel = null;

        private static Dictionary<uint,DataModelInfo> s_dataModels =
            new Dictionary<uint, DataModelInfo>();

        #endregion
    }

    #region Class for temporarily suppress document events

    /// <summary>
    /// Helper class for temporarily suppress document events.
    /// </summary>
    public class SuppressDocumentEventBlock : IDisposable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public SuppressDocumentEventBlock()
        {
            m_bSuppressingEvents = DocumentManager.IsSuppressingDocumentEvents;

            DocumentManager.IsSuppressingDocumentEvents = true;
        }


        /// <summary>
        /// Dispose.
        /// </summary>
        public void Dispose()
        {
            DocumentManager.IsSuppressingDocumentEvents = m_bSuppressingEvents;
        }


        private bool m_bSuppressingEvents;
    }

    #endregion
}
