﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Xml;

using NWCore.Utility;

using NWCore.DataModel;
using NWCore.Serializer;

namespace App.Data
{
    #region Interface for data model proxy owner

    /// <summary>
    /// Interface for data model proxy owner.
    /// </summary>
    public interface IDataModelProxyOwner
    {
        /// <summary>
        /// Determine if the data mode with the given ID is enabled.
        /// </summary>
        /// <param name="iDataModelID">The data model ID.</param>
        /// <returns>True on enabled.</returns>
        bool IsDataModelEnabled( uint iDataModelID );
    }

    #endregion

    #region Custom type description provider for data model proxy

    /// <summary>
    /// Abstract class for the classes to inherit from if they want
    /// to use dynamic properties.
    /// Because the type description provider does not describe
    /// interfaces, so we need to inherit them from abstract class.
    /// </summary>
    public abstract class DynamicCRCNamedPropertyContainer
    {
        #region Abstract methods to implement

        /// <summary>Get the dictionary of all the dynamic properties.</summary>
        public abstract Dictionary<uint, object> DynamicProperties { get; }


        /// <summary>
        /// Get the information of the property.
        /// </summary>
        /// <param name="iNameCRC">CRC hash code for the property name.</param>
        /// <param name="name">Name of the property.</param>
        /// <param name="type">Type of the property.</param>
        /// <returns>False if the property is not found.</returns>
        public abstract bool GetPropertyInfo( uint iNameCRC,
                                              out string name,
                                              out Type type );

        #endregion
    }


    /// <summary>
    /// Class for the type descriptor provider for data models.
    /// </summary>
    public class DataModelTypeDescriptionProvider : TypeDescriptionProvider
    {
        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public DataModelTypeDescriptionProvider() :
            this( TypeDescriptor.GetProvider(typeof(DynamicCRCNamedPropertyContainer)) )
        {
        }


        /// <summary>
        /// Constructor initializing with the given parent provider.
        /// </summary>
        /// <param name="parent">The parent provider.</param>
        public DataModelTypeDescriptionProvider( TypeDescriptionProvider parent ) :
            base(parent)
        {
        }

        #endregion

        #region Overriden GetTypeDescriptor method

        /// <summary>
        /// Get type descriptor.
        /// </summary>
        /// <param name="objectType">The object type.</param>
        /// <param name="instance">The instance for getting the descriptor.</param>
        /// <returns>The type descriptor for the instance.</returns>
        public override ICustomTypeDescriptor GetTypeDescriptor( Type objectType,
                                                                 object instance )
        {
            if ( m_customDescriptor==null )
            {
                ICustomTypeDescriptor baseDescriptor =
                    base.GetTypeDescriptor( objectType, instance );

                m_customDescriptor = new DataModelCustomTypeDescriptor( baseDescriptor );
            }

            m_customDescriptor.Instance = instance as DynamicCRCNamedPropertyContainer;

            return m_customDescriptor;
        }

        #endregion

        #region Member variables

        private DataModelCustomTypeDescriptor m_customDescriptor = null;

        #endregion
    }


    /// <summary>
    /// Class for the custom type descriptors for data models.
    /// </summary>
    public class DataModelCustomTypeDescriptor : CustomTypeDescriptor
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="parent">Parent descriptor for initialization.</param>
        public DataModelCustomTypeDescriptor( ICustomTypeDescriptor parent ) :
            base( parent )
        {
        }

        #endregion

        #region Properties

        /// <summary>Get or set the instance of the type to describe.</summary>
        public DynamicCRCNamedPropertyContainer Instance
        {
            get;
            set;
        }

        #endregion

        #region Overriden GetProperty method

        /// <summary>
        /// Get property collection for the type.
        /// </summary>
        /// <returns></returns>
        public override PropertyDescriptorCollection GetProperties()
        {
            if ( this.Instance==null )
            {
                // Return an empty property descriptor collection.
                PropertyDescriptor[] descriptors = new PropertyDescriptor[0];
                return new PropertyDescriptorCollection( descriptors );
            }

            // Get the instance type.
            Type instanceType = this.Instance.GetType();

            // The ordinary properties the type has.
            PropertyDescriptorCollection staticProperties = base.GetProperties();

            string name;
            Type   type;

            // How many properties do we have in the instance?
            int iNumProperties = this.Instance.DynamicProperties.Count +
                                 staticProperties.Count;

            // Create a list to contain all the property descriptors.
            List<PropertyDescriptor> properties =
                new List<PropertyDescriptor>( iNumProperties );

            foreach ( PropertyDescriptor property in staticProperties )
            {
                properties.Add( property );
            }

            // Enumerate through the dynamic properties and add them to the collection.
            DynamicPropertyDescriptor descriptor;
            foreach ( KeyValuePair<uint,object> pair in this.Instance.DynamicProperties )
            {
                // Get the property type.
                if ( this.Instance.GetPropertyInfo( pair.Key,
                                                    out name,
                                                    out type )==false )
                {
                    continue;
                }

                descriptor = new DynamicPropertyDescriptor( name,
                                                            instanceType,
                                                            type );

                properties.Add( descriptor );
            }

            return new PropertyDescriptorCollection( properties.ToArray() );
        }

        #endregion
    }


    /// <summary>
    /// Class for the property descriptor of the dynamic properties.
    /// </summary>
    public class DynamicPropertyDescriptor : PropertyDescriptor
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="name">Name of the property.</param>
        /// <param name="componentType">Type of the component this property is bound to.</param>
        /// <param name="propertyType">Type of the property.</param>
        public DynamicPropertyDescriptor( string name,
                                          Type componentType,
                                          Type propertyType ) :
            base( name, null )
        {
            m_propertyName    = string.Copy( name );
            m_propertyNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( name );
            m_componentType   = componentType;
            m_propertyType    = propertyType;
        }

        #endregion

        #region Properties

        /// <summary>Get the flag indicating if the property is readonly.</summary>
        public override bool IsReadOnly { get { return false; } }

        /// <summary>Get the type of the component this property is bound to.</summary>
        public override Type ComponentType { get { return m_componentType; } }

        /// <summary>Get the type of the property.</summary>
        public override Type PropertyType { get { return m_propertyType; } }

        #endregion

        #region Derived methods from PropertyDescriptor

        /// <summary>
        /// Resets the value for this property of the component to the default value.
        /// </summary>
        /// <param name="component">The component the property is bound to.</param>
        public override void ResetValue( object component )
        {
            // Do nothing.
        }


        /// <summary>
        /// Resets the value for this property of the component to the default value.
        /// </summary>
        /// <param name="component">The component the property is bound to.</param>
        public override bool CanResetValue( object component )
        {
            return false;
        }


        /// <summary>
        /// Determines a value indicating whether the value of this
        /// property needs to be persisted.
        /// </summary>
        /// <param name="component">The component the property is bound to.</param>
        public override bool ShouldSerializeValue( object component )
        {
            return true;
        }


        /// <summary>
        /// Get the value of the property with the given component.
        /// </summary>
        /// <param name="component">The component the property is bound to.</param>
        public override object GetValue( object component )
        {
            DynamicCRCNamedPropertyContainer container =
                component as DynamicCRCNamedPropertyContainer;
            if ( container==null )
                return null;

            Dictionary<uint,object> properties = container.DynamicProperties;

            object value = null;
            if ( properties.TryGetValue( m_propertyNameCRC, out value )==false )
                return null;

            return value;
        }


        /// <summary>
        /// Set the value of the property with the given value.
        /// </summary>
        /// <param name="component">The component the property is bound to.</param>
        /// <param name="value">The value to set.</param>
        public override void SetValue( object component,
                                       object value )
        {
            DynamicCRCNamedPropertyContainer container =
                component as DynamicCRCNamedPropertyContainer;
            if ( container==null )
                return;

            Dictionary<uint,object> properties = container.DynamicProperties;

            if ( properties.ContainsKey( m_propertyNameCRC )==false )
                return;

            // Set value
            properties[ m_propertyNameCRC ] = value;

            // Trigger value change event
            OnValueChanged( component, EventArgs.Empty );
        }

        #endregion

        #region Member variables

        private string m_propertyName;
        private uint   m_propertyNameCRC;
        private Type   m_componentType;
        private Type   m_propertyType;

        #endregion
    }

    #endregion

    #region Class for data model field information

    /// <summary>
    /// Class for data model field information.
    /// </summary>
    public class DataModelFieldInfo
    {
        #region Constructor

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="name">Name for the field.</param>
        public DataModelFieldInfo( string name )
        {
            this.Name  = name;
            this.Type  = null;
            this.Value = null;
        }


        /// <summary>
        /// Copy constructor
        /// </summary>
        /// <param name="template">Template for create the field.</param>
        public DataModelFieldInfo( DataModelFieldInfo template )
        {
            this.Name  = template.Name;
            this.Type  = template.Type;
            this.Value = template.Value;
        }

        #endregion

        #region Properties

        /// <summary>Name of the field.</summary>
        public string Name
        {
            get { return m_name; }
            set
            {
                m_name = string.Copy( value );

                this.NameCRC = TheApp.CRC32Helper.ComputeCRC32Str( value );
            }
        }

        /// <summary>CRC32 hash for the name of the field.</summary>
        public uint   NameCRC { get; private set; }

        /// <summary>Value type of the field.</summary>
        public Type   Type    { get; set; }

        /// <summary>Value for the field.</summary>
        public object Value
        {
            get { return m_value; }
            set
            {
                if ( value is ICloneable )
                {
                    m_value = (value as ICloneable).Clone();
                }
                else
                {
                    m_value = value;
                }
            }
        }

        #endregion

        #region Utilities

        /// <summary>
        /// Check if the given value is valid to this field.
        /// </summary>
        /// <param name="value">The value.</param>
        /// <returns>True if valid.</returns>
        public bool IsValueValid( object value )
        {
            if ( this.Type.IsInstanceOfType( value )==true )
                return true;

            return false;
        }

        #endregion

        #region Member variables

        private string m_name  = string.Empty;
        private object m_value = null;

        #endregion
    }

    #endregion

    #region Class for data model information

    /// <summary>
    /// Class for data model information.
    /// </summary>
    [TypeDescriptionProvider( typeof( DataModelTypeDescriptionProvider ) )]
    public class DataModelInfo : DynamicCRCNamedPropertyContainer
    {
        #region Constructor

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="label">Label for the property page.</param>
        /// <param name="name">Name for the data model.</param>
        /// <param name="id">Data model ID.</param>
        public DataModelInfo( string label,
                              string name,
                              uint id )
        {
            this.Label   = string.Copy( label );
            this.Name    = string.Copy( name );
            this.ID      = id;
            this.Enabled = true;
        }


        /// <summary>
        /// Copy constructor
        /// </summary>
        /// <param name="template">Template to create the data model from.</param>
        public DataModelInfo( DataModelInfo template )
        {
            this.Label   = string.Copy( template.Label );
            this.Name    = string.Copy( template.Name );
            this.ID      = template.ID;
            this.Enabled = true;

            foreach ( DataModelFieldInfo field in template.m_fieldList )
            {
                AddField( field );
            }
        }

        #endregion

        #region Properties

        /// <summary>Label for the property page for the data model.</summary>
        public string Label { get; private set; }

        /// <summary>Name of the data model.</summary>
        public string Name
        {
            get { return m_name; }
            set
            {
                m_name = string.Copy( value );

                this.NameCRC = TheApp.CRC32Helper.ComputeCRC32Str( value );
            } // End of setter
        }

        /// <summary>CRC32 hash for the name of the data model.</summary>
        public uint NameCRC  { get; private set; }

        /// <summary>ID of the data model.</summary>
        public uint ID       { get; private set; }

        /// <summary>Get or set the flag indicating whether enable this data model or not.</summary>
        public bool Enabled  { get; set; }


        /// <summary>Get the dictionary of all the dynamic properties.</summary>
        public override Dictionary<uint, object> DynamicProperties
        {
            get { return m_fields; }
        }

        #endregion

        #region Field information

        /// <summary>
        /// Add a new field to the data model.
        /// </summary>
        /// <param name="name">Name for the field.</param>
        /// <param name="valueType">Type of the field value.</param>
        /// <param name="value">Value for the field.</param>
        /// <returns>The created field.</returns>
        public DataModelFieldInfo AddField( string name,
                                            Type valueType,
                                            object value )
        {
            DataModelFieldInfo field = GetField( name );
            if ( field!=null )
            {
                field.Value = value;
                return field;
            }

            field = new DataModelFieldInfo( name );
            field.Value = value;
            field.Type  = valueType;

            m_fields.Add( field.NameCRC, field );
            m_fieldList.Add( field );

            return field;
        }


        /// <summary>
        /// Add a new field to the data model.
        /// </summary>
        /// <param name="template">Template field to add from.</param>
        /// <returns>The created field.</returns>
        public DataModelFieldInfo AddField( DataModelFieldInfo template )
        {
            DataModelFieldInfo field = GetField( template.Name );
            if ( field!=null )
            {
                field.Value = template.Value;
                return field;
            }

            field = new DataModelFieldInfo( template );

            m_fields.Add( field.NameCRC, field );
            m_fieldList.Add( field );

            return field;
        }


        /// <summary>
        /// Find a field with the specified field name.
        /// </summary>
        /// <param name="name">The field name.</param>
        /// <returns>The found field information.</returns>
        public DataModelFieldInfo GetField( string name )
        {
            uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( name );

            object value = null;
            if ( m_fields.TryGetValue( iNameCRC, out value )==false )
                return null;

            DataModelFieldInfo field = value as DataModelFieldInfo;
            if ( field==null )
                return null;

            return field;
        }


        /// <summary>
        /// Find a field with the specified field name.
        /// </summary>
        /// <param name="iNameCRC">The CRC32 hash code of the field name.</param>
        /// <returns>The found field information.</returns>
        public DataModelFieldInfo GetFieldByNameCRC( uint iNameCRC )
        {
            object value = null;
            if ( m_fields.TryGetValue( iNameCRC, out value )==false )
                return null;

            DataModelFieldInfo field = value as DataModelFieldInfo;
            if ( field==null )
                return null;

            return field;
        }


        /// <summary>
        /// Get the number of fields this data model has.
        /// </summary>
        /// <returns>The number of fields this data model has.</returns>
        public int GetNumFields()
        {
            return m_fieldList.Count;
        }


        /// <summary>
        /// Get a field with the specified index.
        /// </summary>
        /// <param name="iIndex">The index to the field.</param>
        /// <returns>The field information.</returns>
        public DataModelFieldInfo GetFieldByIndex( int iIndex )
        {
            if ( iIndex<0 || iIndex>=m_fieldList.Count )
                return null;

            return m_fieldList[ iIndex ];
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Get the information of the property.
        /// </summary>
        /// <param name="iNameCRC">CRC hash code for the property name.</param>
        /// <param name="name">Name of the property.</param>
        /// <param name="type">Type of the property.</param>
        /// <returns>False if the property is not found.</returns>
        public override bool GetPropertyInfo( uint iNameCRC,
                                              out string name,
                                              out Type type )
        {
            name = string.Empty;
            type = null;

            object value = null;
            if ( m_fields.TryGetValue( iNameCRC, out value )==false )
                return false;

            DataModelFieldInfo info = value as DataModelFieldInfo;
            if ( info==null )
                return false;

            name = info.Name;
            type = info.Type;

            return true;
        }

        #endregion

        #region Member variables

        private string m_name = string.Empty;

        private Dictionary<uint,object> m_fields =
            new Dictionary<uint, object>();

        // This list has exactly the same field informations as m_fields,
        // except for the fields are saved in the same order they were
        // added.
        private List<DataModelFieldInfo> m_fieldList =
            new List<DataModelFieldInfo>();

        #endregion
    }

    #endregion

    #region Class for data model serialize list

    public class DataModelSerializeList : List<ICustomDataModelItem>, IDataModelSerializeList
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public DataModelSerializeList()
        {
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="name">Name of the data model.</param>
        /// <param name="id">ID of the data model.</param>
        public DataModelSerializeList( DataModelInfo dataModel )
        {
            this.Name = string.Copy(dataModel.Name);
            this.ID   = dataModel.ID;

            int iNumFields = dataModel.GetNumFields();
            for ( int i=0;i<iNumFields;++i )
            {
                this.Add( new DataModelSerializeItem( dataModel.GetFieldByIndex( i ) ) );
            }
        }


        /// <summary>Name of the data model.</summary>
        public string Name { get; private set; }

        /// <summary>ID of the data model.</summary>
        public uint   ID   { get; private set; }

        /// <summary>Clone</summary>
        public object Clone()
        {
            var copyItem = new DataModelSerializeList();

            copyItem.Name = string.Copy(this.Name);
            copyItem.ID = this.ID;

            // Copy list
            foreach (var item in this)
            {
                copyItem.Add(item.Clone() as ICustomDataModelItem);
            }

            return copyItem;
        }
    }

    #endregion

    #region Class for data model serialize list item

    /// <summary>
    /// Class for data model serialize list item.
    /// Each of the instances represents a field
    /// in a data model.
    /// </summary>
    public class DataModelSerializeItem : ICustomDataModelItem
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public DataModelSerializeItem()
        {
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="name">Name of the field.</param>
        /// <param name="type">Type of the field.</param>
        /// <param name="value">Value of the field.</param>
        public DataModelSerializeItem(DataModelFieldInfo field)
        {
            this.Name  = string.Copy(field.Name);
            this.Type  = field.Type;
            this.Value = field.Value;
        }


        /// <summary>Name of the field.</summary>
        public string Name  { get; private set; }

        /// <summary>Type of the field.</summary>
        public Type   Type  { get; private set; }

        /// <summary>Value of the field.</summary>
        public object Value
        {
            get { return m_value; }
            private set
            {
                if ( value is ICloneable )
                {
                    m_value = (value as ICloneable).Clone();
                }
                else
                {
                    m_value = value;
                }
            }
        }

        /// <summary>Clone</summary>
        public object Clone()
        {
            var copyItem = new DataModelSerializeItem();

            copyItem.Name = string.Copy(this.Name);
            copyItem.Type = this.Type;

            if (this.Value is ICloneable)
            {
                copyItem.Value = (this.Value as ICloneable).Clone();
            }
            else
            {
                copyItem.Value = this.Value;
            }

            return copyItem;
        }

        private object m_value = null;
    }

    #endregion

    /// <summary>
    /// Class for data model proxy.
    /// </summary>
    [TypeDescriptionProvider(typeof(DataModelTypeDescriptionProvider))]
    public class DataModelProxy : DynamicCRCNamedPropertyContainer, IDataModelProxy
    {
        #region Constructor

        /// <summary>
        /// Constructor.
        /// </summary>
        public DataModelProxy( IDataModelProxyOwner owner )
        {
            m_owner = owner;
        }

        #endregion

        #region Properties

        /// <summary>Get the dictionary of all the dynamic properties.</summary>
        public override Dictionary<uint, object> DynamicProperties
        {
            get { return m_dataModels; }
        }

        #endregion

        #region Data models

        /// <summary>
        /// Begin creating a new data model.
        /// </summary>
        /// <param name="iID">ID for the data model.</param>
        /// <param name="name">Name for the data model.</param>
        /// <returns>The created data model.</returns>
        public bool AddDataModel( DataModelInfo template )
        {
            // Does the data model already exist?
            DataModelInfo dataModel = GetDataModel( template.Name );
            if ( dataModel!=null )
                return false;

            // Create a new data model.
            dataModel = new DataModelInfo( template );
            m_dataModels.Add( dataModel.NameCRC, dataModel );

            dataModel.Enabled = m_owner.IsDataModelEnabled( dataModel.ID );

            return true;
        }


        /// <summary>
        /// Deserialize the user function.
        /// </summary>
        /// <param name="content">The deserialized content of the user function.</param>
        /// <returns>True on success.</returns>
        public bool LoadUserFunction( UserFunctionXML content )
        {
            // Does the data model already exist?
            DataModelInfo dataModel = GetDataModel( content.UserFuncName );
            if ( dataModel==null )
                return false;

            int i;
            int iNumFields = dataModel.GetNumFields();
            for ( i=0;i<iNumFields;++i )
            {
                DataModelFieldInfo info = dataModel.GetFieldByIndex( i );
                if ( info==null )
                    continue;

                // Get the value from source user function.
                object value = content.Get( info.NameCRC );
                if ( value==null )
                {
                    DebugConsole.WriteLine( "User function field {0}.{1} is defined but cannot be found in the loaded file, this field will be skipped.",
                                            content.UserFuncName,
                                            info.Name );
                    continue;
                }

                // Check if the value is valid.
                if ( info.IsValueValid( value )==false )
                {
                    DebugConsole.WriteLine( "Failed loading user function field {0}.{1}, the type does not match.",
                                            content.UserFuncName,
                                            info.Name );
                    continue;
                }

                // Set value
                info.Value = value;
            }

            return true;
        }


        /// <summary>
        /// Find the data model with the given name.
        /// </summary>
        /// <param name="name">Name of the data model.</param>
        /// <returns>The found data model or null if not found.</returns>
        public DataModelInfo GetDataModel( string name )
        {
            uint iNameCRC = TheApp.CRC32Helper.ComputeCRC32Str( name );

            // Does the data model already exist?
            object value = null;
            if ( m_dataModels.TryGetValue( iNameCRC, out value )==true )
            {
                return value as DataModelInfo;
            }

            return null;
        }


        /// <summary>
        /// Get data model for serialization.
        /// </summary>
        /// <param name="iNameCRC">CRC hash code of the name of the data model.</param>
        /// <returns>The data model serialize list object.</returns>
        public DataModelSerializeList GetDataModelForSerialize( uint iNameCRC )
        {
            // Get the data model.
            object value = null;
            if ( m_dataModels.TryGetValue( iNameCRC, out value )==false )
                return null;

            DataModelInfo dataModel = value as DataModelInfo;
            if ( dataModel==null )
                return null;

            DataModelSerializeList list = new DataModelSerializeList( dataModel );

            return list;
        }


        /// <summary>
        /// Get all the enabled data models for serialization.
        /// </summary>
        /// <returns>A list of all the enabled data model serialize list object.</returns>
        public IList<IDataModelSerializeList> GetAllDataModelsForSerialize()
        {
            List<IDataModelSerializeList> dataModels = new List<IDataModelSerializeList>();

            foreach ( object item in m_dataModels.Values )
            {
                DataModelInfo dataModel = item as DataModelInfo;
                if ( dataModel==null ||
                     dataModel.Enabled==false )
                {
                    continue;
                }

                DataModelSerializeList serializeObj =
                    new DataModelSerializeList( dataModel );

                dataModels.Add( serializeObj );
            }

            return dataModels;
        }

        #endregion

        #region Utility methods

        /// <summary>
        /// Get the information of the property.
        /// </summary>
        /// <param name="iNameCRC">CRC hash code for the property name.</param>
        /// <param name="name">Name of the property.</param>
        /// <param name="type">Type of the property.</param>
        /// <returns>False if the property is not found.</returns>
        public override bool GetPropertyInfo( uint iNameCRC,
                                              out string name,
                                              out Type type )
        {
            name = string.Empty;
            type = null;

            object value = null;
            if ( m_dataModels.TryGetValue( iNameCRC, out value )==false )
                return false;

            DataModelInfo info = value as DataModelInfo;
            if ( info==null )
                return false;

            name = info.Name;
            type = typeof( DataModelInfo );

            return true;
        }

        #endregion

        #region Member varialbles

        private IDataModelProxyOwner     m_owner;

        private Dictionary<uint, object> m_dataModels =
            new Dictionary<uint, object>();

        #endregion
    }
}
