﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using App.IO;
using NWCore.DataModel;

namespace App.Data
{
    /// <summary>
    /// ドキュメント構成変更イベントハンドラデリゲート。
    /// </summary>
    public delegate void DocumentComponentChangedEventHandler(
                            object sender, DocumentComponentChangedEventArgs e);

    /// <summary>
    /// ドキュメント構成変更イベントデータクラス。
    /// </summary>
    public sealed class DocumentComponentChangedEventArgs : EventArgs
    {
        // 開いたドキュメントリスト
        private readonly List<IDocument> _openedDocuments = new List<IDocument>();
        // 閉じたドキュメントリスト
        private readonly List<IDocument> _closedDocuments = new List<IDocument>();

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public DocumentComponentChangedEventArgs() {}

        /// <summary>
        /// 変更があったか。
        /// </summary>
        public bool HasChanged()
        {
            return (HasOpenedDocument() || HasClosedDocument());
        }

        #region OpenedDocuments
        /// <summary>
        /// 開いたドキュメントリスト。
        /// </summary>
        public ReadOnlyList<IDocument> OpenedDocuments
        {
            get { return new ReadOnlyList<IDocument>(_openedDocuments); }
        }

        /// <summary>
        /// 開いたドキュメントの追加。
        /// </summary>
        public void AddOpenedDocument(IDocument document)
        {
            _openedDocuments.Add(document);
        }

        /// <summary>
        /// 開いたドキュメントがあるか。
        /// </summary>
        public bool HasOpenedDocument()
        {
            return (_openedDocuments.Count > 0);
        }

        /// <summary>
        /// 開くイベントか。
        /// </summary>
        public bool IsOpenEvent()
        {
            return (HasOpenedDocument() && (!HasClosedDocument()));
        }
        #endregion

        #region ClosedDocuments

        /// <summary>
        /// 閉じたドキュメントリスト。
        /// </summary>
        public ReadOnlyList<IDocument> ClosedDocuments
        {
            get { return new ReadOnlyList<IDocument>(_closedDocuments); }
        }

        /// <summary>
        /// 閉じたドキュメントの追加。
        /// </summary>
        public void AddClosedDocument(IDocument document)
        {
            _closedDocuments.Add(document);
        }

        /// <summary>
        /// 閉じたドキュメントがあるか。
        /// </summary>
        public bool HasClosedDocument()
        {
            return (_closedDocuments.Count > 0);
        }

        /// <summary>
        /// 閉じるイベントか。
        /// </summary>
        public bool IsCloseEvent()
        {
            return (HasClosedDocument() && (!HasOpenedDocument()));
        }

        #endregion

#if fasle
        #region Model

        /// <summary>
        /// モデルが開かれたか。
        /// </summary>
        public bool HasOpenedModel()
        {
            return checkExtension(_openedDocuments, DocumentConstants.Fmdl);
        }

        /// <summary>
        /// モデルが閉じられたか。
        /// </summary>
        public bool HasClosedModel()
        {
            return checkExtension(_closedDocuments, DocumentConstants.Fmdl);
        }

        /// <summary>
        /// モデルに変更があったか。
        /// </summary>
        public bool HasChangedModel()
        {
            return (HasOpenedModel() || HasClosedModel());
        }
        #endregion

        #region Texture

        /// <summary>
        /// テクスチャが開かれたか。
        /// </summary>
        public bool HasOpenedTexture()
        {
            return checkExtension(_openedDocuments, DocumentConstants.Ftex);
        }

        /// <summary>
        /// テクスチャが閉じられたか。
        /// </summary>
        public bool HasClosedTexture()
        {
            return checkExtension(_closedDocuments, DocumentConstants.Ftex);
        }

        /// <summary>
        /// テクスチャに変更があったか。
        /// </summary>
        public bool HasChangedTexture()
        {
            return (HasOpenedTexture() || HasClosedTexture());
        }
        #endregion
#endif
        #region Utility

        // 指定した拡張子のドキュメントを持っているかチェック
        private bool checkExtension(List<IDocument> documents, string fileExt)
        {
            foreach (IDocument document in documents)
            {
                if (document.FileExt == fileExt)
                {
                    return true;
                }
            }
            return false;
        }

        #endregion
    }
}
