﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using NWCore.DataModel;
using NWCore.Protocols;

namespace App.Data
{
    /// <summary>
    /// ビューアからの要求オブジェクトお
    /// </summary>
    public class RequireDocument : BaseProjectPanelDocument, IRequireDocument
    {
        RequireDocData m_dataModel;
        IProjectDocument m_projectDoc;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RequireDocument(IProjectDocument projectDoc, RequireDocData dataModel)
            : base(dataModel.Name)
        {
            m_dataModel = dataModel;
            m_projectDoc = projectDoc;
        }

        /// <summary>
        /// 名前
        /// </summary>
        public override string Name
        {
            get
            {
                return (m_dataModel != null ? m_dataModel.Name : base.Name);
            }
            set
            {
                if (m_dataModel != null)
                {
                    m_dataModel.Name = value;
                }
                base.Name = value;
            }
        }

        /// <summary>
        /// 要求ノード名
        /// </summary>
        public string RequireName
        {
            get { return m_dataModel.Name; }
            set { m_dataModel.Name = value; }
        }

        /// <summary>
        /// 要求タイプ
        /// </summary>
        public BinaryDataType BinaryType
        {
            get { return m_dataModel.BinaryType; }
            set { m_dataModel.BinaryType = value; }
        }

        /// <summary>
        /// データモデル
        /// </summary>
        public RequireDocData DataModel
        {
            get { return m_dataModel; }
        }

        /// <summary>
        /// オブジェクトＩＤ
        /// </summary>
        public override GuiObjectID ObjectID
        {
            get { return GuiObjectID.Require; }
        }

        /// <summary>
        /// data source path without data model
        /// </summary>
        public override string RelativeDataScrPath
        {
            get
            {
                return "RequireDocData";
            }
        }

        /// <summary>
        /// path of the data source
        /// </summary>
        public override string DataScrPath
        {
            get
            {
                return m_projectDoc.DataScrPath +
                       ".Require_uid" +
                       this.DocumentID.ToString();
            }
        }

        /// <summary>
        /// 親のドキュメント
        /// </summary>
        public override IDocument OwnerDocument
        {
            get { return m_projectDoc; }
        }


        /// <summary>
        /// Required document 用のパスを取得します。（環境変数対応）
        /// </summary>
        /// <returns>Full Path</returns>
        public static string GetFullPathForRequireDocument(string requirePath)
        {
            if (String.IsNullOrEmpty(requirePath))
            {
                return String.Empty;
            }

            string resultPath = requirePath.Clone() as String;

            // 環境変数があったら置き換える
            int ofsIndex = 2;
            int ofsLength = 3;
            var collect = Regex.Matches(requirePath, @"\$\(\w+\)", RegexOptions.IgnoreCase);
            if (collect.Count == 0)
            {
                ofsIndex = 1;
                ofsLength = 2;
                collect = Regex.Matches(requirePath, @"\%\w+\%", RegexOptions.IgnoreCase);
            }
            foreach (Match item in collect)
            {
                string varName = item.Value.Substring(ofsIndex, item.Value.Length - ofsLength);
                string envVar = Environment.GetEnvironmentVariable(varName);
                if (String.IsNullOrEmpty(envVar) == false)
                {
                    resultPath = requirePath.Replace(item.Value, envVar);
                }
                else
                {
                    TheApp.Logger.Info.AddMessage(
                        res.Strings.WARNING_NOTFIND_ENVVAR + String.Format("\"{0}\"", varName));
                }
            }

            return resultPath;
        }
    }
}
