﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;
using System.Globalization;

namespace App.Data
{
    #region Exception class for XML deserialization

    /// <summary>
    /// Exception class for XML deserialization.
    /// </summary>
    public class XmlReadException : Exception
    {
    }

    #endregion

    #region Base class for user UI control definitions

    /// <summary>
    /// The base class for user UI control definitions.
    /// </summary>
    public class BaseUserUIControlDef
    {
        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public BaseUserUIControlDef()
        {
            this.Index = 0;
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        /// <returns>True on success.</returns>
        public virtual bool Init()
        {
            // We don't initialize the index
            // for it should be set by the group.
            this.Label   = string.Empty;
            this.Visible = true;

            return true;
        }

        #endregion

        #region Properties

        /// <summary>Get or set the index to this control definition in the group.</summary>
        public int Index { get; set; }

        /// <summary>Get or set the label of the control.</summary>
        public string Label { get; set; }

        /// <summary>Get or set the flag indicating whether this control is visible.</summary>
        public bool Visible { get; set; }

        #endregion

        #region Load definition

        /// <summary>
        /// Load the definition from the given XML element.
        /// </summary>
        /// <param name="xml">The XML element containing the definition.</param>
        /// <param name="template">The XML element for the control definition template.</param>
        /// <returns>True on success.</returns>
        public virtual bool LoadDefinition( XmlElement xml,
                                            XmlElement template )
        {
            ////// Load the label for the control definition.
            ////if ( xml.HasAttribute(App.Controls.UIManager.XMLUIPrimaryLabelNodeName)==true )
            ////{
            ////    this.Label = xml.GetAttribute( App.Controls.UIManager.XMLUIPrimaryLabelNodeName );
            ////}
            ////else if ( template!=null &&
            ////          template.HasAttribute(App.Controls.UIManager.XMLUIPrimaryLabelNodeName)==true )
            ////{
            ////    this.Label = template.GetAttribute( App.Controls.UIManager.XMLUIPrimaryLabelNodeName );
            ////}
            ////else if ( xml.HasAttribute(App.Controls.UIManager.XMLUISecondaryLabelNodeName)==true )
            ////{
            ////    this.Label = xml.GetAttribute( App.Controls.UIManager.XMLUISecondaryLabelNodeName );
            ////}
            ////else if ( template!=null &&
            ////          template.HasAttribute(App.Controls.UIManager.XMLUISecondaryLabelNodeName)==true )
            ////{
            ////    this.Label = template.GetAttribute( App.Controls.UIManager.XMLUISecondaryLabelNodeName );
            ////}

            // Load the visibility flag for the control definition.
            if ( xml.HasAttribute("Visible")==true )
            {
                bool bVisible = true;
                if ( bool.TryParse( xml.GetAttribute("Visible"), out bVisible )==true )
                    this.Visible = bVisible;
            }
            else if ( template!=null &&
                      template.HasAttribute("Visible")==true )
            {
                bool bVisible = true;
                if ( bool.TryParse( template.GetAttribute("Visible"), out bVisible )==true )
                    this.Visible = bVisible;
            }

            return true;
        }

        #endregion
    }

    #endregion

    #region Class for user UI check box definitions

    /// <summary>
    /// The class for user UI check box definitions.
    /// </summary>
    public class CheckBoxUserUIDef : BaseUserUIControlDef
    {
        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public CheckBoxUserUIDef() :
            base()
        {
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Init()
        {
            if ( base.Init()==false )
                return false;

            this.Label        = StringResource.Get( "UI_USER_UI_DEFINITION_DEFAULT_LABEL_CHECKBOX",
                                                    this.Index );
            this.DefaultValue = false;

            return true;
        }

        #endregion

        #region Properties

        /// <summary>Get or set the default value for the definition.</summary>
        public bool DefaultValue { get; set; }

        #endregion

        #region Load definition

        /// <summary>
        /// Load the definition from the given XML element.
        /// </summary>
        /// <param name="xml">The XML element containing the definition.</param>
        /// <param name="template">The XML element for the control definition template.</param>
        /// <returns>True on success.</returns>
        public override bool LoadDefinition( XmlElement xml,
                                             XmlElement template )
        {
            if ( base.LoadDefinition(xml, template)==false )
                return false;

            // Load the default value for the check box definition.
            if ( xml.HasAttribute("Default")==true )
            {
                bool bChecked = false;
                if ( bool.TryParse( xml.GetAttribute("Default"), out bChecked )==true )
                    this.DefaultValue = bChecked;
            }
            else if ( template!=null &&
                      template.HasAttribute("Default")==true )
            {
                bool bChecked = false;
                if ( bool.TryParse( template.GetAttribute("Default"), out bChecked )==true )
                    this.DefaultValue = bChecked;
            }

            return true;
        }

        #endregion
    }

    #endregion

    #region Class for user UI radio button definitions

    /// <summary>
    /// The class for user UI radio button definitions.
    /// </summary>
    public class RadioButtonUserUIDef : BaseUserUIControlDef
    {
        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public RadioButtonUserUIDef() :
            base()
        {
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Init()
        {
            if ( base.Init()==false )
                return false;

            this.Label        = StringResource.Get( "UI_USER_UI_DEFINITION_DEFAULT_LABEL_RADIOBUTTON",
                                                    this.Index );
            this.DefaultValue = false;

            return true;
        }

        #endregion

        #region Properties

        /// <summary>Get or set the default value for the definition.</summary>
        public bool DefaultValue { get; set; }

        #endregion

        #region Load definition

        /// <summary>
        /// Load the definition from the given XML element.
        /// </summary>
        /// <param name="xml">The XML element containing the definition.</param>
        /// <param name="template">The XML element for the control definition template.</param>
        /// <returns>True on success.</returns>
        public override bool LoadDefinition( XmlElement xml,
                                             XmlElement template )
        {
            if ( base.LoadDefinition(xml, template)==false )
                return false;

            // Load the default value for the check box definition.
            if ( xml.HasAttribute("Default")==true )
            {
                bool bChecked = false;
                if ( bool.TryParse( xml.GetAttribute("Default"), out bChecked )==true )
                    this.DefaultValue = bChecked;
            }
            else if ( template!=null &&
                      template.HasAttribute("Default")==true )
            {
                bool bChecked = false;
                if ( bool.TryParse( template.GetAttribute("Default"), out bChecked )==true )
                    this.DefaultValue = bChecked;
            }

            return true;
        }

        #endregion
    }

    #endregion

    #region Class for user UI float slider definitions

    /// <summary>
    /// The class for user UI slider definitions.
    /// </summary>
    public class FloatSliderUserUIDef : BaseUserUIControlDef
    {
        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public FloatSliderUserUIDef() :
            base()
        {
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Init()
        {
            if ( base.Init()==false )
                return false;

            this.Label        = StringResource.Get( "UI_USER_UI_DEFINITION_DEFAULT_LABEL_SLIDER",
                                                    this.Index );

            this.DefaultValue = 0.0f;
            this.FixRange     = false;
            this.InitMinRange = -10.0f;
            this.InitMaxRange = 10.0f;
            this.MinMaxRange  = 0.0001f;
            this.MaxMaxRange  = 1000000.0f;
            this.FixMinRange  = false;

            return true;
        }

        #endregion

        #region Properties

        /// <summary>Get or set the default value for the definition.</summary>
        public float DefaultValue { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the range of the slider should be fixed.
        /// </summary>
        public bool FixRange { get; set; }

        /// <summary>Get or set the initial minimum range of the value.</summary>
        public float InitMinRange { get; set; }

        /// <summary>Get or set the initial maximum range of the value.</summary>
        public float InitMaxRange { get; set; }

        /// <summary>
        /// Get or set the minimum of the maximum range of the value.
        /// This is only effective when the range is not fixed.
        /// </summary>
        public float MinMaxRange { get; set; }

        /// <summary>
        /// Get or set the maximum of the maximum range of the value.
        /// This is only effective when the range is not fixed.
        /// </summary>
        public float MaxMaxRange { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the minimum range of the slider should be fixed.
        /// </summary>
        public bool FixMinRange { get; set; }

        #endregion

        #region Load definition

        /// <summary>
        /// Load the definition from the given XML element.
        /// </summary>
        /// <param name="xml">The XML element containing the definition.</param>
        /// <param name="template">The XML element for the control definition template.</param>
        /// <returns>True on success.</returns>
        public override bool LoadDefinition( XmlElement xml,
                                             XmlElement template )
        {
            if ( base.LoadDefinition(xml, template)==false )
                return false;

            // Load the default value for the slider definition.
            if ( xml.HasAttribute("Default")==true )
            {
                float fValue = 0.0f;
                if ( float.TryParse( xml.GetAttribute("Default"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.DefaultValue = fValue;
            }
            else if ( template!=null &&
                      template.HasAttribute("Default")==true )
            {
                float fValue = 0.0f;
                if ( float.TryParse( template.GetAttribute("Default"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.DefaultValue = fValue;
            }

            // Load the flag indicating whether the value range is fixed.
            if ( xml.HasAttribute("FixRange")==true )
            {
                bool bFixRange = false;
                if ( bool.TryParse( xml.GetAttribute("FixRange"), out bFixRange )==true )
                    this.FixRange = bFixRange;
            }
            else if ( template!=null &&
                      template.HasAttribute("FixRange")==true )
            {
                bool bFixRange = false;
                if ( bool.TryParse( template.GetAttribute("FixRange"), out bFixRange )==true )
                    this.FixRange = bFixRange;
            }


            // Load the initial minimum range for the slider definition.
            if ( xml.HasAttribute("InitMinRange")==true )
            {
                float fValue = -10.0f;
                if ( float.TryParse( xml.GetAttribute("InitMinRange"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.InitMinRange = fValue;
            }
            else if ( template!=null &&
                      template.HasAttribute("InitMinRange")==true )
            {
                float fValue = -10.0f;
                if ( float.TryParse( template.GetAttribute("InitMinRange"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.InitMinRange = fValue;
            }

            // Load the initial maximum range for the slider definition.
            if ( xml.HasAttribute("InitMaxRange")==true )
            {
                float fValue = 10.0f;
                if ( float.TryParse( xml.GetAttribute("InitMaxRange"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.InitMaxRange = fValue;
            }
            else if ( template!=null &&
                      template.HasAttribute("InitMaxRange")==true )
            {
                float fValue = 10.0f;
                if ( float.TryParse( template.GetAttribute("InitMaxRange"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.InitMaxRange = fValue;
            }

            // Load the minimum of the maximum range for the slider definition.
            if ( xml.HasAttribute("MinMaxRange")==true )
            {
                float fValue = 0.0001f;
                if ( float.TryParse( xml.GetAttribute("MinMaxRange"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.MinMaxRange = fValue;
            }
            else if ( template!=null &&
                      template.HasAttribute("MinMaxRange")==true )
            {
                float fValue = 0.0001f;
                if ( float.TryParse( template.GetAttribute("MinMaxRange"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.MinMaxRange = fValue;
            }

            // Load the maximum of the maximum range for the slider definition.
            if ( xml.HasAttribute("MaxMaxRange")==true )
            {
                float fValue = 1000000.0f;
                if ( float.TryParse( xml.GetAttribute("MaxMaxRange"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.MaxMaxRange = fValue;
            }
            else if ( template!=null &&
                      template.HasAttribute("MaxMaxRange")==true )
            {
                float fValue = 1000000.0f;
                if ( float.TryParse( template.GetAttribute("MaxMaxRange"),
                                     NumberStyles.Float,
                                     CultureInfo.InvariantCulture,
                                     out fValue )==true )
                    this.MaxMaxRange = fValue;
            }

            // Load the flag indicating whether the minimum range is fixed.
            if ( xml.HasAttribute("FixMinRange")==true )
            {
                bool bFixMinRange = false;
                if ( bool.TryParse( xml.GetAttribute("FixMinRange"), out bFixMinRange )==true )
                    this.FixMinRange = bFixMinRange;
            }
            else if ( template!=null &&
                      template.HasAttribute("FixMinRange")==true )
            {
                bool bFixMinRange = false;
                if ( bool.TryParse( template.GetAttribute("FixMinRange"), out bFixMinRange )==true )
                    this.FixMinRange = bFixMinRange;
            }

            return true;
        }

        #endregion
    }

    #endregion

    #region Class for user UI integer slider definitions

    /// <summary>
    /// The class for user UI slider definitions.
    /// </summary>
    public class IntSliderUserUIDef : BaseUserUIControlDef
    {
        #region Constructor

        /// <summary>
        /// Default constructor.
        /// </summary>
        public IntSliderUserUIDef() :
            base()
        {
        }

        #endregion

        #region Initialize

        /// <summary>
        /// Initialize the data.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Init()
        {
            if ( base.Init()==false )
                return false;

            this.Label        = StringResource.Get( "UI_USER_UI_DEFINITION_DEFAULT_LABEL_SLIDER",
                                                    this.Index );

            this.DefaultValue = 0;
            this.FixRange     = true;
            this.InitMinRange = 0;
            this.InitMaxRange = 255;
            this.MinMaxRange  = 10;
            this.MaxMaxRange  = 1000000;
            this.FixMinRange  = true;

            return true;
        }

        #endregion

        #region Properties

        /// <summary>Get or set the default value for the definition.</summary>
        public int DefaultValue { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the range of the slider should be fixed.
        /// </summary>
        public bool FixRange { get; set; }

        /// <summary>Get or set the initial minimum range of the value.</summary>
        public int InitMinRange { get; set; }

        /// <summary>Get or set the initial maximum range of the value.</summary>
        public int InitMaxRange { get; set; }

        /// <summary>
        /// Get or set the minimum of the maximum range of the value.
        /// This is only effective when the range is not fixed.
        /// </summary>
        public int MinMaxRange { get; set; }

        /// <summary>
        /// Get or set the maximum of the maximum range of the value.
        /// This is only effective when the range is not fixed.
        /// </summary>
        public int MaxMaxRange { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the minimum range of the slider should be fixed.
        /// </summary>
        public bool FixMinRange { get; set; }

        #endregion

        #region Load definition

        /// <summary>
        /// Load the definition from the given XML element.
        /// </summary>
        /// <param name="xml">The XML element containing the definition.</param>
        /// <param name="template">The XML element for the control definition template.</param>
        /// <returns>True on success.</returns>
        public override bool LoadDefinition( XmlElement xml,
                                             XmlElement template )
        {
            if ( base.LoadDefinition(xml, template)==false )
                return false;

            // Load the default value for the slider definition.
            if ( xml.HasAttribute("Default")==true )
            {
                int iValue = 0;
                if ( int.TryParse(xml.GetAttribute("Default"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.DefaultValue = iValue;
                }
            }
            else if ( template!=null &&
                      template.HasAttribute("Default")==true )
            {
                int iValue = 0;
                if ( int.TryParse(template.GetAttribute("Default"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.DefaultValue = iValue;
                }
            }

            // Load the flag indicating whether the value range is fixed.
            if ( xml.HasAttribute("FixRange")==true )
            {
                bool bFixRange = true;
                if ( bool.TryParse( xml.GetAttribute("FixRange"), out bFixRange )==true )
                    this.FixRange = bFixRange;
            }
            else if ( template!=null &&
                      template.HasAttribute("FixRange")==true )
            {
                bool bFixRange = false;
                if ( bool.TryParse( template.GetAttribute("FixRange"), out bFixRange )==true )
                    this.FixRange = bFixRange;
            }


            // Load the initial minimum range for the slider definition.
            if ( xml.HasAttribute("InitMinRange")==true )
            {
                int iValue = 0;
                if ( int.TryParse(xml.GetAttribute("InitMinRange"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.InitMinRange = iValue;
                }
            }
            else if ( template!=null &&
                      template.HasAttribute("InitMinRange")==true )
            {
                int iValue = 0;
                if ( int.TryParse(template.GetAttribute("InitMinRange"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.InitMinRange = iValue;
                }
            }

            // Load the initial maximum range for the slider definition.
            if ( xml.HasAttribute("InitMaxRange")==true )
            {
                int iValue = 255;
                if ( int.TryParse(xml.GetAttribute("InitMaxRange"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.InitMaxRange = iValue;
                }
            }
            else if ( template!=null &&
                      template.HasAttribute("InitMaxRange")==true )
            {
                int iValue = 255;
                if ( int.TryParse(template.GetAttribute("InitMaxRange"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.InitMaxRange = iValue;
                }
            }

            // Load the minimum of the maximum range for the slider definition.
            if ( xml.HasAttribute("MinMaxRange")==true )
            {
                int iValue = 10;
                if ( int.TryParse(xml.GetAttribute("MinMaxRange"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.MinMaxRange = iValue;
                }
            }
            else if ( template!=null &&
                      template.HasAttribute("MinMaxRange")==true )
            {
                int iValue = 10;
                if ( int.TryParse(template.GetAttribute("MinMaxRange"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.MinMaxRange = iValue;
                }
            }

            // Load the maximum of the maximum range for the slider definition.
            if ( xml.HasAttribute("MaxMaxRange")==true )
            {
                int iValue = 1000000;
                if ( int.TryParse(xml.GetAttribute("MaxMaxRange"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.MaxMaxRange = iValue;
                }
            }
            else if ( template!=null &&
                      template.HasAttribute("MaxMaxRange")==true )
            {
                int iValue = 1000000;
                if ( int.TryParse(template.GetAttribute("MaxMaxRange"),
                                  NumberStyles.Integer,
                                  CultureInfo.InvariantCulture,
                                  out iValue)==true )
                {
                    this.MaxMaxRange = iValue;
                }
            }

            // Load the flag indicating whether the minimum range is fixed.
            if ( xml.HasAttribute("FixMinRange")==true )
            {
                bool bFixMinRange = true;
                if ( bool.TryParse( xml.GetAttribute("FixMinRange"), out bFixMinRange )==true )
                    this.FixMinRange = bFixMinRange;
            }
            else if ( template!=null &&
                      template.HasAttribute("FixMinRange")==true )
            {
                bool bFixMinRange = false;
                if ( bool.TryParse( template.GetAttribute("FixMinRange"), out bFixMinRange )==true )
                    this.FixMinRange = bFixMinRange;
            }

            return true;
        }

        #endregion
    }

    #endregion
}
