﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the binary conversion source data model definition.
    /// </summary>
    public class SourceDataModelInstanceViewModel : ViewModelBase
    {
        /// <summary>The source data model property view models.</summary>
        private List<SourcePropertyInstanceViewModel> propertyViewModels =
            new List<SourcePropertyInstanceViewModel>();

        /// <summary>The property view model that is currently selected.</summary>
        private SourcePropertyInstanceViewModel selectedPropertyViewModel = null;

        /// <summary>The data model instance definition.</summary>
        private SourceDataModelInstanceDefinition instanceDefinition = null;

        /// <summary>The data model definition of the data model instance.</summary>
        private EditorDataModelDefinition dataModelDefinition = null;

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public SourceDataModelInstanceViewModel(
            ViewModelBase parent,
            SourceDataModelInstanceDefinition dataModel) :
            base(parent, dataModel)
        {
            this.instanceDefinition = dataModel;

            // Find the data model definition for the data model instance.
            this.dataModelDefinition =
                WorkspaceManager.FindDefinition(dataModel.DataModelDefinitionGuid) as EditorDataModelDefinition;

            foreach (SourcePropertyInstanceDefinition def in dataModel.Properties)
            {
                var child = new SourcePropertyInstanceViewModel(this, def);
                this.Children.Add(child);
                this.propertyViewModels.Add(child);
            }
        }

        /// <summary>
        /// Get the namespace of the data model.
        /// </summary>
        public string DataModelNamespace
        {
            get { return this.dataModelDefinition.Namespace; }
        }

        /// <summary>
        /// Get the name of the data model.
        /// </summary>
        public string DataModelName
        {
            get { return this.dataModelDefinition.Name; }
        }

        /// <summary>
        /// Get the description of the data model.
        /// </summary>
        public string DataModelDescription
        {
            get { return this.dataModelDefinition.Description; }
        }

        /// <summary>
        /// Enumerate the source data model property view models.
        /// </summary>
        public IEnumerable PropertyInstanceViewModels
        {
            get { return this.propertyViewModels; }
        }

        /// <summary>
        /// Get or set the selected property instance view model.
        /// </summary>
        public object SelectedPropertyViewModel
        {
            get
            {
                return this.selectedPropertyViewModel;
            }

            set
            {
                this.SetValue(
                    ref this.selectedPropertyViewModel,
                    value as SourcePropertyInstanceViewModel);
            }
        }

        /// <summary>
        /// Get the Guid of the data model definition.
        /// </summary>
        public EditorDataModelDefinition DataModelDefinition
        {
            get { return this.dataModelDefinition; }
        }
    }
}
