﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the editor data model property definition.
    /// </summary>
    public class EditorDataModelPropertyDefinitionViewModel : ViewModelBase
    {
        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public EditorDataModelPropertyDefinitionViewModel(
            ViewModelBase parent,
            EditorDataModelPropertyDefinition dataModel) :
            base(parent, dataModel)
        {
            this.OnCommitEditingExecutable = new AnonymousExecutable(this.OnCommitEditing);
            this.OnCancelEditingExecutable = new AnonymousExecutable(this.OnCancelEditing);
            this.OnLockEditingVersionExecutable = new AnonymousExecutable(this.OnLockEditingVersion);
            this.OnUnlockEditingVersionExecutable = new AnonymousExecutable(this.OnUnlockEditingVersion);
            this.OnAddVersionExecutable = new AnonymousExecutable(this.OnAddVersion);
            this.OnRemoveVersionExecutable = new AnonymousExecutable(this.OnRemoveVersion);
            this.OnMoveVersionExecutable = new AnonymousExecutable(this.OnMoveVersion);
        }

        /// <summary>
        /// Static event triggered when the data model property is modified.
        /// </summary>
        public static event EventHandler DataModelPropertyModified = null;

        /// <summary>
        /// Enumerates all versions and the action took to the property.
        /// </summary>
        public IEnumerable<KeyValuePair<Version, VersionActions>> VersionsAndActions
        {
            get
            {
                var rootDataModel = this.DataModel.FindParent<EditorDataModelRootDefinition>();
                if (rootDataModel == null)
                {
                    yield break;
                }

                var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
                foreach (var v in rootDataModel.Versions)
                {
                    var versionDef = myDataModel.Versions.FirstOrDefault(
                        vd => vd.Version.Version == v.Version.Version);

                    if (versionDef != null)
                    {
                        yield return new KeyValuePair<Version, VersionActions>(v.Version, versionDef.Action);
                    }
                    else
                    {
                        yield return new KeyValuePair<Version, VersionActions>(v.Version, VersionActions.None);
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the type of the property without the namespace.
        /// </summary>
        public string PropertyTypeWithoutNamespace
        {
            get
            {
                var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
                if (myDataModel == null)
                {
                    return string.Empty;
                }

                if (string.IsNullOrEmpty(myDataModel.Type) == true)
                {
                    return string.Empty;
                }

                // Find the information of the property value type.
                EditorTypeInfo primaryInfo =
                    TypeManager.FindEditorType(myDataModel.Namespace, myDataModel.Type);
                if (primaryInfo == null)
                {
                    return myDataModel.Type;
                }

                // Compose the complete type string.
                string typeString = primaryInfo.TypeName;
                if (primaryInfo.IsGeneric == true)
                {
                    EditorTypeInfo elementInfo =
                       TypeManager.FindEditorType(myDataModel.ElementNamespace, myDataModel.ElementType);
                    if (elementInfo != null)
                    {
                        typeString += "<" + elementInfo.TypeName + ">";
                    }
                    else
                    {
                        typeString += "<" + myDataModel.ElementType + ">";
                    }
                }

                return typeString;
            }
        }

        /// <summary>
        /// Get or set the type of the property.
        /// </summary>
        public string PropertyType
        {
            get
            {
                var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
                if (myDataModel == null)
                {
                    return string.Empty;
                }

                if (string.IsNullOrEmpty(myDataModel.Type) == true)
                {
                    return string.Empty;
                }

                // Find the information of the property value type.
                EditorTypeInfo primaryInfo =
                    TypeManager.FindEditorType(myDataModel.Namespace, myDataModel.Type);
                if (primaryInfo == null)
                {
                    if (string.IsNullOrEmpty(myDataModel.Namespace) == true)
                    {
                        return myDataModel.Type;
                    }
                    else
                    {
                        return myDataModel.Namespace + "." + myDataModel.Type;
                    }
                }

                // Compose the complete type string.
                string typeString = primaryInfo.FullName;
                if (primaryInfo.IsGeneric == true &&
                    string.IsNullOrEmpty(myDataModel.ElementType) == false)
                {
                    EditorTypeInfo elementInfo =
                       TypeManager.FindEditorType(myDataModel.ElementNamespace, myDataModel.ElementType);
                    if (elementInfo != null)
                    {
                        typeString += "<" + elementInfo.FullName + ">";
                    }
                    else
                    {
                        if (string.IsNullOrEmpty(myDataModel.ElementNamespace) == true)
                        {
                            typeString += "<" + myDataModel.ElementType + ">";
                        }
                        else
                        {
                            typeString += "<" + myDataModel.ElementNamespace + "." + myDataModel.ElementType + ">";
                        }
                    }
                }

                return typeString;
            }

            set
            {
                var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
                if (myDataModel == null)
                {
                    return;
                }

                string primaryNamespace;
                string primaryTypeName;
                string elementNamespace;
                string elementTypeName;
                bool result = TypeManager.ParseEditorType(
                    value,
                    out primaryNamespace,
                    out primaryTypeName,
                    out elementNamespace,
                    out elementTypeName);
                if (result == false)
                {
                    this.OnPropertyChanged("IsPropertyTypeInvalid");
                    return;
                }

                myDataModel.Namespace = primaryNamespace;
                myDataModel.Type = primaryTypeName;
                myDataModel.ElementNamespace = elementNamespace;
                myDataModel.ElementType = elementTypeName;

                this.OnPropertyChanged("PropertyType");
                this.OnPropertyChanged("PropertyTypeWithoutNamespace");
                this.OnPropertyChanged("IsPropertyTypeInvalid");
            }
        }

        /// <summary>
        /// Get the flag indicating whether the property type is invalid or not.
        /// </summary>
        public bool IsPropertyTypeInvalid
        {
            get
            {
                var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
                if (myDataModel == null)
                {
                    return true;
                }

                if (string.IsNullOrEmpty(myDataModel.Type) == true)
                {
                    return true;
                }

                // Find the information of the property value type.
                EditorTypeInfo primaryInfo =
                    TypeManager.FindEditorType(myDataModel.Namespace, myDataModel.Type);
                if (primaryInfo == null)
                {
                    return true;
                }

                // Find the element type info if the primary type is a generic type.
                if (primaryInfo.IsGeneric == true)
                {
                    EditorTypeInfo elementInfo =
                       TypeManager.FindEditorType(myDataModel.ElementNamespace, myDataModel.ElementType);
                    if (elementInfo == null)
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        /// <summary>
        /// Get or set the executable for committing modification of the property.
        /// </summary>
        public IExecutable OnCommitEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for canceling modification of the property.
        /// </summary>
        public IExecutable OnCancelEditingExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for locking the editing version of the property.
        /// </summary>
        public IExecutable OnLockEditingVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for unlocking the editing version of the property.
        /// </summary>
        public IExecutable OnUnlockEditingVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for adding new version to the property.
        /// </summary>
        public IExecutable OnAddVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for removing version from the property.
        /// </summary>
        public IExecutable OnRemoveVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for moving a version of the property to another version.
        /// </summary>
        public IExecutable OnMoveVersionExecutable { get; set; }

        /// <summary>
        /// Set editing version of the property.
        /// </summary>
        /// <param name="version">The version to set to.</param>
        public void SetEditingVersion(Version version)
        {
            var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
            if (myDataModel.EditingVersionLock != null)
            {
                myDataModel.EditingVersionLock.Dispose();
            }

            myDataModel.EditingVersionLock = new EditingVersionLock(myDataModel, version);

            // Notify UI to update.
            var myParent = this.Parent as EditorDataModelDefinitionViewModel;
            if (myParent != null)
            {
                myParent.NotifyPropertyDefinitionChanged(this);
            }
        }

        /// <summary>
        /// Set editing version of the property to the latest version.
        /// </summary>
        public void SetEditingVersionToLatest()
        {
            var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
            if (myDataModel.EditingVersionLock != null)
            {
                myDataModel.EditingVersionLock.Dispose();
            }

            myDataModel.EditingVersionLock = null;

            // Notify UI to update.
            var myParent = this.Parent as EditorDataModelDefinitionViewModel;
            if (myParent != null)
            {
                myParent.NotifyPropertyDefinitionChanged(this);
            }
        }

        /// <summary>
        /// Handle execution when committing modification of the property.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCommitEditing(object parameter)
        {
            var values = parameter as KeyValuePair<string, object>[];
            if (values == null)
            {
                return;
            }

            try
            {
                foreach (var pair in values)
                {
                    var binder = new EffectMakerSetMemberBinder(pair.Key, true);
                    this.TrySetMember(binder, pair.Value);
                }

                // Notify UI to update.
                var myParent = this.Parent as EditorDataModelDefinitionViewModel;
                if (myParent != null)
                {
                    myParent.NotifyPropertyDefinitionChanged(this);
                }

                if (DataModelPropertyModified != null)
                {
                    DataModelPropertyModified(this, EventArgs.Empty);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "EditorDataModelPropertyDefinitionViewModel.OnCommitEditing => Failed setting values to the editor data model property.");
                Logger.Log(LogLevels.Error, "Exception : {0}, message : {1}", ex.GetType().ToString(), ex.Message);
            }
        }

        /// <summary>
        /// Handle execution when canceling modification of the property.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnCancelEditing(object parameter)
        {
            var myParent = this.Parent as EditorDataModelDefinitionViewModel;
            if (myParent == null)
            {
                return;
            }

            myParent.DeleteProperty(this);
        }

        /// <summary>
        /// Handle execution when locking editing version of the property.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnLockEditingVersion(object parameter)
        {
            var version = parameter as Version;
            if (version == null)
            {
                throw new ArgumentException("The parameter is not a version.");
            }

            this.SetEditingVersion(version);
        }

        /// <summary>
        /// Handle execution when unlocking editing version of the property.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnUnlockEditingVersion(object parameter)
        {
            this.SetEditingVersionToLatest();
        }

        /// <summary>
        /// Handle execution when adding new version to the property.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnAddVersion(object parameter)
        {
            var args = parameter as object[];
            if (args == null)
            {
                throw new ArgumentException("parameter");
            }

            var version = args[0] as Version;
            if (version == null)
            {
                throw new ArgumentException("The first parameter is not a version.");
            }

            if ((args[1] is VersionActions) == false)
            {
                throw new ArgumentException("The second parameter is not a version action.");
            }

            var action = (VersionActions)args[1];

            var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
            var rootDataModel = this.DataModel.FindParent<EditorDataModelRootDefinition>();

            // Find the index to the next version definition and
            // the closest property definition.
            // (the previous one or the next one if the new version is the first one)
            int index = -1;
            PropertyDefinitionData closestPropDef = null;
            for (int i = 0; i < myDataModel.Versions.Count; ++i)
            {
                var versionDef = myDataModel.Versions[i];
                if (versionDef.Version > version)
                {
                    if (closestPropDef == null)
                    {
                        // Previous version does not exist, just take
                        // from the next version.
                        closestPropDef = versionDef.PropertyDefinition;
                    }

                    index = i;
                    break;
                }

                closestPropDef = versionDef.PropertyDefinition;
            }

            var newVersionDef = new EditorPropertyVersionDefinition()
            {
                VersionIndex = rootDataModel.FindVersionIndex(version),
                Action = action,
                PropertyDefinition = (PropertyDefinitionData)closestPropDef.Clone(),
            };

            if (index < 0)
            {
                myDataModel.Versions.Add(newVersionDef);
            }
            else
            {
                myDataModel.Versions.Insert(index, newVersionDef);
            }

            // Notify UI to update.
            this.OnPropertyChanged("VersionsAndActions");

            var myParent = this.Parent as EditorDataModelDefinitionViewModel;
            if (myParent != null)
            {
                myParent.NotifyPropertyDefinitionChanged(this);
            }

            if (DataModelPropertyModified != null)
            {
                DataModelPropertyModified(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution when removing version from the property.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnRemoveVersion(object parameter)
        {
            var version = parameter as Version;
            if (version == null)
            {
                throw new ArgumentException("The parameter is not a version.");
            }

            var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
            var versionDef = myDataModel.Versions.FirstOrDefault(vd => vd.Version == version);
            if (versionDef == null)
            {
                return;
            }

            myDataModel.Versions.Remove(versionDef);

            // Notify UI to update.
            this.OnPropertyChanged("VersionsAndActions");

            var myParent = this.Parent as EditorDataModelDefinitionViewModel;
            if (myParent != null)
            {
                myParent.NotifyPropertyDefinitionChanged(this);
            }

            if (DataModelPropertyModified != null)
            {
                DataModelPropertyModified(this, EventArgs.Empty);
            }
        }

        /// <summary>
        /// Handle execution when moving version history of the property to another version.
        /// </summary>
        /// <param name="parameter">The execution parameters.</param>
        private void OnMoveVersion(object parameter)
        {
            var args = parameter as object[];
            if (args == null)
            {
                throw new ArgumentException("parameter");
            }

            var origVersion = args[0] as Version;
            if (origVersion == null)
            {
                throw new ArgumentException("The first parameter is not a version.");
            }

            var newVersion = args[1] as Version;
            if (newVersion == null)
            {
                throw new ArgumentException("The second parameter is not a version.");
            }

            var myDataModel = this.DataModel as EditorDataModelPropertyDefinition;
            var rootDataModel = this.DataModel.FindParent<EditorDataModelRootDefinition>();

            // First find the property version definition.
            var versionDef =
                myDataModel.Versions.FirstOrDefault(def => origVersion == def.Version);
            if (versionDef == null)
            {
                throw new ArgumentException("The definition of the original version is not found.");
            }

            // Find the index of the new version.
            int index = rootDataModel.FindVersionIndex(newVersion);
            if (index < 0)
            {
                throw new ArgumentException("The new version is not found.");
            }

            // Change the version.
            versionDef.VersionIndex = index;

            // Sort the version definitions, the order could change
            // since we are moving one version definition to another
            // version.
            myDataModel.SortVersionDefinitions();

            // Notify UI to update.
            this.OnPropertyChanged("VersionsAndActions");

            var myParent = this.Parent as EditorDataModelDefinitionViewModel;
            if (myParent != null)
            {
                myParent.NotifyPropertyDefinitionChanged(this);
            }

            if (DataModelPropertyModified != null)
            {
                DataModelPropertyModified(this, EventArgs.Empty);
            }
        }
    }
}
