﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;

using EffectMaker.DataModelMaker.Core.Core;
using EffectMaker.DataModelMaker.Core.DataTypes;
using EffectMaker.DataModelMaker.Core.Definitions;

using EffectMaker.Foundation.Dynamic;
using EffectMaker.Foundation.Input;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UILogic.ViewModels
{
    /// <summary>
    /// View model class for the binary data field group.
    /// </summary>
    public class BinaryFieldGroupDefinitionViewModel : ViewModelBase
    {
        /// <summary>The binary field view models.</summary>
        private List<BinaryFieldDefinitionViewModel> fieldViewModels =
            new List<BinaryFieldDefinitionViewModel>();

        /// <summary>
        /// The constructor.
        /// </summary>
        /// <param name="parent">The parent view model.</param>
        /// <param name="dataModel">The data model to encapsulate.</param>
        public BinaryFieldGroupDefinitionViewModel(
            ViewModelBase parent,
            BinaryFieldGroupDefinition dataModel) :
            base(parent, dataModel)
        {
            foreach (BinaryFieldDefinition def in dataModel.Fields)
            {
                var child = new BinaryFieldDefinitionViewModel(this, def);
                this.Children.Add(child);
                this.fieldViewModels.Add(child);
            }
        }

        /// <summary>
        /// Get the flag indicating whether the field group target is a binary data or not.
        /// </summary>
        public bool IsTargetBinaryData
        {
            get
            {
                DefinitionBase def = this.TargetDefinition;
                if (def == null)
                {
                    return false;
                }

                if (def is BinaryDataDefinition)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// Get the name of the binary field group target.
        /// </summary>
        public string TargetName
        {
            get
            {
                string name = string.Empty;
                DefinitionBase def = this.TargetDefinition;
                if (def != null)
                {
                    if (def is BinaryDataDefinition)
                    {
                        return ((BinaryDataDefinition)def).Name;
                    }
                    else if (def is RuntimeDataModelDefinition)
                    {
                        var fieldVM = this.Parent as BinaryFieldDefinitionViewModel;
                        return fieldVM.TargetType;
                    }
                }

                return name;
            }
        }

        /// <summary>
        /// Enumerate the binary field view models.
        /// </summary>
        public IEnumerable<BinaryFieldDefinitionViewModel> BinaryFieldViewModels
        {
            get { return this.fieldViewModels; }
        }

        /// <summary>
        /// Get the target definition.
        /// </summary>
        public DefinitionBase TargetDefinition
        {
            get
            {
                var myDataModel = this.DataModel as BinaryFieldGroupDefinition;
                if (myDataModel == null)
                {
                    return null;
                }

                return WorkspaceManager.FindDefinition(myDataModel.TargetGuid);
            }
        }

        /// <summary>
        /// Update the binary field group according to the target definition.
        /// </summary>
        public void UpdateFromTargetDefinition()
        {
            var myDataModel = this.DataModel as BinaryFieldGroupDefinition;
            if (myDataModel == null)
            {
                return;
            }

            // The target must be a runtime data model definition, otherwise
            // the parent binary field wouldn't have created this field group.
            var dataModelDef = this.TargetDefinition as RuntimeDataModelDefinition;
            if (dataModelDef == null)
            {
                return;
            }

            // First remove the fields that the target property definition no longer exist.
            for (int i = this.fieldViewModels.Count - 1; i >= 0; --i)
            {
                BinaryFieldDefinitionViewModel fieldVM = this.fieldViewModels[i];

                // Does the data model property this field targets to still exist?
                bool propertyExists =
                    (fieldVM.TargetDefinition != null) &&
                    (dataModelDef.Properties.Exists(p => p.Guid == fieldVM.TargetDefinition.Guid) == true);

                // Or maybe this field targets to one of the super class of the data model?
                bool superClassExists =
                    (fieldVM.TargetDefinition != null) &&
                    (dataModelDef.SuperClassGuidList.Exists(guid => guid == fieldVM.TargetDefinition.Guid) == true);

                if (propertyExists == false && superClassExists == false)
                {
                    // Remove the binary field because the target doesn't exist anymore.
                    this.Children.Remove(fieldVM);
                    this.fieldViewModels.RemoveAt(i);

                    var fieldDef = fieldVM.DataModel as BinaryFieldDefinition;
                    if (fieldDef != null)
                    {
                        myDataModel.Fields.Remove(fieldDef);
                    }

                    fieldDef.Dispose();
                    fieldVM.Dispose();
                }
            }

            // Temporarily save the binary fields to a map
            // with the target definition Guid as the key.
            var tmpFieldMap = this.fieldViewModels.ToDictionary(
                vm => vm.TargetDefinition.Guid);

            // Clear the binary field list.
            // We will add them back in the order of the property
            // definitions in the data model definition, so that
            // they will all be in the correct order.
            this.fieldViewModels.Clear();
            myDataModel.Fields.Clear();

            // Loop through the super classes of the target data model definition.
            foreach (Guid guid in dataModelDef.SuperClassGuidList)
            {
                // Find the binary field view model that targets this super class.
                BinaryFieldDefinitionViewModel fieldVM = null;
                if (tmpFieldMap.TryGetValue(guid, out fieldVM) == true)
                {
                    // Remove this binary field view model from the temporary map.
                    // We will add it back to our binary field view model list later.
                    tmpFieldMap.Remove(guid);
                }
                else
                {
                    // The binary field that targets this super class is not found,
                    // create the binary field view model and definition.
                    fieldVM = new BinaryFieldDefinitionViewModel(
                        this,
                        new BinaryFieldDefinition()
                        {
                            OutputFieldGuid = guid,
                            GroupDefinition = new BinaryFieldGroupDefinition()
                            {
                                TargetGuid = guid
                            }
                        });
                }

                // Update the child field group of the binary field.
                if (fieldVM.BinaryFieldGroupViewModel != null)
                {
                    fieldVM.BinaryFieldGroupViewModel.UpdateFromTargetDefinition();
                }

                // Add the view model and definition back to their lists.
                this.fieldViewModels.Add(fieldVM);
                myDataModel.Fields.Add(fieldVM.DataModel as BinaryFieldDefinition);
            }

            // Loop through the properties in the target data model definition.
            foreach (RuntimeDataModelPropertyDefinition propertyDef in dataModelDef.Properties)
            {
                // Check if the type of this property is a data model.
                RuntimeTypeInfo info = TypeManager.FindRuntimeType(
                    propertyDef.TypeNamespace,
                    propertyDef.TypeName);

                // Find the binary field view model that targets this property.
                BinaryFieldDefinitionViewModel fieldVM = null;
                if (tmpFieldMap.TryGetValue(propertyDef.Guid, out fieldVM) == true)
                {
                    // Remove this binary field view model from the temporary map.
                    // We will add it back to our binary field view model list later.
                    tmpFieldMap.Remove(propertyDef.Guid);
                }
                else
                {
                    // The binary field that targets this property is not found,
                    // create the field.
                    BinaryFieldGroupDefinition groupDef = null;
                    if (info.IsStructure == true &&
                        info.StructureDefinition != null)
                    {
                        // The property type is a data model, create
                        // field group for the binary field.
                        groupDef = new BinaryFieldGroupDefinition()
                        {
                            TargetGuid = info.StructureDefinition.Guid
                        };
                    }

                    // Create the binary field view model and definition.
                    fieldVM = new BinaryFieldDefinitionViewModel(
                        this,
                        new BinaryFieldDefinition()
                        {
                            OutputFieldGuid = propertyDef.Guid,
                            GroupDefinition = groupDef,
                        });
                }

                // Update the child field group of the binary field.
                if (fieldVM.BinaryFieldGroupViewModel != null)
                {
                    if (info.IsStructure == false)
                    {
                        fieldVM.BinaryFieldGroupViewModel.Dispose();
                        fieldVM.BinaryFieldGroupViewModel = null;

                        var fieldDef = fieldVM.DataModel as BinaryFieldDefinition;
                        fieldDef.GroupDefinition.Dispose();
                        fieldDef.GroupDefinition = null;
                    }
                    else
                    {
                        fieldVM.BinaryFieldGroupViewModel.UpdateFromTargetDefinition();
                    }
                }

                // Add the view model and definition back to their lists.
                this.fieldViewModels.Add(fieldVM);
                myDataModel.Fields.Add(fieldVM.DataModel as BinaryFieldDefinition);
            }
        }
    }
}
