﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.Core.CommonData;
using EffectMaker.DataModelMaker.Core.Core;

using EffectMaker.DataModelMaker.UIControls.BasicControls;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Log;

using EffectMaker.UIControls.Extensions;

namespace EffectMaker.DataModelMaker.UIControls.VersionEditor
{
    /// <summary>
    /// Editor data model version editor dialog.
    /// </summary>
    public partial class EditorVersionEditorDialog : BindableDialog
    {
        /// <summary>The versions.</summary>
        private HashSet<Version> versions = new HashSet<Version>();

        /// <summary>The flag indicating whether the selected version can be deleted.</summary>
        private bool canDeleteSelectedVersion = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        public EditorVersionEditorDialog()
        {
            this.InitializeComponent();

            this.versionListView.SelectedIndexChanged += this.OnVersionListViewItemSelected;

            this.AddBinding("DefinedVersions", "DefinedVersions");
            this.AddBinding("VersionedItemsAtSelectedVersion", "VersionedItemsAtSelectedVersion");

            this.AddBinding("OnCreateVersionExecutable", "OnCreateVersionExecutable");
            this.AddBinding("OnDeleteVersionExecutable", "OnDeleteVersionExecutable");
            this.AddBinding("OnSelectVersionExecutable", "OnSelectVersionExecutable");
        }

        /// <summary>
        /// Enumerate the defined versions.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IEnumerable<Version> DefinedVersions
        {
            set
            {
                this.versions.Clear();

                ListViewItem lastItem = null;
                ListViewItem itemToSelect = null;

                Version selectedVersion = null;
                if (this.versionListView.SelectedItems.Count > 0)
                {
                    selectedVersion = this.versionListView.SelectedItems[0].Tag as Version;
                }

                this.versionListView.BeginUpdate();

                this.versionListView.Items.Clear();
                foreach (Version version in value)
                {
                    var item = new ListViewItem(version.ToString());
                    item.Tag = version;

                    lastItem = item;
                    if (version == selectedVersion)
                    {
                        itemToSelect = item;
                    }

                    this.versionListView.Items.Add(item);
                    this.versions.Add(version);
                }

                this.versionListView.EndUpdate();

                // Select the originally selected version or the latest version.
                if (itemToSelect != null)
                {
                    itemToSelect.Selected = true;
                }
                else if (lastItem != null)
                {
                    lastItem.Selected = true;
                }
            }
        }

        /// <summary>
        /// Get the versioned items at the selected version.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IEnumerable<VersionedItemData> VersionedItemsAtSelectedVersion
        {
            set
            {
                // Update the tree view with the search results.
                this.versionedItemsTreeView.BeginUpdate();

                // First clear the tree view.
                this.versionedItemsTreeView.Nodes.Clear();

                // Loop through the items.
                bool hasItem = false;
                foreach (VersionedItemData data in value)
                {
                    // Create the tree view item.
                    string nodeText = data.Name;
                    if (data.VersionAction != VersionActions.None)
                    {
                        nodeText += " (" + data.VersionAction + ")";
                    }

                    var node = new TreeNode(nodeText);

                    // If there are children in the item, add them to the tree view.
                    foreach (VersionedItemData child in data.Children)
                    {
                        nodeText = child.Name;
                        if (child.VersionAction != VersionActions.None)
                        {
                            nodeText += " (" + child.VersionAction + ")";
                        }

                        node.Nodes.Add(new TreeNode(nodeText));
                    }

                    // Expand all child nodes.
                    node.ExpandAll();

                    // Add the node to the tree view.
                    this.versionedItemsTreeView.Nodes.Add(node);

                    hasItem = true;
                }

                this.versionedItemsTreeView.EndUpdate();

                // Set up the flag so we know if we can deleted the version.
                this.canDeleteSelectedVersion = !hasItem;
            }
        }

        /// <summary>
        /// Get or set the executable for creating a new version definition.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnCreateVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for deleting the specified version definition.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnDeleteVersionExecutable { get; set; }

        /// <summary>
        /// Get or set the executable for selecting version.
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public IExecutable OnSelectVersionExecutable { get; set; }

        /// <summary>
        /// Handles Click event for the create version button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnCreateVersionButtonClicked(object sender, EventArgs e)
        {
            // Find the maximum version on the list.
            var maxVersion = new Version(0, 0, 0, 0);
            var versions = new HashSet<Version>();
            foreach (ListViewItem item in this.versionListView.Items)
            {
                var version = item.Tag as Version;
                if (version != null)
                {
                    versions.Add(version);
                    if (version > maxVersion)
                    {
                        maxVersion = version;
                    }
                }
            }

            // Show dialog for inputting the version to create.
            Version newVersion = null;
            using (var dialog = new VersionInputDialog())
            {
                dialog.Version = maxVersion;
                dialog.VersionValidator = (v) =>
                {
                    bool valid = !versions.Contains(v);
                    if (valid == false)
                    {
                        MessageBox.Show(
                            dialog,
                            string.Format(Properties.Resources.VersionEditor_WarningInputVersionAlreadyExists, v),
                            Properties.Resources.WarningMessageBoxTitle,
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Exclamation);
                    }

                    return valid;
                };

                DialogResult result = dialog.ShowDialog(this);
                if (result != DialogResult.OK)
                {
                    return;
                }

                newVersion = dialog.Version;
            }

            if (this.OnCreateVersionExecutable != null)
            {
                this.OnCreateVersionExecutable.Execute(newVersion);
            }
        }

        /// <summary>
        /// Handles Click event for the delete version button.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnDeleteVersionButtonClicked(object sender, EventArgs e)
        {
            if (this.versionListView.SelectedItems.Count <= 0)
            {
                return;
            }

            var version = this.versionListView.SelectedItems[0].Tag as Version;

            // Do we have any items use this version?
            if (this.canDeleteSelectedVersion == false)
            {
                MessageBox.Show(
                    this,
                    Properties.Resources.VersionEditor_WarningVersionIsBeingUsed,
                    Properties.Resources.WarningMessageBoxTitle,
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Hand);
                return;
            }

            if (this.OnDeleteVersionExecutable != null)
            {
                this.OnDeleteVersionExecutable.Execute(version);
            }
        }

        /// <summary>
        /// Handles SelectedIndexChanged event for the version list view.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnVersionListViewItemSelected(object sender, EventArgs e)
        {
            if (this.versionListView.SelectedItems.Count <= 0)
            {
                return;
            }

            // Create the parameters for executing search of the versioned items.
            var version = this.versionListView.SelectedItems[0].Tag as Version;
            if (version == null)
            {
                return;
            }

            // Set selected version.
            if (this.OnSelectVersionExecutable != null)
            {
                this.OnSelectVersionExecutable.Execute(version);
            }
        }
    }
}
