﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Log;

namespace EffectMaker.DataModelMaker.UIControls.Selection
{
    /// <summary>
    /// Enum for selection groups.
    /// </summary>
    public enum SelectionGroups
    {
        /// <summary>The binary conversion view connections.</summary>
        BinaryConvertConnection,
    }

    /// <summary>
    /// Selection manager class.
    /// </summary>
    public static class SelectionManager
    {
        /// <summary>The selection groups.</summary>
        private static Dictionary<SelectionGroups, SelectionData> selectionGroups =
            new Dictionary<SelectionGroups, SelectionData>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static SelectionManager()
        {
            selectionGroups.Add(SelectionGroups.BinaryConvertConnection, new SelectionData());
        }

        /// <summary>
        /// Event triggers when the selection changed.
        /// </summary>
        public static event EventHandler<SelectionChangedEventArgs> SelectionChanged = null;

        /// <summary>
        /// Event triggers when the cursor moved to a different object.
        /// </summary>
        public static event EventHandler<SelectionChangedEventArgs> MouseOverChanged = null;

        /// <summary>
        /// Set selected object to the specified group.
        /// </summary>
        /// <param name="group">The selection group.</param>
        /// <param name="selectedObject">The selected object.</param>
        public static void SetSelection(
            SelectionGroups group,
            object selectedObject)
        {
            // Does the group exist?
            if (selectionGroups.ContainsKey(group) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "SelectionManager.SetSelection : The selection group {0} is not found, please add the group in the static constructor.",
                    group.ToString());
                return;
            }

            // Get the selection data of the group.
            SelectionData data = selectionGroups[group];
            if (object.ReferenceEquals(data.SelectedObject, selectedObject) == true)
            {
                return;
            }

            // Set the selected object.
            data.SelectedObject = selectedObject;

            // Trigger the event.
            if (SelectionChanged != null)
            {
                SelectionChanged(
                    selectedObject,
                    new SelectionChangedEventArgs(group, selectedObject));
            }
        }

        /// <summary>
        /// Set the object the cursor is hovering over.
        /// </summary>
        /// <param name="group">The selection group.</param>
        /// <param name="mouseOverObject">The object.</param>
        public static void SetMouseOverObject(
            SelectionGroups group,
            object mouseOverObject)
        {
            // Does the group exist?
            if (selectionGroups.ContainsKey(group) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "SelectionManager.SetMouseOverObject : The selection group {0} is not found, please add the group in the static constructor.",
                    group.ToString());
                return;
            }

            // Get the selection data of the group.
            SelectionData data = selectionGroups[group];
            if (object.ReferenceEquals(data.MouseOverObject, mouseOverObject) == true)
            {
                return;
            }

            // Set the mouse over object.
            data.MouseOverObject = mouseOverObject;

            // Trigger the event.
            if (MouseOverChanged != null)
            {
                MouseOverChanged(
                    mouseOverObject,
                    new SelectionChangedEventArgs(group, mouseOverObject));
            }
        }

        /// <summary>
        /// Check if the given object is selected.
        /// </summary>
        /// <param name="group">The selection group.</param>
        /// <param name="obj">The object to check.</param>
        /// <returns>True if the object is selected.</returns>
        public static bool IsObjectSelected(
            SelectionGroups group,
            object obj)
        {
            // Does the group exist?
            if (selectionGroups.ContainsKey(group) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "SelectionManager.SetMouseOverObject : The selection group {0} is not found, please add the group in the static constructor.",
                    group.ToString());
                return false;
            }

            // Get the selection data of the group.
            SelectionData data = selectionGroups[group];
            if (object.ReferenceEquals(data.SelectedObject, obj) == true)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// Check if the cursor is hovering over the given object.
        /// </summary>
        /// <param name="group">The selection group.</param>
        /// <param name="obj">The object to check.</param>
        /// <returns>True if the cursor is over the object.</returns>
        public static bool IsObjectMouseOver(
            SelectionGroups group,
            object obj)
        {
            // Does the group exist?
            if (selectionGroups.ContainsKey(group) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "SelectionManager.SetMouseOverObject : The selection group {0} is not found, please add the group in the static constructor.",
                    group.ToString());
                return false;
            }

            // Get the selection data of the group.
            SelectionData data = selectionGroups[group];
            if (object.ReferenceEquals(data.MouseOverObject, obj) == true)
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// Get the selected object in the group.
        /// </summary>
        /// <param name="group">The selection group.</param>
        /// <returns>The selected object.</returns>
        public static object GetSelectedObject(SelectionGroups group)
        {
            // Does the group exist?
            if (selectionGroups.ContainsKey(group) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "SelectionManager.SetMouseOverObject : The selection group {0} is not found, please add the group in the static constructor.",
                    group.ToString());
                return false;
            }

            // Get the selection data of the group.
            return selectionGroups[group].SelectedObject;
        }

        /// <summary>
        /// Get the mouse over object in the group.
        /// </summary>
        /// <param name="group">The selection group.</param>
        /// <returns>The mouse over object.</returns>
        public static object GetMouseOverObject(SelectionGroups group)
        {
            // Does the group exist?
            if (selectionGroups.ContainsKey(group) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "SelectionManager.SetMouseOverObject : The selection group {0} is not found, please add the group in the static constructor.",
                    group.ToString());
                return false;
            }

            // Get the selection data of the group.
            return selectionGroups[group].MouseOverObject;
        }

        /// <summary>
        /// Clear the selected object in the group.
        /// </summary>
        /// <param name="group">The selection group.</param>
        public static void ClearSelection(SelectionGroups group)
        {
            // Does the group exist?
            if (selectionGroups.ContainsKey(group) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "SelectionManager.SetMouseOverObject : The selection group {0} is not found, please add the group in the static constructor.",
                    group.ToString());
                return;
            }

            // Clear the selected object.
            selectionGroups[group].SelectedObject = null;
        }

        /// <summary>
        /// Clear the mouse over object in the group.
        /// </summary>
        /// <param name="group">The selection group.</param>
        public static void ClearMouseOver(SelectionGroups group)
        {
            // Does the group exist?
            if (selectionGroups.ContainsKey(group) == false)
            {
                Logger.Log(
                    LogLevels.Warning,
                    "SelectionManager.SetMouseOverObject : The selection group {0} is not found, please add the group in the static constructor.",
                    group.ToString());
                return;
            }

            // Clear the mouse over object.
            selectionGroups[group].MouseOverObject = null;
        }

        /// <summary>
        /// The class that holds the selection and mouse over object data.
        /// </summary>
        private class SelectionData
        {
            /// <summary>
            /// Constructor.
            /// </summary>
            public SelectionData()
            {
                this.SelectedObject = null;
                this.MouseOverObject = null;
            }

            /// <summary>
            /// Get or set the selected object.
            /// </summary>
            public object SelectedObject { get; set; }

            /// <summary>
            /// Get or set the object the cursor is hovering over.
            /// </summary>
            public object MouseOverObject { get; set; }
        }
    }

    /// <summary>
    /// The event argument class for selection changed event.
    /// </summary>
    public class SelectionChangedEventArgs : EventArgs
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="group">The selection group.</param>
        /// <param name="selectedObject">The selected object.</param>
        public SelectionChangedEventArgs(SelectionGroups group, object selectedObject)
        {
            this.Group = group;
            this.SelectedObject = selectedObject;
        }

        /// <summary>
        /// Get the selection group.
        /// </summary>
        public SelectionGroups Group { get; private set; }

        /// <summary>
        /// Get the selected object.
        /// </summary>
        public object SelectedObject { get; private set; }
    }
}
