﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView.Runtime
{
    /// <summary>
    /// Class for rendering the runtime data model information.
    /// </summary>
    internal class RuntimeDataModelInfoRenderable : DataModelInfoRenderable
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public RuntimeDataModelInfoRenderable(DataModelInfoViewport owner) :
            base(owner)
        {
            this.Size = new SizeF(0.0f, this.DataModelInfoHeight);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "Namespace", "Namespace"));
            this.Bindings.Add(new Binder(this, "DataModelName", "Name"));
            this.Bindings.Add(new Binder(this, "Description", "Description"));
            this.Bindings.Add(new Binder(this, "InheritingDataModels", "InheritingDataModelsWithoutNamespace"));
            this.Bindings.Add(new Binder(this, "PropertySource", "PropertyDefinitionViewModels"));
        }

        /// <summary>
        /// Get or set the namespace of the data model class.
        /// </summary>
        public string Namespace { get; set; }

        /// <summary>
        /// Get or set the class name of the data model.
        /// </summary>
        public string DataModelName { get; set; }

        /// <summary>
        /// Get or set the data model description.
        /// </summary>
        public string Description { get; set; }

        /// <summary>
        /// Enumerate the inheriting data models.
        /// </summary>
        public IEnumerable<string> InheritingDataModels { get; set; }

        /// <summary>
        /// Get the height of the render area that the data model information takes.
        /// </summary>
        protected override float DataModelInfoHeight
        {
            get { return 45.0f; }
        }

        /// <summary>
        /// Create property renderable from the specified source.
        /// </summary>
        /// <param name="propertySource">The property data source.</param>
        /// <returns>The created property renderable.</returns>
        protected override DataModelInfoPropertyRenderable CreatePropertyRenderable(object propertySource)
        {
            var renderable = new RuntimeDataModelInfoPropertyRenderable(this)
            {
                DataContext = propertySource
            };

            return renderable;
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            var rect = this.TransformedBounds;

            using (var strFormat = new StringFormat())
            {
                // Middle-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Near;

                strFormat.Trimming = StringTrimming.EllipsisPath;

                // Compose the namespace and the data model name.
                string strDataModel = this.DataModelName;
                if (string.IsNullOrEmpty(this.Namespace) == false)
                {
                    strDataModel = this.Namespace + "::" + strDataModel;
                }

                // Measure string size.
                SizeF dataModelStrSize = g.MeasureString(
                    strDataModel,
                    DataModelListItemRenderable.DataModelNameFont,
                    (int)rect.Width,
                    strFormat);

                // Render the data model name.
                g.DrawString(
                    strDataModel,
                    DataModelListItemRenderable.DataModelNameFont,
                    Brushes.DarkBlue,
                    rect,
                    strFormat);

                // Compose the string for all the inheriting type names.
                string strInheritingDataModels = string.Empty;
                foreach (string name in this.InheritingDataModels)
                {
                    if (strInheritingDataModels.Length > 0)
                    {
                        strInheritingDataModels += ", ";
                    }

                    strInheritingDataModels += name;
                }

                if (strInheritingDataModels.Length > 0)
                {
                    // Compute the location and render the inheriting type names.
                    rect.Y += dataModelStrSize.Height;
                    rect.Height -= dataModelStrSize.Height;

                    g.DrawString(
                        strInheritingDataModels,
                        DataModelListItemRenderable.NamespaceFont,
                        Brushes.DarkCyan,
                        rect,
                        strFormat);
                }
            }
        }
    }
}
