﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Serialization;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView.Editor
{
    /// <summary>
    /// Renderable class for rendering a editor data model list item.
    /// </summary>
    internal class EditorDataModelListItemRenderable : DataModelListItemRenderable
    {
        /// <summary>
        /// The flag indicating whether the data model is deleted in the current version.
        /// </summary>
        private bool isDeletedInCurrentVersion = false;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public EditorDataModelListItemRenderable(DataModelListViewport owner) :
            base(owner)
        {
            this.Size = new SizeF(0.0f, 55.0f);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "Namespace", "Namespace"));
            this.Bindings.Add(new Binder(this, "DataModelName", "Name"));
            this.Bindings.Add(new Binder(this, "Description", "Description"));
            this.Bindings.Add(new Binder(this, "CreatedVersion", "CreatedVersion"));
            this.Bindings.Add(new Binder(this, "DeletedVersion", "DeletedVersion"));
            this.Bindings.Add(new Binder(this, "IsDeletedInCurrentVersion", "IsDeletedInCurrentVersion"));
        }

        /// <summary>
        /// Get or set the namespace of the data model class.
        /// </summary>
        public string Namespace { get; set; }

        /// <summary>
        /// Get or set the class name of the data model.
        /// </summary>
        public string DataModelName { get; set; }

        /// <summary>
        /// Get or set the data model description.
        /// </summary>
        public string Description { get; set; }

        /// <summary>
        /// Get or set the created version of the data model.
        /// </summary>
        public VersionXml CreatedVersion { get; set; }

        /// <summary>
        /// Get or set the deleted version of the data model.
        /// </summary>
        public VersionXml DeletedVersion { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether the data model
        /// has been deleted in the current version.
        /// </summary>
        public bool IsDeletedInCurrentVersion
        {
            get
            {
                return this.isDeletedInCurrentVersion;
            }

            set
            {
                this.isDeletedInCurrentVersion = value;
                this.UpdateAppearance();
            }
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            var rect = this.TransformedBounds;

            // Compute the text size of the namespace.
            SizeF namespaceTypeTextSize = g.MeasureString(
                this.Namespace,
                DataModelListItemRenderable.NamespaceFont);

            // Compute the text size of the data model type.
            SizeF dataModelTypeTextSize = g.MeasureString(
                this.DataModelName,
                DataModelListItemRenderable.DataModelNameFont);

            // The rectangle for the namespace.
            RectangleF namespaceTextRect = rect;
            namespaceTextRect.X += 3.0f;
            namespaceTextRect.Y += 6.0f;
            namespaceTextRect.Width = rect.Width - 3.0f;
            namespaceTextRect.Height = namespaceTypeTextSize.Height;

            // The rectangle for the versions.
            RectangleF versionTextRect = rect;
            versionTextRect.X += 4.0f;
            versionTextRect.Y = namespaceTextRect.Bottom - 8.0f;
            versionTextRect.Width = rect.Width - 3.0f;
            versionTextRect.Height = 20;

            // The rectangle for the data model type name.
            RectangleF dataModelTypeTextRect = rect;
            dataModelTypeTextRect.X += 2.0f;
            dataModelTypeTextRect.Y = rect.Bottom - dataModelTypeTextSize.Height - 3.0f;
            dataModelTypeTextRect.Width = rect.Width - 2.0f;
            dataModelTypeTextRect.Height = dataModelTypeTextSize.Height + 5.0f;

            // Set up the text colors.
            Brush namespaceBrush = Brushes.DarkCyan;
            Brush versionBrush = Brushes.SteelBlue;
            Brush dataModelNameBrush = Brushes.DarkBlue;
            if (this.IsDeletedInCurrentVersion == true)
            {
                namespaceBrush = Brushes.Firebrick;
                versionBrush = Brushes.Crimson;
                dataModelNameBrush = Brushes.Firebrick;
            }

            // Compose the versions. (created and deleted versions)
            string versionText = this.CreatedVersion.Value + " - ";
            if (this.DeletedVersion != null)
            {
                versionText += this.DeletedVersion.Value;
            }
            else
            {
                versionText += "the latest version";
            }

            // Render.
            using (var strFormat = new StringFormat())
            {
                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisPath;

                g.DrawString(
                    this.Namespace,
                    DataModelListItemRenderable.NamespaceFont,
                    namespaceBrush,
                    namespaceTextRect,
                    strFormat);

                g.DrawString(
                    versionText,
                    DataModelListItemRenderable.VersionFont,
                    versionBrush,
                    versionTextRect,
                    strFormat);

                strFormat.LineAlignment = StringAlignment.Near;
                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    this.DataModelName,
                    DataModelListItemRenderable.DataModelNameFont,
                    dataModelNameBrush,
                    dataModelTypeTextRect,
                    strFormat);
            }
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        protected override void UpdateAppearance()
        {
            if (this.IsDeletedInCurrentVersion == true)
            {
                if (this.IsSelected == true)
                {
                    this.BorderThickness = 2.0f;
                    this.BorderColor = Color.Red;
                    this.FillColor = Color.LightPink;
                }
                else if (this.IsMouseOver == true)
                {
                    this.BorderThickness = 0.0f;
                    this.BorderColor = Color.Transparent;
                    this.FillColor = Color.LightPink;
                }
                else
                {
                    this.BorderThickness = 0.0f;
                    this.BorderColor = Color.Transparent;
                    this.FillColor = Color.Linen;
                }
            }
            else
            {
                base.UpdateAppearance();
            }
        }
    }
}
