﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView
{
    /// <summary>
    /// The viewport class for rendering the data model list.
    /// </summary>
    internal abstract class DataModelListViewport : Viewport, ILogicalTreeElement
    {
        /// <summary>Constant list item spacing.</summary>
        private const float ListItemSpacing = 5.0f;

        /// <summary>Constant list item spacing.</summary>
        private const float ListItemMargin = 5.0f;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>
        /// Backing field for Controls property.
        /// (should be empty, just to satisfy ILogicalTreeElement)
        /// </summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The data model list item that the cursor is currently on.</summary>
        private DataModelListItemRenderable mouseOverItem = null;

        /// <summary>The data model list item that is currently selected.</summary>
        private DataModelListItemRenderable selectedItem = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="renderingControl">The control to be rendered to.</param>
        public DataModelListViewport(Control renderingControl) :
            base(renderingControl)
        {
            this.Bindings = new BindingContainer(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            // Set up binding.
            this.Bindings.Add(new Binder(this, "SelectedDataModelSource", "SelectedDataModel"));
            this.Bindings.Add(new Binder(this, "DisplayedDataModelSource", "DisplayedDataModel"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return null; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the data model source.
        /// </summary>
        public IEnumerable DataModelListSource
        {
            get
            {
                IEnumerable result =
                    from ch in this.Children
                    where ch is DataModelListItemRenderable
                    select ch.DataContext;

                return result;
            }

            set
            {
                this.UpdateItemsFromDataSource(value);
            }
        }

        /// <summary>
        /// Get or set the data model source to be displayed.
        /// </summary>
        public object DisplayedDataModelSource
        {
            get
            {
                if (this.MouseOverItem != null)
                {
                    return this.MouseOverItem.DataContext;
                }
                else
                {
                    return null;
                }
            }

            set
            {
                if (value == null)
                {
                    this.MouseOverItem = null;
                    return;
                }

                foreach (ILogicalTreeElement child in this.Children)
                {
                    if (child.DataContext == value)
                    {
                        this.MouseOverItem = child as DataModelListItemRenderable;
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the selected data model source.
        /// </summary>
        public object SelectedDataModelSource
        {
            get
            {
                if (this.SelectedItem != null)
                {
                    return this.SelectedItem.DataContext;
                }
                else
                {
                    return null;
                }
            }

            set
            {
                if (value == null)
                {
                    this.SelectedItem = null;
                    return;
                }

                foreach (ILogicalTreeElement child in this.Children)
                {
                    if (child.DataContext == value)
                    {
                        this.SelectedItem = child as DataModelListItemRenderable;
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// Get or set the data model list item that the cursor is currently hovering over.
        /// </summary>
        public DataModelListItemRenderable MouseOverItem
        {
            get
            {
                return this.mouseOverItem;
            }

            set
            {
                if (this.mouseOverItem != value)
                {
                    if (this.mouseOverItem != null)
                    {
                        this.mouseOverItem.IsMouseOver = false;
                    }

                    this.mouseOverItem = value;

                    if (this.mouseOverItem != null)
                    {
                        this.mouseOverItem.IsMouseOver = true;
                    }

                    this.logicalTreeElementExtender.NotifyPropertyChanged(propertyName: "DisplayedDataModelSource");
                }
            }
        }

        /// <summary>
        /// Get or set the selected data model list item.
        /// </summary>
        public DataModelListItemRenderable SelectedItem
        {
            get
            {
                return this.selectedItem;
            }

            set
            {
                if (this.selectedItem != value)
                {
                    if (this.selectedItem != null)
                    {
                        this.selectedItem.IsSelected = false;
                    }

                    this.selectedItem = value;

                    if (this.selectedItem != null)
                    {
                        this.selectedItem.IsSelected = true;
                    }

                    this.logicalTreeElementExtender.NotifyPropertyChanged(propertyName: "SelectedDataModelSource");
                }
            }
        }

        /// <summary>
        /// Get the size of all the list items.
        /// </summary>
        public SizeF ContentSize { get; private set; }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Select the first data model.
        /// </summary>
        public void SelectFirstItem()
        {
            this.SelectedItem = this.Renderables.FirstOrDefault(
                rd => rd is DataModelListItemRenderable) as DataModelListItemRenderable;
            this.MouseOverItem = this.SelectedItem;
        }

        /// <summary>
        /// Select the last data model.
        /// </summary>
        public void SelectLastItem()
        {
            this.SelectedItem = this.Renderables.LastOrDefault(
                rd => rd is DataModelListItemRenderable) as DataModelListItemRenderable;
            this.MouseOverItem = this.SelectedItem;
        }

        /// <summary>
        /// Select the next data model.
        /// </summary>
        public void SelectNextItem()
        {
            if (this.SelectedItem == null)
            {
                // If nothing has been selected yet, select the first data model.
                this.SelectFirstItem();
            }
            else
            {
                bool found = false;
                DataModelListItemRenderable itemToSelect = null;
                foreach (RenderableBase rd in this.Renderables)
                {
                    if (found == false)
                    {
                        found = object.ReferenceEquals(rd, this.SelectedItem);
                    }
                    else if (rd is DataModelListItemRenderable)
                    {
                        itemToSelect = (DataModelListItemRenderable)rd;
                        break;
                    }
                }

                if (itemToSelect == null)
                {
                    // The original selected data model might already be the last one.
                    // Try to wrap over the list and select the first data model.
                    this.SelectFirstItem();
                }
                else
                {
                    this.SelectedItem = itemToSelect;
                    this.MouseOverItem = itemToSelect;
                }
            }
        }

        /// <summary>
        /// Select the previous data model.
        /// </summary>
        public void SelectPreviousItem()
        {
            if (this.SelectedItem == null)
            {
                // If nothing has been selected yet, select the last data model.
                this.SelectLastItem();
            }
            else
            {
                bool found = false;
                DataModelListItemRenderable itemToSelect = null;
                foreach (RenderableBase rd in this.Renderables.Reverse())
                {
                    if (found == false)
                    {
                        found = object.ReferenceEquals(rd, this.SelectedItem);
                    }
                    else if (rd is DataModelListItemRenderable)
                    {
                        itemToSelect = (DataModelListItemRenderable)rd;
                        break;
                    }
                }

                if (itemToSelect == null)
                {
                    // The original selected data model might already be the first one.
                    // Try to wrap over the list and select the last data model.
                    this.SelectLastItem();
                }
                else
                {
                    this.SelectedItem = itemToSelect;
                    this.MouseOverItem = itemToSelect;
                }
            }
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float childWidth = this.Width - (ListItemMargin * 2.0f);

            // Loop through data model list items to update the content size.
            foreach (RenderableBase child in this.Renderables)
            {
                child.Size = new SizeF(childWidth, child.Height);
            }

            base.UpdateChildren(context);

            float height = ListItemMargin;

            // Loop through data model list items to update the content size.
            foreach (RenderableBase child in this.Renderables)
            {
                if ((child is DataModelListItemRenderable) == false)
                {
                    continue;
                }

                // Setup location and size.
                child.Location = new PointF(ListItemMargin, height);

                height += child.Height + ListItemSpacing;
            }

            this.ContentSize = new SizeF(childWidth, height);
        }

        /// <summary>
        /// Create list item renderable from the specified source.
        /// </summary>
        /// <param name="itemSource">The list item data source.</param>
        /// <returns>The created property renderable.</returns>
        protected abstract DataModelListItemRenderable CreateItemRenderable(object itemSource);

        /// <summary>
        /// Update items from data source.
        /// </summary>
        /// <param name="items">The items from data source.</param>
        private void UpdateItemsFromDataSource(IEnumerable items)
        {
            this.DisplayedDataModelSource = null;

            object savedSelectedItemSource = this.SelectedDataModelSource;

            // Clear all the data model renderers first.
            var dataModelRendererList =
                this.Renderables.Where(itm => itm is DataModelListItemRenderable).AsEnumerable();

            this.RemoveRenderableRange(dataModelRendererList);

            float height = ListItemMargin;
            float maxWidth = 0.0f;

            // Loop through data model source items.
            foreach (object item in items)
            {
                // Create a data model view for the data model source.
                var renderable = this.CreateItemRenderable(item);

                // Setup location and size.
                renderable.Location = new PointF(ListItemMargin, height);

                height += renderable.Height + ListItemSpacing;
                maxWidth = Math.Max(maxWidth, renderable.Width + (ListItemMargin * 2.0f));

                // Add the list item to the child renderable list.
                this.AddRenderable(renderable);
            }

            this.ContentSize = new SizeF(maxWidth, height);

            // Restore the selected item.
            if (savedSelectedItemSource != null)
            {
                this.SelectedDataModelSource = savedSelectedItemSource;
            }
        }
    }
}
