﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.DataModelMaker.UIControls.DataModelEditor;
using EffectMaker.DataModelMaker.UIControls.DataModelPropertyEditor;

using EffectMaker.DataModelMaker.UILogic.ViewModels;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Render.ScrollBar;

using EffectMaker.UIControls.BaseControls;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView
{
    /// <summary>
    /// The panel that renders the data model list.
    /// </summary>
    internal class DataModelListPanel : Panel
    {
        /// <summary>Constant for the factor applying to the vertical mouse scroll speed.</summary>
        private const int VerticalScrollSpeedFactor = 50;

        /// <summary>The viewport for rendering the data model list.</summary>
        private DataModelListViewport dataModelListViewport = null;

        /// <summary>The scroll bars.</summary>
        private AutoHiddenScrollBar scrollBars = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        public DataModelListPanel()
        {
            // Enable double buffer.
            this.DoubleBuffered = true;

            // Make the panel selectable.
            this.SetStyle(ControlStyles.Selectable, true);
            this.TabStop = true;
        }

        /// <summary>
        /// Get or set the data context.
        /// </summary>
        public object DataContext
        {
            get
            {
                if (this.dataModelListViewport == null)
                {
                    return null;
                }

                return this.dataModelListViewport.DataContext;
            }

            set
            {
                if (value == null)
                {
                    if (this.dataModelListViewport != null)
                    {
                        this.dataModelListViewport.Dispose();
                        this.dataModelListViewport = null;
                    }

                    this.scrollBars = null;
                }
                else
                {
                    // Create the viewport for rendering.
                    if (value is RootEditorDataModelDefinitionViewModel)
                    {
                        this.dataModelListViewport = new Editor.EditorDataModelListRenderable(this);
                    }
                    else if (value is RootRuntimeDataModelDefinitionViewModel)
                    {
                        this.dataModelListViewport = new Runtime.RuntimeDataModelListRenderable(this);
                    }
                    else
                    {
                        Logger.Log(LogLevels.Error, "DataModelListPanel.DataContext : The data context is not of valid type.");
                        return;
                    }

                    this.dataModelListViewport.Bounds = new Rectangle(
                        0,
                        0,
                        this.ClientRectangle.Width,
                        this.ClientRectangle.Height);

                    // Set up the data context to the viewport for binding.
                    this.dataModelListViewport.DataContext = value;

                    // Create the scroll bars.
                    this.scrollBars = new AutoHiddenScrollBar(this)
                    {
                        ForeColor = Color.Black,
                        BackColor = Color.Black,
                        IncrementV = VerticalScrollSpeedFactor
                    };

                    this.scrollBars.RenderRequired += (s, e) => { this.Invalidate(); };
                    this.scrollBars.Scroll += (s, e) =>
                    {
                        this.dataModelListViewport.Translation =
                            new PointF(-this.ScrollPosition.X, -this.ScrollPosition.Y);
                        this.Invalidate();
                    };

                    this.scrollBars.SetContentSize(new Size(
                        (int)this.dataModelListViewport.ContentSize.Width,
                        (int)this.dataModelListViewport.ContentSize.Height));
                }
            }
        }

        /// <summary>
        /// Get or set the scroll position.
        /// </summary>
        public Point ScrollPosition
        {
            get
            {
                if (this.scrollBars == null)
                {
                    return Point.Empty;
                }

                return this.scrollBars.ScrollPosition;
            }

            set
            {
                this.SetScrollPosition(value.X, value.Y);
            }
        }

        /// <summary>
        /// Set scroll position.
        /// </summary>
        /// <param name="x">The horizontal position.</param>
        /// <param name="y">The vertical position.</param>
        public void SetScrollPosition(int x, int y)
        {
            if (this.scrollBars == null)
            {
                return;
            }

            var contentSize = new Size(
                (int)this.dataModelListViewport.ContentSize.Width,
                (int)this.dataModelListViewport.ContentSize.Height);

            int maxX = contentSize.Width - this.Width;
            int maxY = contentSize.Height - this.Height;

            int posX = Math.Max(Math.Min(x, maxX), 0);
            int posY = Math.Max(Math.Min(y, maxY), 0);

            this.scrollBars.ScrollPosition = new Point(posX, posY);

            this.dataModelListViewport.Translation = new PointF(-posX, -posY);
        }

        /// <summary>
        /// Show data model editor dialog for editing the selected data model, if any.
        /// </summary>
        public void ShowSelectedDataModelEditorDialog()
        {
            if (this.dataModelListViewport.SelectedItem != null)
            {
                var dialog = new DataModelEditorDialog()
                {
                    DataSource = this.dataModelListViewport.SelectedItem.DataContext,
                    StartPosition = FormStartPosition.CenterParent
                };

                dialog.ShowDialog(this);
            }
        }

        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        /// <param name="disposing">true if managed resources should be disposed; otherwise, false.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing == true)
            {
                if (this.dataModelListViewport != null)
                {
                    this.dataModelListViewport.Dispose();
                }
            }

            base.Dispose(disposing);
        }

        /// <summary>
        /// Handle MouseWheel event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            base.OnMouseWheel(e);

            int increment = -(e.Delta / 120) * VerticalScrollSpeedFactor;

            this.SetScrollPosition(this.ScrollPosition.X, this.ScrollPosition.Y + increment);

            this.Invalidate();
        }

        /// <summary>
        /// Handle MouseDoubleClick event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDoubleClick(MouseEventArgs e)
        {
            base.OnMouseDoubleClick(e);

            // Do mouse picking.
            ObjectPickContext context = this.dataModelListViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] is DataModelListItemRenderable)
            {
                this.dataModelListViewport.SelectedItem =
                    context.PickedObjects[0] as DataModelListItemRenderable;
                this.ShowSelectedDataModelEditorDialog();
            }
        }

        /// <summary>
        /// Handle MouseDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            // Do mouse picking.
            if (e.Button == MouseButtons.Left)
            {
                ObjectPickContext context = this.dataModelListViewport.DoPicking(e.Location);
                if (context.PickedObjects != null &&
                    context.PickedObjects.Count > 0 &&
                    context.PickedObjects[0] is DataModelListItemRenderable)
                {
                    this.dataModelListViewport.SelectedItem =
                        context.PickedObjects[0] as DataModelListItemRenderable;
                }
            }
            else if (e.Button == MouseButtons.Right &&
                this.dataModelListViewport.SelectedItem != null)
            {
                this.dataModelListViewport.SelectedItem = null;
            }
        }

        /// <summary>
        /// Handle MouseUp event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseUp(MouseEventArgs e)
        {
            base.OnMouseUp(e);
        }

        /// <summary>
        /// Handle MouseMove event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseMove(MouseEventArgs e)
        {
            if (this.Focused == false)
            {
                this.Focus();
            }

            base.OnMouseMove(e);

            // Do mouse picking.
            ObjectPickContext context = this.dataModelListViewport.DoPicking(e.Location);
            if (context.PickedObjects != null &&
                context.PickedObjects.Count > 0 &&
                context.PickedObjects[0] is DataModelListItemRenderable)
            {
                this.dataModelListViewport.MouseOverItem =
                    context.PickedObjects[0] as DataModelListItemRenderable;
            }
        }

        /// <summary>
        /// Handle MouseLeave event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnMouseLeave(EventArgs e)
        {
            base.OnMouseLeave(e);

            if (this.dataModelListViewport.MouseOverItem != null)
            {
                this.dataModelListViewport.MouseOverItem = null;
            }
        }

        /// <summary>
        /// Handle PreviewKeyDown event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPreviewKeyDown(PreviewKeyDownEventArgs e)
        {
            base.OnPreviewKeyDown(e);

            switch (e.KeyCode)
            {
                case Keys.Down:
                    this.dataModelListViewport.SelectNextItem();
                    break;

                case Keys.Up:
                    this.dataModelListViewport.SelectPreviousItem();
                    break;

                case Keys.Enter:
                    this.ShowSelectedDataModelEditorDialog();
                    break;
            }
        }

        /// <summary>
        /// Handle SizeChanged event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnSizeChanged(EventArgs e)
        {
            if (this.dataModelListViewport != null)
            {
                this.dataModelListViewport.Size = this.ClientSize;
            }

            if (this.scrollBars != null)
            {
                Point scrollPos = this.scrollBars.ScrollPosition;

                this.scrollBars.SetViewSize(this.Size);
                this.scrollBars.SetDisplayRectangle(this.DisplayRectangle);

                // Reset to the same scroll position again.
                this.scrollBars.ScrollPosition = scrollPos;
            }
        }

        /// <summary>
        /// Handle Paint event.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnPaint(PaintEventArgs e)
        {
            e.Graphics.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.HighQuality;
            e.Graphics.TextRenderingHint = System.Drawing.Text.TextRenderingHint.ClearTypeGridFit;

            // Render the viewport.
            if (this.dataModelListViewport != null && this.scrollBars != null)
            {
                this.dataModelListViewport.Update(e.Graphics);
                this.dataModelListViewport.Draw(e.Graphics);

                // The data model list is updated, now set the updated content size to the scroll bars.
                this.scrollBars.SetContentSize(new Size(
                    (int)this.dataModelListViewport.ContentSize.Width,
                    (int)this.dataModelListViewport.ContentSize.Height));

                // Render the scroll bars.
                this.scrollBars.Render(e.Graphics);
            }
        }
    }
}
