﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.UIControls.Extenders;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelView
{
    /// <summary>
    /// Class for rendering data model information.
    /// </summary>
    internal abstract class DataModelInfoRenderable : RoundedRectangleShape, ILogicalTreeElement
    {
        /// <summary>Constant for the margin of the property renderables.</summary>
        private const float ChildMargin = 5.0f;

        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The extender for easily implement operations for child renderables.</summary>
        private ChildRenderableOperationExtender childRenderableOperationExtender = null;

        /// <summary>The child renderables for rendering the properties of the data model.</summary>
        private List<DataModelInfoPropertyRenderable> propertyRenderables =
            new List<DataModelInfoPropertyRenderable>();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static DataModelInfoRenderable()
        {
            // Setup the fonts for rendering.
            NamespaceFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            VersionFont = new Font(FontFamily.GenericSansSerif, 7.0f);
            DataModelNameFont = new Font(FontFamily.GenericSansSerif, 14.0f, FontStyle.Bold);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public DataModelInfoRenderable(DataModelInfoViewport owner) :
            base()
        {
            this.Translation = new PointF(ChildMargin, this.DataModelInfoHeight);
            this.BorderThickness = 2.0f;
            this.FillColor = Color.Beige;
            this.SetCornerRadius(5.0f);

            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);
            this.childRenderableOperationExtender = new ChildRenderableOperationExtender(this);

            this.Owner = owner;
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the font for rendering the data model name.
        /// </summary>
        public static Font DataModelNameFont { get; private set; }

        /// <summary>
        /// Get the font for rendering the version string.
        /// </summary>
        public static Font VersionFont { get; private set; }

        /// <summary>
        /// Get the font for rendering the namespace string.
        /// </summary>
        public static Font NamespaceFont { get; private set; }

        /// <summary>
        /// Get the owner data model info viewport.
        /// </summary>
        public DataModelInfoViewport Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner; }
            protected set { this.Owner = value as DataModelInfoViewport; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the data model property source.
        /// </summary>
        public IEnumerable PropertySource
        {
            get { return this.propertyRenderables.Select(pr => pr.DataContext).AsEnumerable(); }
            set { this.UpdateItemsFromDataSource(value); }
        }

        /// <summary>
        /// Enumerates all the child property renderables.
        /// </summary>
        public IEnumerable<DataModelInfoPropertyRenderable> PropertyRenderables
        {
            get { return this.propertyRenderables; }
        }

        /// <summary>
        /// Get the height of the render area that the data model information takes.
        /// </summary>
        protected abstract float DataModelInfoHeight { get; }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public override void AddRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.AddRenderable(child);
            base.AddRenderable(child);

            var propertyRenderable = child as DataModelInfoPropertyRenderable;
            if (propertyRenderable != null)
            {
                this.propertyRenderables.Add(propertyRenderable);
            }
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public override void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.AddRenderableRange(children);
            base.AddRenderableRange(children);

            foreach (RenderableBase child in children)
            {
                var propertyRenderable = child as DataModelInfoPropertyRenderable;
                if (propertyRenderable != null)
                {
                    this.propertyRenderables.Add(propertyRenderable);
                }
            }
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public override void RemoveRenderable(RenderableBase child)
        {
            this.childRenderableOperationExtender.RemoveRenderable(child);
            base.RemoveRenderable(child);

            var propertyRenderable = child as DataModelInfoPropertyRenderable;
            if (propertyRenderable != null)
            {
                this.propertyRenderables.Remove(propertyRenderable);
            }
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public override void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            this.childRenderableOperationExtender.RemoveRenderableRange(children);
            base.RemoveRenderableRange(children);

            // To prevent modifying the property renderable list while looping it,
            // copy the children to another array first.
            var tmpChildren = children;
            if (tmpChildren == this.propertyRenderables)
            {
                tmpChildren = children.ToArray();
            }

            foreach (RenderableBase child in tmpChildren)
            {
                var propertyRenderable = child as DataModelInfoPropertyRenderable;
                if (propertyRenderable != null)
                {
                    this.propertyRenderables.Remove(propertyRenderable);
                }
            }
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public override void ClearRenderables()
        {
            this.childRenderableOperationExtender.ClearRenderables();
            base.ClearRenderables();

            this.propertyRenderables.Clear();
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected override void UpdateChildren(RenderContext context)
        {
            float childWidth = this.Width - (ChildMargin * 2.0f);

            // Loop through data model list items to update the content size.
            foreach (RenderableBase child in this.propertyRenderables)
            {
                child.Size = new SizeF(childWidth, child.Height);
            }

            base.UpdateChildren(context);
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawBackground(Graphics g)
        {
            base.DrawBackground(g);

            // Draw the separation line on the top edge of the child renderables.
            using (var pen = new Pen(Color.Silver, 1.0f))
            {
                foreach (var child in this.propertyRenderables)
                {
                    var rect = child.TransformedBounds;
                    g.DrawLine(pen, rect.Location, new PointF(rect.Right, rect.Top));
                }
            }
        }

        /// <summary>
        /// Create property renderable from the specified source.
        /// </summary>
        /// <param name="propertySource">The property data source.</param>
        /// <returns>The created property renderable.</returns>
        protected abstract DataModelInfoPropertyRenderable CreatePropertyRenderable(object propertySource);

        /// <summary>
        /// Update items from data source.
        /// </summary>
        /// <param name="items">The items from data source.</param>
        private void UpdateItemsFromDataSource(IEnumerable items)
        {
            // Clear all the data model renderers first.
            this.RemoveRenderableRange(this.propertyRenderables);

            float height = 0.0f;

            // Loop through data model source items.
            foreach (object item in items)
            {
                // Create a property renderable for the item source.
                var renderable = this.CreatePropertyRenderable(item);

                // Setup location and size.
                renderable.Location = new PointF(0.0f, height);

                height += renderable.Height;

                // Add the list item to the child renderable list.
                this.AddRenderable(renderable);
            }

            this.Size = new SizeF(this.Width, height + this.DataModelInfoHeight);
        }
    }
}
