﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Linq;

using EffectMaker.DataModelMaker.Core.Core;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.Renderable;
using EffectMaker.Foundation.Utility;

using EffectMaker.UIControls;
using EffectMaker.UIControls.DataBinding;
using EffectMaker.UIControls.Extenders;

namespace EffectMaker.DataModelMaker.UIControls.DataModelVersionView
{
    /// <summary>
    /// Class for rendering data model property version information.
    /// </summary>
    internal class DataModelPropertyVersionRenderable : RectangleShape, ILogicalTreeElement
    {
        /// <summary>The extender for easily implementing ILogicalTreeElement.</summary>
        private LogicalTreeElementExtender logicalTreeElementExtender = null;

        /// <summary>Backing field for Controls property.</summary>
        private IIndexableCollection<ILogicalTreeElement> controlsWrapper;

        /// <summary>The flag indicating if mouse is over this item.</summary>
        private bool isMouseOver = false;

        /// <summary>The flag indicating if the item is selected.</summary>
        private bool isSelected = false;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static DataModelPropertyVersionRenderable()
        {
            // Set up the fonts for rendering.
            VersionFont = new Font(FontFamily.GenericSansSerif, 6.0f);
            ItemSmallFont = new Font(FontFamily.GenericSansSerif, 8.0f);
            ItemLargeFont = new Font(FontFamily.GenericSansSerif, 9.0f);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner of the view.</param>
        public DataModelPropertyVersionRenderable(DataModelVersionRenderable owner) :
            base()
        {
            this.Bindings = new BindingContainer(this);
            this.logicalTreeElementExtender = new LogicalTreeElementExtender(this);

            this.BorderColor = Color.Transparent;
            this.BorderThickness = 0.0f;
            this.FillColor = Color.Transparent;
            this.Size = new SizeF(0.0f, 30.0f);

            this.Owner = owner;

            // Set up bindings.
            this.Bindings.Add(new Binder(this, "PropertyType", "PropertyTypeWithoutNamespace"));
            this.Bindings.Add(new Binder(this, "IsPropertyTypeInvalid", "IsPropertyTypeInvalid"));
            this.Bindings.Add(new Binder(this, "PropertyName", "Name"));
            this.Bindings.Add(new Binder(this, "PropertyDesc", "Description"));
            this.Bindings.Add(new Binder(this, "VersionsAndActions", "VersionsAndActions"));
        }

        /// <summary>
        /// Raised when the value of a property on this control changed.
        /// </summary>
#pragma warning disable 67
        public event PropertyChangedEventHandler PropertyChanged;
#pragma warning restore 67

        /// <summary>
        /// Get the font for rendering the version on the time line.
        /// </summary>
        public static Font VersionFont { get; private set; }

        /// <summary>
        /// Get the small font.
        /// </summary>
        public static Font ItemSmallFont { get; private set; }

        /// <summary>
        /// Get the large font.
        /// </summary>
        public static Font ItemLargeFont { get; private set; }

        /// <summary>
        /// Get the owner data model collection.
        /// </summary>
        public DataModelVersionRenderable Owner { get; private set; }

        /// <summary>
        /// gets the parent control.
        /// </summary>
        public new ILogicalTreeElement Parent
        {
            get { return this.Owner; }
            protected set { this.Owner = value as DataModelVersionRenderable; }
        }

        /// <summary>
        /// Gets a collection of children controls of the the current control instance.
        /// </summary>
        public IIndexableCollection<ILogicalTreeElement> Children
        {
            get
            {
                if (this.controlsWrapper == null)
                {
                    this.controlsWrapper = new IndexableCollection<ILogicalTreeElement>();
                }

                return this.controlsWrapper;
            }
        }

        /// <summary>
        /// Gets the control extender instance of this control.
        /// </summary>
        public LogicalTreeElementExtender LogicalTreeElementExtender
        {
            get { return this.logicalTreeElementExtender; }
        }

        /// <summary>
        /// Gets or sets the DataContext.
        /// This property may raise a 'DataContext' change notification.
        /// See LogicalTreeElementExtender for more information.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public object DataContext
        {
            get { return this.logicalTreeElementExtender.DataContext; }
            set { this.logicalTreeElementExtender.DataContext = value; }
        }

        /// <summary>
        /// Gets the binders collection.
        /// </summary>
        public BindingContainer Bindings { get; private set; }

        /// <summary>
        /// Get or set the property's value type.
        /// </summary>
        public string PropertyType { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether property type is invalid.
        /// </summary>
        public bool IsPropertyTypeInvalid { get; set; }

        /// <summary>
        /// Get or set the property name.
        /// </summary>
        public string PropertyName { get; set; }

        /// <summary>
        /// Get or set the property description.
        /// </summary>
        public string PropertyDesc { get; set; }

        /// <summary>
        /// Enumerates all versions and the action took to the property.
        /// </summary>
        public IEnumerable<KeyValuePair<Version, VersionActions>> VersionsAndActions { get; set; }

        /// <summary>
        /// Get or set the flag indicating if mouse is over this item.
        /// </summary>
        public bool IsMouseOver
        {
            get
            {
                return this.isMouseOver;
            }

            set
            {
                if (this.isMouseOver != value)
                {
                    this.isMouseOver = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating if this item is selected.
        /// </summary>
        public bool IsSelected
        {
            get
            {
                return this.isSelected;
            }

            set
            {
                if (this.isSelected != value)
                {
                    this.isSelected = value;
                    this.UpdateAppearance();
                }
            }
        }

        /// <summary>
        /// Clears the DataContext.
        /// See LogicalTreeElementExtender for more details.
        /// <see cref="LogicalTreeElementExtender"/>
        /// </summary>
        public void ClearDataContext()
        {
            this.logicalTreeElementExtender.ClearDataContext();
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void DrawForeground(Graphics g)
        {
            SizeF valueTypeTextSize = g.MeasureString(
                this.PropertyType,
                DataModelPropertyVersionRenderable.ItemSmallFont);

            SizeF propertyNameTextSize = g.MeasureString(
                this.PropertyName,
                DataModelPropertyVersionRenderable.ItemSmallFont);

            // The value type text rectangle.
            RectangleF valueTypeTextRect = this.TransformedBounds;
            valueTypeTextRect.Width = Math.Min(valueTypeTextSize.Width, this.TransformedBounds.Width);
            valueTypeTextRect.Height = valueTypeTextSize.Height;

            // The property name occupies the right size of the item.
            RectangleF propertyNameRect = this.TransformedBounds;
            propertyNameRect.X = valueTypeTextRect.Right + 5.0f;
            propertyNameRect.Width = Math.Max(0.0f, this.TransformedBounds.Width - propertyNameRect.X);
            propertyNameRect.Height = propertyNameTextSize.Height;

            using (var strFormat = new StringFormat())
            {
                // Bottom-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Far;

                strFormat.Trimming = StringTrimming.EllipsisPath;

                g.DrawString(
                    this.PropertyType,
                    DataModelPropertyVersionRenderable.ItemSmallFont,
                    this.IsPropertyTypeInvalid == true ? Brushes.Red : Brushes.DimGray,
                    valueTypeTextRect,
                    strFormat);

                // Top-left alignment.
                strFormat.Alignment = StringAlignment.Near;
                strFormat.LineAlignment = StringAlignment.Near;

                strFormat.Trimming = StringTrimming.EllipsisCharacter;

                g.DrawString(
                    this.PropertyName,
                    DataModelPropertyVersionRenderable.ItemSmallFont,
                    Brushes.Blue,
                    propertyNameRect,
                    strFormat);
            }

            // The rectangle for the time line.
            RectangleF timelineRect = this.TransformedBounds;
            timelineRect.Y = propertyNameRect.Bottom + 3.0f;
            timelineRect.Height = this.TransformedBounds.Bottom - timelineRect.Y - 5.0f;

            // Render the time line.
            this.DrawTimeLine(g, timelineRect);
        }

        /// <summary>
        /// Update the appearance of the item.
        /// </summary>
        private void UpdateAppearance()
        {
            if (this.IsSelected == true)
            {
                this.BorderColor = Color.Red;
                this.BorderThickness = 2.0f;
                this.FillColor = Color.Khaki;
            }
            else if (this.IsMouseOver == true)
            {
                this.BorderColor = Color.Transparent;
                this.BorderThickness = 0.0f;
                this.FillColor = Color.Khaki;
            }
            else
            {
                this.BorderColor = Color.Transparent;
                this.BorderThickness = 0.0f;
                this.FillColor = Color.Transparent;
            }
        }

        /// <summary>
        /// Render the time line.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <param name="rect">The render boundary for the time line.</param>
        private void DrawTimeLine(Graphics g, RectangleF rect)
        {
            const int TimeLinePadding = 7;
            const int VersionLength = 35;

            float x = (float)Math.Ceiling(rect.X);
            float y = (float)Math.Ceiling(rect.Y);
            float w = (float)Math.Ceiling(rect.Width);
            float h = (float)Math.Ceiling(rect.Height);

            var versionRect = new Rectangle(
                (int)x,
                (int)y + TimeLinePadding,
                VersionLength,
                (int)h - TimeLinePadding);

            Brush brush;
            bool renderVersionNumber = false;
            bool propertyCreated = false;
            foreach (KeyValuePair<Version, VersionActions> entry in this.VersionsAndActions)
            {
                // Determine if we should render this version on the time line,
                // and decide the color depending on the action took at the version.
                renderVersionNumber = false;
                brush = Brushes.AliceBlue;
                if (entry.Value == VersionActions.Create)
                {
                    // The property was created at this version.
                    propertyCreated = true;
                    renderVersionNumber = true;
                    brush = Brushes.DodgerBlue;
                }
                else if (entry.Value == VersionActions.Delete)
                {
                    // The property was deleted at this version.
                    propertyCreated = false;
                    renderVersionNumber = true;
                    brush = Brushes.Crimson;
                }
                else if (propertyCreated == true)
                {
                    if (entry.Value == VersionActions.Modify)
                    {
                        // The property was modified at this version.
                        brush = Brushes.Orange;
                        renderVersionNumber = true;
                    }
                    else
                    {
                        // The property exists at this version, but
                        // has no modification record whatsoever.
                        brush = Brushes.PaleGreen;
                    }
                }

                // Draw the version on the time line.
                g.FillRectangle(brush, versionRect);

                if (renderVersionNumber == true)
                {
                    // Draw the version number.
                    g.DrawString(
                        entry.Key.ToString(),
                        VersionFont,
                        brush,
                        versionRect.X,
                        versionRect.Y - TimeLinePadding - 2);
                }

                // Advance the location of the version boundary on the time line.
                versionRect.X += VersionLength;
            }
        }
    }
}
